/* (c) by G. Caronni in 1995                              18.11.95
 *
 * Contains certificate storage management, IO and calculation of 
 * shared secrets
 * 
 * changes: Robert Muchsel 1996-1997
 */

/* #define DEBUG_CERTDB */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>

#include <sys/types.h>		/* some of these might be unneeded */
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "cert_defs.h"
#include "cert_db.h"
#include "cert_int.h"
#include "../include/id.h"

#ifdef USE_SUNCERT
#include "suncert/enlink.h"
#endif

#ifdef __GNUC__
#ident "$Id: cert_db.c,v 1.8 1996/06/07 09:49:04 skip Exp $"
#else
static char rcsid[] = "$Id: cert_db.c,v 1.8 1996/06/07 09:49:04 skip Exp $";
#endif


char *data_path = NULL;

dh_data *cert_first = NULL;


static int get_line(FILE * fin, char *buf, int bufsize)
{
  char tmpbuf[10240], *s, *t, c;
  int spare = bufsize - 1;

  *buf = 0;
  while (spare && (s = fgets(tmpbuf, spare, fin))) {
    while (isspace(*s))
      s++;
    t = s + strlen(s);
    *(t--) = 0;
    if (t >= s && *t == '\n')
      *(t--) = 0;
    if (*s == 0 || *s == '#')
      continue;
    if ((c = *t) == '\\')
      *(t--) = 0;
    strcat(buf, s);
    spare -= t - s + 1;
    if (c != '\\')
      break;
  }
  return bufsize - spare - 1;
}

char *make_name(u_char nsid, u_char * mkid)
{
  char buf[10240];
  int i;
  struct in_addr a;

  assert(nsid && nsid <= MAX_NSID && mkid);

  sprintf(buf, "%02X-", nsid);
  if (nsid == SKIP_NSID_IPV4) {
    memcpy(&(a.s_addr), mkid, sizeof(u_long));
    strcat(buf, inet_ntoa(a));
  }
  else {
    for (i = 0; i < nsid_len[nsid]; i++)
      sprintf(buf + 3 + 2 * i, "%02X", mkid[i]);
  }
  strcat(buf, "-");
  return strdup(buf);
}

/* load all matching certificates in the named paths flag=1 -> secret */
static dh_data *disk_load(u_char nsid, u_char *mkid, char *path[], int flag)
{
  char p[10240];
  char nam[10240];
  int cnt = 0;
  DIR *dir;
  struct dirent *entry;
  dh_data *t, *last = NULL;
  char *name = make_name(nsid, mkid);

  while (path[cnt]) {
    strcpy(p, data_path);
    strcat(p, path[cnt]);
    dir = opendir(p);
    while (dir && (entry = readdir(dir))) {
      if (!strncmp(entry->d_name, name, strlen(name))) {
	strcpy(nam, p);
	strcat(nam, entry->d_name);
	if (flag) {
	  if ((t = db_secret_read(nam))) {
#ifdef DEBUG_CERTDB
            printf("diskload: secret_read %s OK\n", nam);
#endif
	    t->ons = nsid;
	    t->omk = memcpy(malloc(nsid_len[nsid]), mkid, nsid_len[nsid]);
	  }
	}
	else {
	  if ((t = db_public_read(nam))) {
#ifdef DEBUG_CERTDB
            printf("diskload: public_read %s OK\n", nam);
#endif
	    t->rns = nsid;
	    t->rmk = memcpy(malloc(nsid_len[nsid]), mkid, nsid_len[nsid]);
	  }
	}
	/* load failed if e.g. certificate invalid or timed out */
	/* could be removed from disk */
	if (t)
	  last = t;
      }
    }
    if (dir)
      closedir(dir);
    cnt++;
  }
  free(name);
  return last;
}

extern void db_unique_disk_save(dh_data ** res, char *from, int flag)
{
  dh_data *find = NULL;
  char *name = NULL;
  char p[10240];
  u_char *rmk;
  FILE *f;

  /* no sense to save secret values for now! */
  assert(!flag);

  /* fist check if such a dh_data already exists, if yes, don't bother
   * with this one. Later on we will have to check validity values of
   * the two, and save nevertheless... 
   */

  if (!(*res)->rns) 
    rmk = (u_char *) (*res)->rip.a;
  else
    rmk = (*res)->rmk;

  while (find == *res)
    find = db_public_find_load(NULL, (*res)->rns ? (*res)->rns : DEFAULT_NSID,
			       rmk);

  if (find == (dh_data *) (-1))
    find = db_public_find_load(find, (*res)->rns ? (*res)->rns : DEFAULT_NSID,
			       rmk);


  /* quick fix to let add_cert find the name so create_reply does work! */
  /* this is not needed anymore. don't care! */

  if (!(*res)->public) {
    name = make_name((*res)->rns ? (*res)->rns : DEFAULT_NSID, rmk);

    strcpy(p, data_path);
    strcat(p, "cache/");
    strcat(p, name);
    strcat(p, "1");		/* could count up, later */
    (*res)->public = strdup(p);
  }

  if (find) {
    /* already there -- don't bother */
    db_cert_drop(*res);
    *res = find;
    return;
  }

  if (!(f = fopen(p, "w"))) {
    if (name)
      free(name);
    return;
  }
  if (name)
    db_public_write(f, *res, name, from);
  fclose(f);
  chmod(p, 0644);
  if (name)
    free(name);
  return;
}

dh_data *db_public_find_load(dh_data * next, u_char nsid, u_char * mkid)
{
  int len;
  u_char *a;
  char *w[3] = {"public/", "cache/", NULL};

  if (next == (dh_data *) (-1))
    return disk_load(nsid, mkid, w, 0);
  if (!next)
    next = cert_first;
  else
    next = next->next;
  while (next) {
    if (next->rns) {
      len = nsid_len[next->rns];
      a = next->rmk;
    }
    else {
      len = nsid_len[DEFAULT_NSID];
      a = (u_char *) &(next->rip);
    }
    if (nsid == (next->rns ? next->rns : DEFAULT_NSID) &&
	nsid_len[nsid ? nsid : DEFAULT_NSID] == len &&
	!memcmp(a, mkid, len))
      break;
    next = next->next;
  }
  if (next)
    return next;
  else
    return (dh_data *) (-1);
}

int db_secret_add(dh_data * public)
{
  int ons, len;
  char *w[2] =
  {"secret/", NULL};
  u_char *a;
  dh_data *data = cert_first;
  int load = 0;

  assert(data);			/* otherwise no load would ever happen */
  assert(public->ons || public->oip.a[0]);	/*IPv4 */
  if (public->ons)
    assert(public->omk);
  ons = public->ons ? public->ons : DEFAULT_NSID;
  a = public->ons ? public->omk : (u_char *) & public->oip;
  len = nsid_len[ons];

  while (data) {
    if ((data->ons ? data->ons : DEFAULT_NSID) == ons &&
	!memcmp(a, data->ons ? data->omk : (u_char *) & data->oip, len) &&
	data->private &&
	!int_compare(data->g, public->g) &&
	!int_compare(data->p, public->p)) {
      break;
    }
    data = data->next;
    if (load == 0 && data == 0) {
      load = 1;
      data = disk_load(ons, a, w, 1);
    }
  }
  if (!data)
    return -1;
  if (!public->j)
    int_init(&(public->j));
  if (!public->kij)
    int_init(&(public->kij));
  if (public->private != NULL &&
      strcmp(public->private, data->private) == 0 &&
      data->kij != NULL) {
    assert(!int_compare(public->j, data->j));
    int_copy(public->kij, data->kij);
  }
  else {
    if (public->private)
      free(public->private);
    public->private = strdup(data->private);
    int_copy(public->j, data->j);
    int_modexp(public->kij, public->ki, public->j, public->p);
  }
  if (public->secret)
    free(public->secret);
  public->secret_len = int_extract_raw(&(public->secret), public->kij,
				       (int_bitsize(public->p) + 7) / 8);
  return 0;
}

int db_init(void)
{
  return 0;
}

void db_exit(void)
{
  while (cert_first) {
    db_cert_drop(cert_first);
  }
}

dh_data *
 db_cert_init(void)
{
  dh_data *res = (dh_data *) malloc(sizeof(dh_data));

  memset(res, 0, sizeof(dh_data));
  res->next = cert_first;
  if (cert_first)
    cert_first->prev = res;
  cert_first = res;
  return res;
}

void db_cert_drop(dh_data * data)
{
  if (data->secret)
    free(data->secret);
  if (data->private)
    free(data->private);
  if (data->public)
    free(data->public);
  if (data->p)
    int_drop(data->p);
  if (data->g)
    int_drop(data->g);
  if (data->j)
    int_drop(data->j);
  if (data->kij)
    int_drop(data->kij);
  if (data->omk)
    free(data->omk);
  if (data->rmk)
    free(data->rmk);
  if (data->old_omk && data->old_omk != data->omk)
    free(data->old_omk);
  if (data->old_rmk && data->old_rmk != data->rmk)
    free(data->old_rmk);
#ifdef USE_SUNCERT
  if (data->cert_data)
    enlink_drop(data->cert_data);
#else
  assert(!data->cert_data);
#endif
  if (data->prev)
    data->prev->next = data->next;
  else
    cert_first = data->next;
  if (data->next)
    data->next->prev = data->prev;
  free(data);
}

dh_data *db_secret_read(char *fnam)
{
  dh_data *res;
  char buf[10240];
  int len;
  FILE *f;

  if (!(f = fopen(fnam, "r")))
    return NULL;

  res = db_cert_init();
  res->private = strdup(fnam);

  int_init(&(res->p));
  int_init(&(res->g));
  int_init(&(res->j));

  len = get_line(f, buf, sizeof(buf));
  if (len > 0) {
    res->begins = strtoul(buf, (char **) NULL, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  if (len > 0) {
    res->expires = strtoul(buf, (char **) NULL, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  if (len > 0) {
    int_set_str(res->g, buf, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  if (len > 0) {
    int_set_str(res->p, buf, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  fclose(f);
  if (len > 0) {
    int_set_str(res->j, buf, 16);

    /* could now check validity date and signatures */
    /* might also create a public key if needed ? */
    /* should also validate timing constraints */
    return res;
  }

  db_cert_drop(res);
  return NULL;
}

dh_data *db_public_read(char *fnam)
{
  dh_data *res;
  char buf[10240];
  int len;
  FILE *f;

  if (!(f = fopen(fnam, "r")))
    return NULL;

  res = db_cert_init();
  res->public = strdup(fnam);

  int_init(&(res->p));
  int_init(&(res->g));
  int_init(&(res->ki));

  len = get_line(f, buf, sizeof(buf));
  if (len > 0) {
    res->begins = strtoul(buf, (char **) NULL, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  if (len > 0) {
    res->expires = strtoul(buf, (char **) NULL, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  if (len > 0) {
    int_set_str(res->g, buf, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  if (len > 0) {
    int_set_str(res->p, buf, 16);
    len = get_line(f, buf, sizeof(buf));
  }
  if (len <= 0) {
    fclose(f);
    db_cert_drop(res);
    return NULL;
  }
  int_set_str(res->ki, buf, 16);

  /* should now read certificate data and validate public key!!! */

#ifdef USE_SUNCERT
  len = get_line(f, buf, sizeof(buf));
  if (len > 0) {
    int certlen = atoi(buf);

    if (certlen) {
      /* only for nsid 1 for now (caretaker) */
      len = get_line(f, buf, sizeof(buf));
      if (len <= 0) {
	fclose(f);
	db_cert_drop(res);
	return NULL;
      }
      /* certlen to long (contains backlash and LF) */
      assert(certlen == len);
      res->cert_data = enlink_readhex(buf, len);
      /*if (res==NULL || !enlink_validate(res->public,res->cert_data)) { */
      if (res == NULL) {
	fclose(f);
	db_cert_drop(res);
	return NULL;
      }

      /* could extract and check public values here */

    }
  }
#else

  /* should validate timing constraints here */

#endif


  fclose(f);
  return res;
}

int db_public_write(FILE * f, dh_data * data, char *name, char *from)
{
  char *hex, *t;
  time_t tim = time(NULL);

  fprintf(f, "#Public Value for %s\n", name);
  fprintf(f, "#provided by %s\n", from);
  fprintf(f, "#%s\n\n", ctime(&tim));
  fprintf(f, "#valid since (hex)\n%08lX\n\n", data->begins);
  fprintf(f, "#valid until (hex)\n%08lX\n\n", data->expires);
  fprintf(f, "#Base g (hex)\n");
  hex = int_extract_hex(data->g);
  for (t = hex; strlen(t) > 70; t += 70)
    fprintf(f, "%.70s\\\n", t);
  fprintf(f, "%s\n\n", t);
  free(hex);
  fprintf(f, "#Modulus p (hex)\n");
  hex = int_extract_hex(data->p);
  for (t = hex; strlen(t) > 70; t += 70)
    fprintf(f, "%.70s\\\n", t);
  fprintf(f, "%s\n\n", t);
  free(hex);
  fprintf(f, "#Public Value (hex)\n");
  hex = int_extract_hex(data->ki);
  for (t = hex; strlen(t) > 70; t += 70)
    fprintf(f, "%.70s\\\n", t);
  fprintf(f, "%s\n\n", t);
  free(hex);
#ifdef USE_SUNCERT
  fprintf(f, "#Certificate Data\n");
  if (data->cert_data)
    enlink_savehex(f, data->cert_data);
  else
    fprintf(f, "0\n");
#else
  fprintf(f, "#Certificate Data (not yet implemented)\n0\n");
#endif
  return 0;
}

int db_secret_write(FILE * f, dh_data * data, char *name, char *from)
{
  char *hex, *t;
  time_t tim = time(NULL);

  fprintf(f, "#Secret value for %s\n", name);
  fprintf(f, "#created by %s\n", from);
  fprintf(f, "#%s\n\n", ctime(&tim));
  fprintf(f, "#valid since (hex)\n%08lX\n\n", data->begins);
  fprintf(f, "#valid until (hex)\n%08lX\n\n", data->expires);
  fprintf(f, "#Base g (hex)\n");
  hex = int_extract_hex(data->g);
  for (t = hex; strlen(t) > 70; t += 70)
    fprintf(f, "%.70s\\\n", t);
  fprintf(f, "%s\n\n", t);
  free(hex);
  fprintf(f, "#Modulus p (hex)\n");
  hex = int_extract_hex(data->p);
  for (t = hex; strlen(t) > 70; t += 70)
    fprintf(f, "%.70s\\\n", t);
  fprintf(f, "%s\n\n", t);
  free(hex);
  fprintf(f, "#Secret Value (hex)\n");
  hex = int_extract_hex(data->j);
  for (t = hex; strlen(t) > 70; t += 70)
    fprintf(f, "%.70s\\\n", t);
  fprintf(f, "%s\n\n", t);
  free(hex);
  return 0;
}

void db_cert_dump(FILE * f, dh_data * data)
{
  /*printf("g: %s\n",int_extract_hex(res->g)); */
}
