/*
   Copyright (C) 1996-1997 Robert Muchsel

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Please address all correspondence concerning the software to
   muchsel@acm.org.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>
#include <unistd.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <time.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "cert_defs.h"
#include "cert_db.h"
#include "cert_int.h"
#include "md5.h"
#include "dh_512_params.h"
#include "dh_1024_params.h"
#include "dh_2048_params.h"
#include "bn/bn.h"
#include "../include/id.h"

/* Unix start */
#define UNIXSTART       2208988800U     /* 1-1-70 0h in NTP time */

/* time now in NTP */
#define NTP_TO_UNIX(n)                  (n-UNIXSTART)
#define UNIX_TO_NTP(n)                  (n+UNIXSTART)
#define NTPNOW          		UNIX_TO_NTP(time(0))


int make_cert_08(dh_data *find, u_char *buf) 
{
    u_char *raw, *t = buf;
    int l;

    *(u_long *) t = htonl(find->begins);
    t += sizeof(u_long);
    *(u_long *) t = htonl(find->expires);
    t += sizeof(u_long);
    l = int_extract_raw(&raw, find->p, 0); 
    assert(l < 65536);
    *t++ = l >> 8;
    *t++ = l & 0xff;
    memcpy(t, raw, l);
    t += l;
    free(raw);
    /* from now on, alignment is not guaranteed */
    l = int_extract_raw(&raw, find->g, 0); 
    assert(l < 65536);
    *t++ = l >> 8;
    *t++ = l & 0xff;
    memcpy(t, raw, l);
    t += l;
    free(raw);
    l = int_extract_raw(&raw, find->ki, 0); 
    assert(l<65536);
    *t++ = l >> 8;
    *t++ = l & 0xff;
    memcpy(t, raw, l);
    t += l;
    free(raw);

    return t - buf;
}


int main(int argc, char *argv[])
{
    dh_data *data;
    u_char buf[1024];
    int bytes_len;
    struct MD5Context ctxt;
#ifdef HAVE_DEV_RANDOM
    int f;
#endif
    int mod_size;
    char random_value[16];
    time_t t;
    char name[1024];
    FILE *out;

    if (argc != 1) {
        fprintf(stderr,"Usage: %s\n",argv[0]);
        exit(-1);
    }

    assert(!db_init());
    assert(!int_mathinit());

    /* Ask the user about his preferences */
    do {
      mod_size = 0;
      printf("Please enter the modulus size (512, 1024 or 2048): ");
      fflush(stdout);
      scanf("%i", &mod_size);
    } while ((mod_size != 512) && (mod_size != 1024) && (mod_size != 2048));

    /* Generate/get secret value */
#ifdef HAVE_DEV_RANDOM
    printf("Generating random value...\n");
    f = open("/dev/random", O_RDONLY);
    if (f < 0) {
      perror("Cannot read random device /dev/random");
      exit(1);
    }
    printf("If the program suspends here, please switch tasks and generate\n");
    printf("some keyboard input.\n");
    read(f, &random_value, sizeof(random_value));
    printf("Random generation succeeded.\n");
    close(f);
#else
    printf("Please enter a secret value: ");
    fflush(stdout);
    scanf("%s", random_value);
#endif

    data = db_cert_init();
    data->rns = SKIP_NSID_DH;

    /* valid from now on (minus one hour) */
    t = NTPNOW;
    data->begins = t - 60 * 60;

    /* make it last 2 years */
    t += 365 * 24 * 60 * 60;

    data->expires = t;

    int_init(&(data->g));
    int_init(&(data->p));
    int_init(&(data->ki));
    int_init(&(data->j));

    if (mod_size == 512) {
      bnInsertBigBytes(data->g, base_512, 0, sizeof(base_512));
      bnInsertBigBytes(data->p, mod_512, 0, sizeof(mod_512));
    }
    else if (mod_size == 1024) {
      bnInsertBigBytes(data->g, base_1024, 0, sizeof(base_1024));
      bnInsertBigBytes(data->p, mod_1024, 0, sizeof(mod_1024));
    } 
    else {
      bnInsertBigBytes(data->g, base_2048, 0, sizeof(base_2048));
      bnInsertBigBytes(data->p, mod_2048, 0, sizeof(mod_2048));
    }
    bnInsertBigBytes(data->j, random_value, 0, sizeof(random_value));

    int_modexp(data->ki, data->g, data->j, data->p);

    bytes_len = make_cert_08(data, buf);
    data->rmk = malloc(nsid_len[data->rns]);
    cert_MD5Init(&ctxt);
    cert_MD5Update(&ctxt, buf, bytes_len);
    cert_MD5Final(data->rmk, &ctxt);

    sprintf(name, "%s/public/%s1", CERTDIR, make_name(data->rns, data->rmk));
    out = fopen(name, "w");
    if (out == NULL) {
      fprintf(stderr, "Cannot create %s\n", name);
      exit(1);
    }
    db_public_write(out, data, "<???>", "<cert_gen>");
    fclose(out);
    chmod(name, 0644);

    printf("Public CERT saved as %s\n", name);

    sprintf(name, "%s/secret/%s1", CERTDIR, make_name(data->rns, data->rmk));
    out = fopen(name, "w");
    if (out == NULL) {
      fprintf(stderr, "Cannot create %s\n", name);
      exit(1);
    }
    db_secret_write(out, data, "<???>", "<cert_gen>");
    fclose(out);
    chmod(name, 0600);

    printf("Secret CERT saved as %s\n", name);

    db_cert_drop(data);

    return 0;
}

