/* (c) by G. Caronni in 1995                              18.11.95
 *
 * Contains an interface to arbitrary MPI packages
 *
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include "cert_defs.h"
#include "cert_int.h"

/******************************************************************/

#if defined(USE_GMP)

/******************************************************************/

#include "gmp/gmp.h"


int int_mathinit(void) {return 0;}
void int_mathexit(void) {}

int int_init(void **where)
{
    *where=(void *)malloc(sizeof(MP_INT));
    mpz_init((MP_INT *)*where);
    return 0;
}

void int_drop(void *mpi)
{
    mpz_clear((MP_INT *)mpi);
    free(mpi);
}

int int_modexp(void *res, void *g, void *j, void *p)
{
    mpz_powm(res,g,j,p);
    return 0;
}

int int_extract_raw(unsigned char **where, void *mpi, int fix_size)
{
    MP_INT extr;
    int len,i;

    mpz_init_set(&extr,(MP_INT *)mpi);
    len=(mpz_sizeinbase(&extr,2)+7)/8;
    *where=(unsigned char *)malloc(len<fix_size?fix_size:len);
    memset(*where,0,len<fix_size?fix_size:len);
    for(i=(len<fix_size?fix_size:len)-1;i>=0;i--) {
        (*where)[i]= (unsigned char) (mpz_get_ui(&extr) & 0xff);
        mpz_div_2exp(&extr,&extr,(unsigned long int)8);
    }
    return len;
}

char *int_extract_hex(void *mpi)
{
    int len,i;
    unsigned char *raw, *s;
    char *buf, *t;

    len=int_extract_raw(&raw,mpi,0);
    buf=(char *)malloc(2*len+1);
    for(t=buf,s=raw,i=0;i<len;i++,t+=2,s++) sprintf(t,"%.2X",*s);
    free(raw);
    return buf;
}

int int_copy(void *to, void *from)
{
    mpz_set((MP_INT *)to,(MP_INT *)from);
    return 0;
}

int int_compare(void *a, void *b)
{
    return mpz_cmp((MP_INT *)a,(MP_INT *)b);
}

int int_set_str(void *val, char *str, int base) 
{
    return mpz_set_str(val,str,base);
}

int int_bitsize(void *mpi)
{
    return mpz_sizeinbase((MP_INT *)mpi,2);
}

/******************************************************************/

#elif defined(USE_BN)

/******************************************************************/

#include "bn/bn.h"

#ifdef __GNUC__
#ident "$Id: cert_int.c,v 1.5 1996/04/25 14:54:20 hauber Exp $"
#else
static char rcsid[] = "$Id: cert_int.c,v 1.5 1996/04/25 14:54:20 hauber Exp $";
#endif


int int_mathinit(void) {bnInit(); return 0;}
void int_mathexit(void) {}

int int_init(void **where)
{
    *where=(void *)malloc(sizeof(struct BigNum));
    bnBegin((struct BigNum *)*where);
    return 0;
}

void int_drop(void *mpi)
{
    bnEnd((struct BigNum *)mpi);
    free(mpi);
}

int int_modexp(void *res, void *g, void *j, void *p)
{
    /* WARNING: Does not work for even p */
    return bnExpMod(res,g,j,p);
}

int int_extract_raw(unsigned char **where, void *mpi, int fix_size)
{
    int len;

    len=((bnBits((struct BigNum *)mpi)+7)/8);
    *where=(unsigned char *)malloc(len<fix_size?fix_size:len);
    memset(*where,0,len<fix_size?fix_size:len);
    bnExtractBigBytes((struct BigNum *)mpi, *where, 0, len);
    return len;
}

char *int_extract_hex(void *mpi)
{
    int len,i;
    unsigned char *raw, *s;
    char *buf, *t;

    len=int_extract_raw(&raw,mpi,0);
    buf=(char *)malloc(2*len+1);
    for(t=buf,s=raw,i=0;i<len;i++,t+=2,s++) sprintf(t,"%.2X",*s);
    free(raw);
    return buf;
}

int int_copy(void *to, void *from)
{
    return bnCopy((struct BigNum *)to,(struct BigNum *)from);
}

int int_compare(void *a, void *b)
{
    return bnCmp((struct BigNum *)a,(struct BigNum *)b);
}

int int_set_str(void *val, char *str, int base) 
{
    struct BigNum *v=(struct BigNum *)val;
    char *t=str;
    int i;

    bnSetQ(v,0);
    if (!str) return 0;
    if (base==0 || base>36) return -1;
    while(*t) {
        if (*t >= '0' && *t <= '9') i= *t-'0';
        else if (*t >= 'A' && *t <= 'Z') i= *t-'A'+10;
        else if (*t >= 'a' && *t <= 'z') i= *t-'a'+10;
        else break;
        if (i>=base) return 0;
        if (base==16) bnLShift(v,4);
        else bnMulQ(v,v,base);
        bnAddQ(v,i);
        t++;
    }
    return 0;
}

int int_bitsize(void *mpi)
{
    return bnBits((struct BigNum *)mpi);
}

/******************************************************************/

#else

#error please specify package in the makefile

#endif
