/* cert_main.c -- (c) by G. Caronni in 1995             19.11.95
 *
 * Interface to the certificate part of the skip daemon
 *
 * small changes: Robert Muchsel 1996-1997
 */

/* #define DEBUG_CERT */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <netinet/in.h>

#include "cert_defs.h"
#include "cert.h"
#include "cert_db.h"
#include "cert_int.h"
#include "cert_event.h"
#include "cert_udp.h"
#include "../include/id.h"

#ifdef UBS_CHIPCARD
#include "cert_ubs.h"
#endif

#ifdef __GNUC__
#ident "$Id: cert_main.c,v 1.13 1996/06/07 11:30:52 skip Exp $"
#else
static char rcsid[] = "$Id: cert_main.c,v 1.13 1996/06/07 11:30:52 skip Exp $";
#endif


#ifdef SOLARIS
/* Missing in Solaris 2.3 */
extern int gettimeofday(struct timeval *tp, struct timezone *tzp);
#endif


/* Is used by do_discover to store the parameters of a pending request
 * discover_timeout gets this from the event handler for timeouts, 
 * db_complete from event find if a reply comes in via udp_process. */

struct request_data {
  ip_addr oip;
  ip_addr rip;
  u_char ons;
  u_char rns;
  u_char *omk;
  u_char *rmk;
  int retry;
  int nr;
  int maxnr;
  ip_addr *addr;
  int ireq;
};


void (*receiver) (dh_data * reply);


static void do_discover(void *opaque_data, struct timeval *when);

static int find_req_event(void **next,
			  struct request_data *req,
			  struct request_data **data)
{
  void *param = NULL;
  void (*fct) (void *, struct timeval *) = do_discover;
  u_char *a, *b;
  int len;
  int full = 0;

  while ((*next = event_find(*next, NULL, &fct, &param))) {
    *data = (struct request_data *) param;
    param = NULL;
    if (req->ons) {
      len = nsid_len[req->ons];
      a = req->omk;
    }
    else {
      len = nsid_len[DEFAULT_NSID];
      a = (u_char *) & (req->oip);
    }
    if ((*data)->ons)
      b = (*data)->omk;
    else
      b = (u_char *) & ((*data)->oip);

    if ((req->ons ? req->ons : DEFAULT_NSID) ==
	((*data)->ons ? (*data)->ons : DEFAULT_NSID) &&
	!memcmp(a, b, len))
      full = 1;
    else
      full = 0;
    if (req->rns) {
      len = nsid_len[req->rns];
      a = req->rmk;
    }
    else {
      len = nsid_len[DEFAULT_NSID];
      a = (u_char *) & (req->rip);
    }
    if ((*data)->rns)
      b = (*data)->rmk;
    else
      b = (u_char *) & ((*data)->rip);
    if ((req->rns ? req->rns : DEFAULT_NSID) ==
	((*data)->rns ? (*data)->rns : DEFAULT_NSID) &&
	!memcmp(a, b, len))
      break;
  }
  return *next ? full + 1 : 0;
}

static void drop_req(struct request_data *req)
{
  if (req->omk)
    free(req->omk);
  if (req->rmk)
    free(req->rmk);
  if (req->addr)
    free(req->addr);
  free(req);
}

/* If remote is already complete and contains correct ID info, return it
 * after incrementing export. Decide when cloning it is needed, and when 
 * updating export counter is sufficent. (depending on n, shared secret?)
 * Otherwise load/find own matching secret dh_data, complete the remote, 
 * and return via receiver backcall, after incrementing export of remote.
 * Remove any scheduled events for this request ID.
 *
 * export does not exist anymore!
 */
static void create_reply(struct request_data *req, dh_data *data)
{
  int clone = 0, negative = 0;

  /* the q&d solution */

  if (!data) {
    negative = 1;
    data = db_cert_init();
  }
  else {
    data = db_public_read(data->public);
  }
  data->ireq = req->ireq;
  data->oip = req->oip;
  data->rip = req->rip;
  data->ons = req->ons;
  data->rns = req->rns;
  data->omk = req->ons ? memcpy(malloc(nsid_len[req->ons]), req->omk,
				nsid_len[req->ons]) : NULL;
  data->rmk = req->rns ? memcpy(malloc(nsid_len[req->rns]), req->rmk,
				nsid_len[req->rns]) : NULL;

  data->old_ons = data->ons;
  data->old_rns = data->rns;
  data->old_omk = data->omk;
  data->old_rmk = data->rmk;

  if (!negative)
    db_secret_add(data);

  /* interoperability fix!!! */

  if (data->omk == data->old_omk)
    clone = 1;

  if (data->ons && data->omk && nsid_len[data->ons] < SKIP_ID_MAXLEN) {
    data->omk = realloc(data->omk, SKIP_ID_MAXLEN);
    memset((data->omk) + nsid_len[data->ons], 0, SKIP_ID_MAXLEN - nsid_len[data->ons]);
  }
  if (clone) {
    clone = 0;
    data->old_omk = data->omk;
  }
  else {
    if (data->old_ons && data->old_omk && nsid_len[data->old_ons] < SKIP_ID_MAXLEN) {
      data->old_omk = realloc(data->old_omk, SKIP_ID_MAXLEN);
      memset((data->old_omk) + nsid_len[data->old_ons], 0,
	     SKIP_ID_MAXLEN - nsid_len[data->old_ons]);
    }
  }
  if (data->rmk == data->old_rmk)
    clone = 1;

  if (data->rns && data->rmk && nsid_len[data->rns] < SKIP_ID_MAXLEN) {
    data->rmk = realloc(data->rmk, SKIP_ID_MAXLEN);
    memset((data->rmk) + nsid_len[data->rns], 0, SKIP_ID_MAXLEN - nsid_len[data->rns]);
  }
  if (clone) {
    clone = 0;
    data->old_rmk = data->rmk;
  }
  else {
    if (data->old_rns && data->old_rmk && nsid_len[data->old_rns] < SKIP_ID_MAXLEN) {
      data->old_rmk = realloc(data->old_rmk, SKIP_ID_MAXLEN);
      memset((data->old_rmk) + nsid_len[data->old_rns], 0,
	     SKIP_ID_MAXLEN - nsid_len[data->old_rns]);
    }
  }

#ifdef DEBUG_CERT
  {
    int i;

    printf("CERT receiver\n");
    printf("OIP=");
    for (i = 0; i < SKIP_ID_MAXLEN; i++)
      printf("%02X", ((u_char *) & data->oip)[i]);
    printf(" ");
    printf("RIP=");
    for (i = 0; i < SKIP_ID_MAXLEN; i++)
      printf("%02X", ((u_char *) & data->rip)[i]);
    printf("\n");
    printf("OMK=");
    if (data->omk) {
      for (i = 0; i < SKIP_ID_MAXLEN; i++)
	printf("%02X", ((u_char *) data->omk)[i]);
    }
    else
      printf("<null>\t\t\t    ");
    printf(" ");
    printf("RMK=");
    if (data->rmk) {
      for (i = 0; i < SKIP_ID_MAXLEN; i++)
	printf("%02X", ((u_char *) data->rmk)[i]);
    }
    else
      printf("<null>");
    printf("\n");
    printf("ONS=%d, RNS=%d, S-LEN=%d\n", data->ons, data->rns,
	   data->secret_len);
  }
#endif
  receiver(data);

  if (negative)
    db_cert_drop(data);

  /*
     how do we handle the issue of matching multiple public values of one
     host against the strongest own available secret value???

     extract own nsid mkid from req 
     look for an exported one with full match in IDs
     if this one is (-1) recall find_load to force load
     if it exists update n if needed and pass it to receive
     if not, look for a partial match that is not exported 
     if found drop secret part
     if there are only exported ones, clone one of them, drop secret part
     set own ID fields
     call db_secret_add
     pass it to receive
   */
}

/* Gets called by cert_discover or udp_process. udp_process provides
 * dh_data in remote on success or NULL on failure.
 * In any case event_find has to be used to find request_data with
 * matching remote ID part. (May be more than one at the same time!)
 * Thus the own part can be read, and handled.
 * If udp_process gives us a failure, we look for any active requests
 * that are interested in this IP currently, and cause them to skip
 * to the next IP immediately, by calling do_discover for them ;-)
 */
void do_discovered(u_char nsid, u_char *mkid, dh_data *remote, u_long ip)
{
  void *next, *last_next;
  struct request_data *found, point;
  struct timeval tim;

  assert(nsid && mkid);

  memset(&point, 0, sizeof(point));
  point.rns = nsid;
  point.rmk = mkid;
  gettimeofday(&tim, NULL);
  next = last_next = NULL;

  /* generates one reply for each partial match if remote */
  while (find_req_event(&next, &point, &found)) {
    if (remote) {
      /* remove any later full match */
      void *full_next = next;
      struct request_data *full_found;

      while (find_req_event(&full_next, found, &full_found) == 2) {
	drop_req(found);
	event_remove(NULL, do_discover, found);
	found = full_found;
      }
      create_reply(found, remote);
      drop_req(found);
      event_remove(NULL, do_discover, found);
      next = last_next;
    }
    else {
#if 0
      /* reschedule if appropriate, delete nothing else */
      /* if there is any later full match, it will try different IPs */
      if (found->nr >= 0 && found->nr < found->maxnr &&
	  found->addr[found->nr].a[0] &&
	  found->addr[found->nr].a[0] == ip &&
	  found->retry < CERT_RETRIES) {
	found->retry = CERT_RETRIES;
	event_remove(NULL, do_discover, found);
	assert(!event_store(&tim, do_discover, found));
	next = last_next;
      }

      /* XXX should generate a 'not found' reply! */
#else
      /* reschedule in any case, as this is used to trigger
       * not found replies in do_discover !
       */
      if (found->nr >= 0) {
	found->retry = CERT_RETRIES;
	event_remove(NULL, do_discover, found);
	assert(!event_store(&tim, do_discover, found));
	next = last_next;
      }

#endif
    }
    last_next = next;
  }
}

/* Gets called by event timeout
 * Creates an udp certificate request for the appropriate destination
 * and re-registers itself.
 * If all destinations timed out, create a failed event entry.
 * Schedule its deletion for after REMEMBER_FAILURE
 * If a response does come in later, the deletion will be removed.
 * if a response comes in after deletion, at least a local cache entry
 * will be established by udp_process.
 */
static void do_discover(void *opaque_data, struct timeval *when)
{
/* currently this simply dies, as we can not yet send requests
 * later it should check if this is a
 * - new start
 * - intermediate timeout for one IP
 * - last timeout for one IP
 * - last timeout for last IP
 * - a REMEMBER_FAILURE timeout
 * If all IPs are timed out, this is treated as a complete failure.
 * Take care that some IP may reply via udp_process and tell us, that
 * they do not know about this certificate. see do_discovered.
 */

  struct request_data *req = (struct request_data *) opaque_data;

  if (req->nr < 0) {
    /* this is a REMEMBER_FAILURE timeout -- simply die */
    drop_req(req);
    return;
  }
  if (req->retry >= CERT_RETRIES) {
    req->retry = 0;
    req->nr++;
  }
  while (req->nr < req->maxnr && !req->addr[req->nr].a[0])
    req->nr++;			/*IPv4 */
  gettimeofday(when, NULL);
  if (req->nr < req->maxnr) {
    req->retry++;
    udp_request(req->addr[req->nr].a[0],	/* IPv4 only, for now */
		req->ons ? req->ons : DEFAULT_NSID,
		req->ons ? req->omk : (u_char *) & (req->oip),
		req->rns ? req->rns : DEFAULT_NSID,
		req->rns ? req->rmk : (u_char *) & (req->rip));
    when->tv_usec += RETRY_DELAY * 1000;
    if (when->tv_usec > 999999) {
      when->tv_usec -= 1000000;
      when->tv_sec++;
    }
  }
  else {
    req->nr = (-1);
    when->tv_sec += REMEMBER_FAILURE;

    /* Now generate negative reply to skipd */
    create_reply(req, NULL);

  }
  assert(!event_store(when, do_discover, req));
}


void cert_discover(u_long * oip, u_long * rip, u_char ons, u_char rns,
		   u_char * omk, u_char * rmk, int lookup_nr,
		   u_long * lookup_hosts, int ireq)
{
  int i, j, match, full;
  dh_data *find;
  void *next;
  struct timeval tim;
  struct request_data *on, *req = (struct request_data *) malloc(sizeof(struct request_data));

#ifdef DEBUG_CERT
  printf("CERT discover\n");
  if (oip) {
    printf("OIP=");
    for (i = 0; i < SKIP_ID_MAXLEN; i++)
      printf("%02X", ((u_char *) oip)[i]);
  }
  else
    printf("<null>\t\t\t     ");
  if (rip) {
    printf("RIP=");
    for (i = 0; i < SKIP_ID_MAXLEN; i++)
      printf("%02X", ((u_char *) rip)[i]);
  }
  else
    printf("<null>");
  printf("\nOMK=");
  if (omk) {
    for (i = 0; i < SKIP_ID_MAXLEN; i++)
      printf("%02X", ((u_char *) omk)[i]);
  }
  else
    printf("<null>\t\t\t    ");
  printf(" RMK=");
  if (rmk) {
    for (i = 0; i < SKIP_ID_MAXLEN; i++)
      printf("%02X", ((u_char *) rmk)[i]);
  }
  else
    printf("<null>");
  printf("\nONS=%d, RNS=%d, L-NR=%d\n", ons, rns, lookup_nr);
#endif

  assert(ons <= MAX_NSID);
  assert(rns <= MAX_NSID);
  assert((!lookup_nr) || lookup_hosts);
  if (!lookup_nr)
    lookup_hosts = NULL;

#ifdef UBS_CHIPCARD
  if (ons == SKIP_NSID_UBS_CHIP || rns == SKIP_NSID_UBS_CHIP) {
    free(req);
    ubs_discover(oip, rip, ons, rns, omk, rmk, ireq);
    return;
  }
#endif

  memcpy(req->oip.a, oip, sizeof(ip_addr));
  memcpy(req->rip.a, rip, sizeof(ip_addr));

  req->ons = ons;
  req->rns = rns;
  req->omk = ons ? memcpy(malloc(nsid_len[ons]), omk, nsid_len[ons]) : NULL;
  req->rmk = rns ? memcpy(malloc(nsid_len[rns]), rmk, nsid_len[rns]) : NULL;
  req->retry = req->nr = 0;
  req->maxnr = lookup_nr + 2;
  /* could filter own IP address here */
  i = lookup_nr * sizeof(ip_addr);
  req->addr = (ip_addr *) malloc(i + 2 * sizeof(ip_addr));
  if (lookup_hosts)
    memcpy(req->addr[2].a, lookup_hosts, i);
  memcpy(req->addr, oip, sizeof(ip_addr));
  memcpy(req->addr[1].a, rip, sizeof(ip_addr));
  req->ireq = ireq;

#ifdef DEBUG_CERT
  printf("L-ADDR=");
  if (req->addr) {
    for (i = 0; i < sizeof(ip_addr) * req->maxnr; i++) {
      printf("%02X", ((u_char *) req->addr)[i]);
      if (i < sizeof(ip_addr) * req->maxnr && (i + 1) % 32 == 0)
	printf("\n       ");
      else if (i < sizeof(ip_addr) * req->maxnr && (i + 1) % 16 == 0)
	printf(" ");
    }
  }
  else
    printf("<null>");
  printf("\n");
#endif

  /* If there is a matching partial request, compare IP addresses to ask,
   * and remove the ones that have already been asked in the old one.
   * They will only be reasked if a new query is submit after the old one
   * incurred a FAILURE_TIMEOUT.
   * If the new request is a full match, and if there are no new IP 
   * numbers for the discovery, then forget the whole query.
   * If this is only a partial match, it does not matter, if there is
   * nobody anymore to ask. Remember the request anyway, perhaps a reply
   * for the old one will come, which would then fulfill this one too.
   */
  next = NULL;
  full = 0;
  while ((match = find_req_event(&next, req, &on))) {
    ip_addr *a = on->addr;

    for (i = 0; i < on->maxnr; i++, a++) {
      ip_addr *b = req->addr;

      for (j = 0; j < req->maxnr; j++, b++)
	if (a->a[0] == b->a[0])
	  b->a[0] = 0;		/* IPv4 */
    }
    if (match == 2)
      full = 1;
  }
  if (full) {
    ip_addr *b = req->addr;

    for (j = 0; j < req->maxnr; j++, b++)
      if (b->a[0])
	break;			/* IPv4 */
    if (j >= req->maxnr) {
      drop_req(req);
      return;
    }
  }

  gettimeofday(&tim, NULL);
  /* nasty - but pushing req always cleans up a lot of other detours */
  assert(!event_store(&tim, do_discover, req));

  find = db_public_find_load(NULL, rns ? rns : DEFAULT_NSID,
			     rns ? rmk : (u_char *) (&rip));
  if (find == (dh_data *) (-1))
    find = db_public_find_load(find, rns ? rns : DEFAULT_NSID,
			       rns ? rmk : (u_char *) (&rip));

  if (find)
    do_discovered(rns ? rns : DEFAULT_NSID, rns ? rmk : (u_char *) (&rip), find, 0);
}


int cert_iowait(void)
{
  fd_set rdfs;
  struct timeval *etim;
  int done;

  etim = event_ripe_handle_time(&done);
  FD_ZERO(&rdfs);
  FD_SET(send_socket, &rdfs);
  FD_SET(recv_socket, &rdfs);
  errno = 0;
  done = select(FD_SETSIZE, &rdfs, NULL, NULL, etim);
  if (done < 0 && errno != EINTR) {
    perror("select");
  }
  assert(done >= 0 || errno == EINTR);
  if (done > 0) {
    if (FD_ISSET(send_socket, &rdfs))
      udp_process_responder(send_socket);
    else
      udp_process_requestor(recv_socket);
  }
  return 0;
}


int cert_init(void (*skipd_receiver) (dh_data * reply), char *basedir)
{
  int r;

  receiver = skipd_receiver;

  data_path = malloc(strlen(basedir) + 2);
  strcpy(data_path, basedir);
  if (*data_path && data_path[strlen(data_path) - 1] != '/')
    strcat(data_path, "/");

  if ((r = db_init()))
    return r;
  if ((r = event_init())) {
    db_exit();
    return r;
  }
  if ((r = udp_init())) {
    event_exit();
    db_exit();
    return r;
  }
  if ((r = int_mathinit())) {
    udp_exit();
    event_exit();
    db_exit();
    return r;
  }
#ifdef UBS_CHIPCARD
  if ((r = ubs_init())) {
    int_mathexit();
    udp_exit();
    event_exit();
    db_exit();
    return r;
  }
#endif

  /* 
   * could now register the aging routine that kicks out dh_data
   * structures which have reached time out and are unexported
   */

  {
#ifdef CERT_TESTER
    static void tester();
#endif
    struct timeval tim;

    gettimeofday(&tim, NULL);
    tim.tv_sec += 2;
    /* assert(!event_store(&tim,tester,NULL)); */
  }

  return 0;
}


void cert_exit(void)
{
#ifdef UBS_CHIPCARD
  ubs_exit();
#endif
  int_mathexit();
  event_exit();
  udp_exit();
  db_exit();
}

#if 0
void cert_dump(FILE * f)
{
  fprintf(f, "You bet...\n");
}
#endif

#ifdef CERT_TESTER
static void tester(void *a, void *b)
{
  u_long f[4] = { htonl(0x81844208), 7, 8, 9 };
  u_long h[4] = { htonl(0x81844206), 7, 8, 9 };

  cert_discover(f, h, SKIP_NSID_UBS_CHIP, SKIP_NSID_UBS_CHIP, 
                (u_char *) f, (u_char *) h, 0, NULL, 0);
}
#endif

