/*
 * Copyright (C) 1995 M. Hauber, Ch. Schneider, G. Caronni
 * See COPYING for more details
 */
#include "id.h"

typedef struct skipcache_stat
{
  u_long entries;		/* number of entries in cache */
  u_long inserted;		/* number of newly inserted entries */
  u_long updated;		/* number of updated entries */
  u_long timedout;		/* number of timedout entries */
  u_long removed;		/* number of removed entries */
  u_long lookuphits;		/* number of lookup hits */
  u_long lookuprequests;	/* number of requests sent */
} skipcache_stat_t;

extern skipcache_stat_t skipcache_stat;


struct skip_Kn				/* Key for time n */
{
  u_int32 n;
  skip_dynamic_t key;
};

struct skipcache
{
  skip_id_t id;				/* IPs and/or Master-Key-IDs */
  struct skipcache *next, *prev;	/* doubly linked hash chain */
  int ttl;				/* time to live in cache */
  int maxttl;				/* maximum ttl of this entry */
  int keyttl;				/* absolute ttl of the shared key */
  int ref;				/* reference count */
  int flags;				/* flags; see below */

  skip_dynamic_t srcip, dstip;		/* ip addresses for master key id */
  void *map;				/* pointer to mappings ip to entry */

  /*
   * fields used by skipd
   * these two fields need not be passed to the kernel
   */
  skip_dynamic_t sharedkey;             /* Kij */
  skip_dynamic_t lookuphosts;           /* offset to list of lookup hosts */

  struct skip_Kn Kijn[3];		/* offset to key Kij */

  u_int filter_before[2];
  u_int filter_after[2];

  /*
   * cache information for IP packets
   * that will be converted to SKIP packets
   */
  u_int enskip_mode[2];         /* Enskip mode: See flags below */
  u_char enskip_mkid[2];	/* Add MKIDs? */
  int enskip_Kij_alg;		/* algorithm to use for shared key Kij */
  int enskip_Crypt_alg;		/* packet encryption algorithm */
  int enskip_MAC_alg;		/* Integrity check value algorithm */
  int enskip_Comp_alg;		/* compression algorithm */

  int enskip_Kp_bytes;		/* Bytes sent with current packet key */
  int enskip_Kp_maxbytes;	/* Maximum bytes until packet key change */
  int enskip_Kp_ttl;		/* Time to live for current packet key */
  int enskip_Kp_maxttl;		/* Maximum time until packet key change */

  skip_dynamic_t enskip_A_Kp, enskip_E_Kp;  /* offset to packet key Kp */
  struct skip_Kn enskip_Kp_Kijn;	/* key Kp encrypted in Kijn */
  skip_dynamic_t enskip_MI;	/* message indicator */

  /*
   * cache information for SKIP packets
   * that will be converted back to normal IP packets
   */
  u_int deskip_mode[2];         /* Deskip mode: See flags below */
  u_char deskip_policy[2];	/* Policy for in ipsp header */
  int deskip_Kij_alg;		/* last received Kij algorithm */
  int deskip_Kp_Kijn_len;	/* len of Kp_Kijn last received */
  skip_dynamic_t deskip_A_Kp, deskip_E_Kp;	/* offset to last received key Kp */
  struct skip_Kn deskip_Kp_Kijn;  /* offset to last received encrypted key Kp */

  /* private data for crypt algorithms */
  /*
   * Note: we do not need a second encrypt state since we cache Kp
   *       encrypted in Kijn.
   */
  skip_dynamic_t encryptstate;	/* offset to private encrypt state */
  skip_dynamic_t Kp_decryptstate;	/* offset to private Kp decrypt state */
  skip_dynamic_t payload_decryptstate;	/* offset to private payload decrypt state */

  /*
   * the q field is only used at kernel level to enqueue packets
   */
  skip_dynamic_t q;                   /* queues in kernel (see struct below) */

  /*
   * this structure and its coresponding data section are
   * allocated as one area with the size (datalen + sizeof(struct skipcache))
   */
  u_char *data;			/* pointer to data section */
  int datalen;			/* length of data section */
};

#ifdef KERNEL
struct _kernel_queue
{
  QUEUE inq, outq;
};
#endif

#define SKIP_VALIDKEY	     0x1	/* shared secret is available */
#define SKIP_NOKEY	     0x2	/* no key available */
#define SKIP_TUNNEL	     0x4	/* tunnel mode */
#define SKIP_REMOVE	     0x8	/* entry will be removed */

#define SKIP_DONTAGE         0x10       /* do not expire entry */
#define SKIP_MANUALKEY       0x20       /* entry has manual secret */
#define SKIP_CHIPCARD        0x40       /* chip card in use */
#define SKIP_REQSENT_GETENSKIPKEY  0x80       /* key request sent to daemon */
#define SKIP_REQSENT_GETDESKIPKEY  0x100      /* key request sent to daemon */
#define SKIP_REQSENT_GETENTRY      0x200      /* key request sent to daemon */

/*
 * if one of these flags is set the corresponding key is in ??skip_Kp_Kijn
 */
#define SKIP_ENSKIP_INVALID  0x400   /* enskip_Kp* is invalid -> request */
#define SKIP_DESKIP_INVALID  0x800   /* deskip_Kp* is invalid -> request */

#define SKIP_SIGNALED        0x1000  /* Signal sent to daemon */


/* Directions for skip_process */
#define SKIP_OUTPUT 0
#define SKIP_INPUT  1

#define FILTER_NONE     0
#define FILTER_IP       1
#define FILTER_IPSP     2

#define FILTER_CHECK(filter,proto) \
  (filter && \
  (((filter == FILTER_IP) && (proto != IPPROTO_SKIP)) || \
   ((filter == FILTER_IPSP) && (proto == IPPROTO_SKIP))))

/* Flags for struct skipcache enskip_mod */
#define ENSKIPMODE_NONE   0x0
#define ENSKIPMODE_TCP    0x1
#define ENSKIPMODE_UDP    0x2
#define ENSKIPMODE_ICMP   0x4
#define ENSKIPMODE_OTHER  ~(ENSKIPMODE_TCP | ENSKIPMODE_UDP | ENSKIPMODE_ICMP)
#define ENSKIPMODE_ALL    ~0

/* Flags for struct skipcache enskip_mkid */
#define ENSKIPMKID_AUTO 0
#define ENSKIPMKID_YES  1
#define ENSKIPMKID_NO   2

/* Flags for struct skipcache deskip_mod */
#define DESKIPMODE_NONE   0x0
#define DESKIPMODE_DESKIP 0x1

#define HASHTBLSIZE     	  73      /* size of hash table */
#define SKIPCACHE_MAXENTRIES      64      /* max entries in hash table */
#define SKIPCACHE_MAXTTL          1024    /* maximum time to live in cache 
					   * (~1.5 h) */

#define QUEUETTL 60  /* max time to live for packets while waiting for key */
#define QUEUESIZE 50 /* max packets in queue while waiting for key */


/* constructor/ destructor */
extern int skipcache_init(int maxttl, int maxent);
extern int skipcache_exit(void);

/*
 * external mapping functions
 */
int skipcache_addmapping(struct skipcache *c, u_char *srcip, u_char *dstip);

/*
 * lookup a certain entry
 * you should release the entry afterwards to allow updates
 */
extern struct skipcache *skipcache_lookup(skip_id_t *id);
extern int skipcache_release(struct skipcache *c);
extern int skipcache_getrequests(u_char *buf, int len);

/*
 * get global occupation information for hash and cache
 */
extern int skipcache_listcache(u_char *buf, int len);


/*
 * cache manipulation functions
 */
extern int skipcache_update(struct skipcache *new);
extern int skipcache_remove(skip_id_t *id);
extern int skipcache_flush(void);
extern int skipcache_change(struct skipcache *c, skip_id_t *id);

extern int skipcache_compressentry(struct skipcache *c, u_char *buf, int len);
extern int skipcache_freeallqueues();

extern u_int32 skip_n;
