/*
 * Copyright (C) 1995 M. Hauber, Ch. Schneider, G. Caronni
 * See COPYING for more details
 */
#include "config.h"

#include <errno.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/netisr.h>
#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <sys/domain.h>
#include <sys/protosw.h>

#include "skip_defs.h"
#include "memblk.h"
#include "dynamic.h"
#include "skipcache.h"
#include "ipsp.h"
#include "skip.h"
#include "ipsum.h"
#include "interface.h"

#ifdef __GNUC__
#ident "$Id: interface.c,v 1.6 1996/04/25 14:57:22 hauber Exp $"
#else
static char rcsid[] = "$Id: interface.c,v 1.6 1996/04/25 14:57:22 hauber Exp $";
#endif

/* Avoid unnecessary softints for ipintrq, ipintr is looping anyway */
#define AVOID_SOFTINTS

/* XXX: Duplicated in queue.c, should be moved to some .h */
struct interface_fbparam
{
  struct ifnet *ifp;
  struct sockaddr_in address;
};

static int interface_output(struct ifnet *, struct mbuf *, struct sockaddr *);

static void(*oldinput)(struct mbuf *, struct ifnet *);

struct interface
{
  u_char IPaddr[IPADDRSIZE];
  struct ifnet *ifp;			/* interface identifier */

  /* function handles */
  int (*output)(struct ifnet *ifp, struct mbuf *m, struct sockaddr *sa);

  struct interface *next, *prev;
};

static struct interface *head = NULL;

static struct ifnet *interface_find(u_char *IPaddr)
{
  struct ifnet *ifp;

  for (ifp = ifnet; ifp; ifp = ifp->if_next)
  {
    struct ifaddr *ifa;

    if (ifp->if_output == interface_output)
      continue;  /* Only interfaces not already patched by SKIP */

    for (ifa = ifp->if_addrlist; ifa; ifa = ifa->ifa_next)
    {
      struct sockaddr_in *sin = (struct sockaddr_in *)&ifa->ifa_addr;

      if ((sin->sin_family == AF_INET) && 
          (MEMCMP(&sin->sin_addr, IPaddr, 4) == 0))
      {
        return ifp;
      }
    }
  }

  return NULL;
}

static struct interface *interface_search(u_char *IPaddr)
{
  struct interface *walk;

  for (walk = head; walk; walk = walk->next)
    if (MEMCMP(walk->IPaddr, IPaddr, 4) == 0)
      break;

  return walk;
}

static struct interface *interface_searchifp(struct ifnet *ifp)
{
  struct interface *walk;

  for (walk = head; walk; walk = walk->next)
    if (ifp == walk->ifp)
      break;

  return walk;
}

static struct mbuf *interface_pullup(struct mbuf *m, int size)
{
  struct mbuf *newm = m;

  if (m->m_len < size)  /* Need to pullup? */
  {
    if ((newm = m_vget(M_DONTWAIT, size, MT_DATA)))
    {
      u_char *dst = mtod(newm, u_char *);

      newm->m_len = 0;

      while (m && (size > 0))
      {
        int len = (m->m_len < size) ? m->m_len : size;

        MEMCPY(dst, mtod(m, u_char *), len);
        dst += len;
        newm->m_len += len;
        size -= len;

        m_adj(m, len);

        if (m->m_len == 0)	/* Else size == 0, don't advance m */
        {
          struct mbuf *next;

          MFREE(m, next);
          m = next;
        }
      }

      newm->m_next = m;
    }
  }

  return newm;
}

static int mbuf2memblk(struct mbuf *m, struct memblk *mb, struct memseg *ms)
{
  int count = 0;

  /* Construct memblk description */
  mb->dynamic = NULL;
  mb->dynamiclen = 0;
  mb->ms = ms;
  mb->offset = 0;
  mb->len = 0;


  while (m)  /* More than one mbuf? */
  {
    int ms_len = m->m_len;

    if (ms_len > 0)  /* Ignore empty mbufs */
    {
      if (count == (STATIC_MEMSEGS - 2))  /* Need dynamic mseg :-\ */
      {
        struct mbuf *t;
        int len;

        for (len = STATIC_MEMSEGS * sizeof(*ms), t = m; t; t = t->m_next)
          len += sizeof(struct memseg);

        mb->dynamiclen = len;
        if ((mb->dynamic = KALLOC(len)) == NULL)
          return -1;

        MEMCPY(mb->dynamic, mb->ms, count * sizeof(struct memseg));
        mb->ms = mb->dynamic;
        ms = mb->dynamic + count;
      }

      ms->ptr = mtod(m, u_char *);
      ms->len = ms_len;
      mb->len += ms_len;
      ms++;
      count++;
    }

    m = m->m_next;
  }

  mb->freems = ms;      /* First free memseg, ESP uses this internally */


  return 0;
}


static void interface_doinput(struct mbuf *m, struct ifnet *ifp)
{
  int result = SKIP_DISCARD;
  struct mbuf *newm = m;
  struct memblk oldmb, newmb;
  struct memseg oldms[STATIC_MEMSEGS], newms[2];

  /* Ensure IP header contiguous, hopefully mostly a no-op */
  if ((newm = interface_pullup(m, sizeof(struct ip))))
  {
    m = newm;

    if ((newm = interface_pullup(m, ipsp_getheadersize(mtod(m, void *)))))
    {
      m = newm;

      MEMZERO(&newmb, sizeof(newmb));
      newmb.ms = newms;
      if (mbuf2memblk(m, &oldmb, oldms) == 0)
      {
        struct interface_fbparam fb;

        fb.ifp = ifp;

        result = skip_process(SKIP_INPUT, &fb, NULL, (void **)&newm, &oldmb,
                              &newmb);
      }

      /* Note: new will never have dynamic memsegs on Irix */
      if (oldmb.dynamic)
        KFREE(oldmb.dynamic, oldmb.dynamiclen);
    }
  }

  switch (result)
  {
    case SKIP_DISCARD:
    {
      m_freem(m);
      if (newm != m)  /* Buffer already allocated before failure? */
        m_freem(newm);
      return;
    }

    case SKIP_PROCESSED:
    {
      /* Feed to IP again */
      struct ifqueue *ifq = &ipintrq;
      struct ifheader *ifh;
      int old;

      if (newm != m)  /* Newly allocated buffer used? */
      {
        m_freem(m);
        /* Adjust length of newly created packet */
        newm->m_off = (BLKSTART(&newmb) - sizeof(struct ifheader)) -
                      (u_char *)newm;
        newm->m_len = newmb.len + sizeof(struct ifheader);
      }
      else
      {
        /* Need to get space for ifheader if no new buffer allocated */
        if (M_HASCL(newm) ||
            (newm->m_off < MMINOFF + sizeof(struct ifheader)))
        {
          if ((m = m_get(M_DONTWAIT, MT_DATA)))
          {
            m->m_len = sizeof(struct ifheader);
            m->m_next = newm;
            newm = m;
          }
          else
          {
            m_freem(newm);
            return;
          }
        }
        else  /* We can put it into existing mbuf */
        {
          newm->m_off -= sizeof(struct ifheader);
          newm->m_len += sizeof(struct ifheader);
        }
      }

      ifh = mtod(newm, struct ifheader *);
      ifh->ifh_ifp = ifp;
      ifh->ifh_hdrlen = sizeof(struct ifheader);
           
      old = splimp();
      IFQ_LOCK(ifq);
      if (IF_QFULL(ifq)) 
      {
        m_freem(newm);
        IF_DROP(ifq);
      } 
      else
      {
        IF_ENQUEUE_NOLOCK(ifq, newm);
#ifndef AVOID_SOFTINTS
        schednetisr(NETISR_IP);
#endif
      }
      IFQ_UNLOCK(ifq);
      splx(old);
      return;
    }

    case SKIP_QUEUED:
    {
      if (newm != m)  /* Buffer already allocated before failure? */
        m_freem(newm);
      return;
    }
  }
}

static void interface_input(struct mbuf *m, struct ifnet *ifp)
{
  struct ip *ip = mtod(m, struct ip *);

  if (ip->ip_p == IPPROTO_SKIP)
  {
    /* Undo changes ip_input did to our packet :-\ */
    ip->ip_len += ip->ip_hl * 4;
    /* XXX: Not sure if the following code went from BSD4.3 into IRIX */
    /* My machine is big endian, so I can't test it */
    ip->ip_len = SKIP_HTONS(ip->ip_len);
    ip->ip_id = SKIP_HTONS(ip->ip_id);
    ip->ip_off = SKIP_HTONS(ip->ip_off);

    interface_doinput(m, ifp);
  }
  else
    oldinput(m, ifp);  /* Not IPPROTO_SKIP, let old input handler do it */
}

static int interface_output(struct ifnet *ifp, struct mbuf *m,
                            struct sockaddr *address)
{
  struct interface *i = interface_searchifp(ifp);
  struct sockaddr_in *sin = (struct sockaddr_in *)address;

  if (i == NULL)
  {
    m_freem(m);
    return -ENOENT;
  }

  if (sin->sin_family == AF_INET)
  {
    int result = SKIP_DISCARD;
    struct mbuf *newm = m;
    struct memblk oldmb, newmb;
    struct memseg oldms[STATIC_MEMSEGS], newms[2];

    /* Ensure IP header contiguous, hopefully mostly a no-op */
    if ((newm = interface_pullup(m, sizeof(struct ip))))
    {
      m = newm;

      if ((newm = interface_pullup(m, ipsp_getheadersize(mtod(m, void *)))))
      {
        m = newm;

        MEMZERO(&newmb, sizeof(newmb));
        newmb.ms = newms;
        if (mbuf2memblk(m, &oldmb, oldms) == 0)
        {
          struct interface_fbparam fb;

          fb.address = *sin;
          fb.ifp = ifp;

          result = skip_process(SKIP_OUTPUT, &fb, NULL, (void **)&newm, &oldmb,
                                &newmb);
        }

        /* Note: new will never have dynamic memsegs on Irix */
        if (oldmb.dynamic)
          KFREE(oldmb.dynamic, oldmb.dynamiclen);
      }
    }

    switch (result)
    {
      case SKIP_DISCARD:
      {
        m_freem(m);
        if (newm != m)  /* Buffer already allocate before failure? */
          m_freem(newm);
        return 0;
      }

      case SKIP_PROCESSED:
      {
        if (newm != m)
        {
          m_freem(m);

          /* Adjust length of newly created packet */
          newm->m_off = BLKSTART(&newmb) - (u_char *)newm;
          newm->m_len = newmb.len;
        }

        return i->output(ifp, newm, address);
      }

      case SKIP_QUEUED:
      {
        if (newm != m)  /* Buffer already allocate before failure? */
          m_freem(newm);
        return 0;
      }
    }  
  }
  else
    return i->output(ifp, m, address);
}

int interface_outfeedback(struct mbuf *m)
{
  struct mbuf *pkt;
  struct interface_fbparam *fb = mtod(m, struct interface_fbparam *);
  struct ifnet *ifp = fb->ifp;
  struct sockaddr_in sin = fb->address;

  MFREE(m, pkt);

  return interface_output(ifp, pkt, (struct sockaddr *)&sin);
}

int interface_infeedback(struct mbuf *m)
{
  struct mbuf *pkt;
  struct interface_fbparam *fb = mtod(m, struct interface_fbparam *);
  struct ifnet *ifp = fb->ifp;

  MFREE(m, pkt);

  interface_doinput(pkt, ifp);
#ifdef AVOID_SOFTINTS
  schednetisr(NETISR_IP);
#endif

  return 0;
}

int interface_getbuf(struct memblk *m, void **mp, int size)
{
  int result = -ENOMEM;
  struct mbuf *new;

  /* Get one large buffer */
  if ((new = m_vget(M_DONTWAIT, size + sizeof(struct ifheader), MT_DATA)))
  {
    m->len = size;
    m->offset = sizeof(struct ifheader);
    m->ms->len = size + sizeof(struct ifheader);
    m->ms->ptr = mtod(new, u_char *) + sizeof(struct ifheader);

    *mp = new;  /* Store newly allocated mbuf */

    result = 0;
  }

  return result;
}

int interface_attach(void *interface, u_char *IPaddr)
{
  struct ifnet *ifn;
  struct interface *new;
  int c;

  if (interface_search(IPaddr))
    return -EEXIST;

  for (c = 0; (ifn = interface_find(IPaddr)); c++)
  {
    if ((new = KALLOC(sizeof(*new))) == NULL)
      return -ENOMEM;

    MEMZERO(new, sizeof(*new));
    MEMCPY(new->IPaddr, IPaddr, 4);
    new->ifp = (struct ifnet *)ifn;
    new->output = ifn->if_output;

    ifn->if_output = interface_output;
    ifn->if_mtu -= ipsp_maxheadergrowth();

    new->next = head;
    new->prev = NULL;
    if (head)
      head->prev = new;
    head = new;
  }

  return c ? 0 : -ENOENT;
}

static int interface_remove(struct interface *i)
{
  struct ifnet *ifn;

  ifn = (struct ifnet *)i->ifp;

  ifn->if_output = i->output;
  ifn->if_mtu += ipsp_maxheadergrowth();

  if (i->next)
    i->next->prev = i->prev;
  if (i->prev)
    i->prev->next = i->next;
  else
    head = i->next;

  KFREE(i, sizeof(*i));
  return 0;
}

int interface_detach(void *interface, u_char *IPaddr)
{
  struct interface *i;
  int c;

  for (c = 0; (i = interface_search(IPaddr)); c++)
    interface_remove(i);

  return c ? 0 : -ENOENT;
}

int interface_init(void)
{
  int result = ENFILE;
  struct protosw *pr;

  if ((pr = pffindproto(AF_INET, IPPROTO_RAW, SOCK_RAW)))
  {
    oldinput = pr->pr_input;
    pr->pr_input = interface_input;
    result = 0;
  }

  head = NULL;

  return result;
}

int interface_exit(void)
{
  struct interface *walk, *next;
  struct protosw *pr;

  for (walk = head; walk; walk = next)
  {
    next = walk->next;
    interface_remove(walk);
  }

  if ((pr = pffindproto(AF_INET, IPPROTO_RAW, SOCK_RAW)))
  {
    if (pr->pr_input == interface_input)
      pr->pr_input = oldinput;
  }

  head = NULL;

  return 0;
}
