/*
 * Copyright (C) 1995,1997 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */
#include "config.h"

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#include "skip_defs.h"
#include "memblk.h"
#include "dynamic.h"
#include "skipcache.h"
#include "ipsp.h"
#include "sign.h"
#include "ah.h"

#ifdef __GNUC__
#ident "$Id: ah.c,v 1.3 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: ah.c,v 1.3 1996/04/25 15:05:30 cschneid Exp $";
#endif

struct ah_hdr
{
  u_char nextproto, len, reserved[2];
  u_int32 skip_spi;
};

int ah_maxheadergrowth(void)
{
  return sizeof(struct ah_hdr) + sign_icvlen(SIGNALG_MAX);
}

int ah_prepare(struct skipcache *c, struct memblk *buf, int nextproto)
{
  struct ah_hdr *hdr = (struct ah_hdr *)BLKSTART(buf);
  int maclen = sign_icvlen(c->enskip_MAC_alg);

  MEMZERO(hdr, sizeof(*hdr) + maclen);
  BLKINC(buf, sizeof(*hdr) + maclen);
  hdr->nextproto = (u_char)nextproto;
  hdr->len = maclen / sizeof(u_int32);    /* Length in 32 bit words */
  hdr->skip_spi = SKIP_HTONL(SKIP_SPI);

  return 0;
}

/*
 * Data points to an IP packet including IP hdr
 * No options handling is done since we produce packets without any options
 */
int ah_calc(struct skipcache *c, struct memblk *hdrdata, struct memblk *data)
{
  int result;
  struct ah_hdr *hdr = (struct ah_hdr *)BLKSTART(hdrdata);
  u_char *buf = (u_char *)(hdr + 1);
#ifndef SMALL_KERNEL_STACK
  struct memblk ah_data = *data;
#else /* SMALL_KERNEL_STACK */
  struct memblk *ah_data;
#endif /* SMALL_KERNEL_STACK */
  struct ip *ip = (struct ip *)BLKSTART(data);
  u_char ttl, tos;
  u_short sum, off;

#ifdef SMALL_KERNEL_STACK
  if ((ah_data = KALLOC(sizeof(*ah_data))) == NULL)
    return -1;
  MEMCPY(ah_data, data, sizeof(*ah_data));
#endif /* SMALL_KERNEL_STACK */

  /* Save fields of IP header possibly modified during transit, null them */
  ttl = ip->ip_ttl;
  sum = ip->ip_sum;
  tos = ip->ip_tos;
  off = ip->ip_off;

  ip->ip_ttl = 0;
  ip->ip_sum = 0;

  ip->ip_tos = 0;
  ip->ip_off = 0;

  if ((result = sign_sign(c->enskip_MAC_alg, c->data + c->enskip_A_Kp.offset,
#ifndef SMALL_KERNEL_STACK
		     c->enskip_A_Kp.len, &ah_data, buf)) < 0)
#else /* SMALL_KERNEL_STACK */
		     c->enskip_A_Kp.len, ah_data, buf)) < 0)
#endif /* SMALL_KERNEL_STACK */
  {
    ipsp_stat.badMACalg++;
  }


  /* Restore saved fields */
  ip->ip_ttl = ttl;
  ip->ip_sum = sum;

  ip->ip_tos = tos;
  ip->ip_off = off;

#ifdef SMALL_KERNEL_STACK
  KFREE(ah_data, sizeof(*ah_data));
#endif /* SMALL_KERNEL_STACK */

  return result;
}

/*
 * Data points to an IP packet including IP hdr
 * XXX: No options handling is done. I.e. if options were added during 
 * transit authentication will fail...
 */
int ah_check(struct skipcache *c, int alg, struct memblk *hdrdata, 
	     struct memblk *data)
{
  int result = -1;
  struct ah_hdr *hdr = (struct ah_hdr *)BLKSTART(hdrdata);
  int maclen = sign_icvlen(alg);
  u_char *mac = (u_char *)(hdr + 1);
#ifndef SMALL_KERNEL_STACK
  u_char buf[64];
  struct memblk ah_data = *data;
#endif /* SMALL_KERNEL_STACK */
  struct ip *ip = (struct ip *)BLKSTART(data);
  u_char ttl, tos;
  u_short sum, off;

  if (data->len < sizeof(*hdr) + maclen)
  {
    ipsp_stat.badpktlen++;
    return -1;
  }

  if (hdr->skip_spi != SKIP_HTONL(SKIP_SPI))
  {
    ipsp_stat.badMACalg++;  /* Should have own error code */
    return -1;
  }

  if ((maclen > 0) && (maclen == hdr->len * sizeof(u_int32)))
  {
#ifdef SMALL_KERNEL_STACK
    struct memblk *ah_data;
    u_char *buf;
#endif /* SMALL_KERNEL_STACK */

#if 1  /* XXX Print warning if options present in ip packet */
    if (ip->ip_hl * 4 != sizeof(struct ip))
    {
      printf(">>ah_check: received ip packet with options, dropping!\n");
      return -1;
    }
#endif

    /* Save old MAC, null it to calculate new MAC */
#ifdef SMALL_KERNEL_STACK
    if ((buf = KALLOC(64)) == NULL)
      return -1;
#endif /* SMALL_KERNEL_STACK */

    MEMCPY(buf, mac, maclen);
    MEMZERO(mac, maclen);

#ifdef SMALL_KERNEL_STACK
    if ((ah_data = KALLOC(sizeof(*ah_data))) == NULL) {
      KFREE(buf, 64);
      return -1;
    }
    MEMCPY(ah_data, data, sizeof(*ah_data));
#endif /* SMALL_KERNEL_STACK */

    /* Save fields of IP header possibly modified during transit, null them */
    ttl = ip->ip_ttl;
    sum = ip->ip_sum;

    tos = ip->ip_tos;
    off = ip->ip_off;

    ip->ip_ttl = 0;
    ip->ip_sum = 0;

    ip->ip_tos = 0;
    ip->ip_off = 0;
    
    if (sign_sign(alg, c->data + c->deskip_A_Kp.offset + 
		  c->deskip_A_Kp.len - SIGN_KEYLEN,  /* low order bytes */
#ifndef SMALL_KERNEL_STACK
		  SIGN_KEYLEN, &ah_data, mac) == 0)
#else /* SMALL_KERNEL_STACK */
		  SIGN_KEYLEN, ah_data, mac) == 0)
#endif /* SMALL_KERNEL_STACK */
    {
      /* Restore saved fields */
      ip->ip_ttl = ttl;
      ip->ip_sum = sum;

      ip->ip_tos = tos;
      ip->ip_off = off;
      
      if (MEMCMP(mac, buf, maclen))  /* Different MAC? */
      {
	MEMCPY(mac, buf, maclen);  /* Copy back old MAC, probably no need to */
	ipsp_stat.badMAC++;
      }
      else
      {
	result = hdr->nextproto;
	BLKINC(hdrdata, sizeof(*hdr) + maclen);  /* AH header done */
      }
    }
#ifdef SMALL_KERNEL_STACK
    KFREE(ah_data, sizeof(*ah_data));
    KFREE(buf, 64);
#endif /* SMALL_KERNEL_STACK */
  }
  else
    ipsp_stat.badMACalg++;
  
  return result;
}
