/*
 * SAFER SK-128 for ENSKIP, Robert Muchsel 11/1996, 01/1997
 * Modifications: removed other SAFER ciphers, static tables
 *
 * These modifications were copied from the tables in the 
 * "SKIP Reference Implementation Beta 2.2" BOOK by Sun Microsystems, Inc.,
 * chapter 18, pp. 18-1 to 18-11
 */

/****************************************************************************
 * SAFER implementation by Richard De Moliner                               *
 ****************************************************************************/

/****************************************************************************/
/*                                                                          */
/*             C R Y P T O G R A P H I C - A L G O R I T H M S              */
/*                                                                          */
/****************************************************************************/
/* Author:       Richard De Moliner (demoliner@isi.ethz.ch)                 */
/*               Signal and Information Processing Laboratory               */
/*               Swiss Federal Institute of Technology                      */
/*               CH-8092 Zuerich, Switzerland                               */
/*                                                                          */
/* Description:  block-cipher algorithm SAFER (Secure And Fast Encryption   */
/*               Routine) in its four versions: SAFER K-64, SAFER K-128,    */
/*               SAFER SK-64 and SAFER SK-128.                              */
/****************************************************************************/

#include "config.h"
#include "memblk.h"
#include "crypt.h"

#define SAFER_K64_DEFAULT_NOF_ROUNDS     6
#define SAFER_K128_DEFAULT_NOF_ROUNDS   10
#define SAFER_SK64_DEFAULT_NOF_ROUNDS    8
#define SAFER_SK128_DEFAULT_NOF_ROUNDS  10
#define SAFER_MAX_NOF_ROUNDS            13
#define SAFER_BLOCK_LEN                  8
#define SAFER_KEY_LEN     (1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS))

typedef unsigned char safer_block_t[SAFER_BLOCK_LEN];
typedef unsigned char safer_key_t[SAFER_KEY_LEN];


#define ROL(x, n)    ((unsigned char)((unsigned int)(x) << (n)\
                     |(unsigned int)((x) & 0xFF) >> (8 - (n))))
#define EXP(x)       exp_tab[(x) & 0xFF]
#define LOG(x)       log_tab[(x) & 0xFF]
#define PHT(x, y)    { y += x; x += y; }
#define IPHT(x, y)   { x -= y; y -= x; }


#define TABLESIZE 256

static
unsigned char exp_tab[TABLESIZE]={	
	0x01, 0x2d, 0xe2, 0x93, 0xbe, 0x45, 0x15, 0xae, 
	0x78, 0x03, 0x87, 0xa4, 0xb8, 0x38, 0xcf, 0x3f, 
	0x08, 0x67, 0x09, 0x94, 0xeb, 0x26, 0xa8, 0x6b, 
	0xbd, 0x18, 0x34, 0x1b, 0xbb, 0xbf, 0x72, 0xf7, 
	0x40, 0x35, 0x48, 0x9c, 0x51, 0x2f, 0x3b, 0x55, 
	0xe3, 0xc0, 0x9f, 0xd8, 0xd3, 0xf3, 0x8d, 0xb1, 
	0xff, 0xa7, 0x3e, 0xdc, 0x86, 0x77, 0xd7, 0xa6, 
	0x11, 0xfb, 0xf4, 0xba, 0x92, 0x91, 0x64, 0x83, 
	0xf1, 0x33, 0xef, 0xda, 0x2c, 0xb5, 0xb2, 0x2b, 
	0x88, 0xd1, 0x99, 0xcb, 0x8c, 0x84, 0x1d, 0x14, 
	0x81, 0x97, 0x71, 0xca, 0x5f, 0xa3, 0x8b, 0x57, 
	0x3c, 0x82, 0xc4, 0x52, 0x5c, 0x1c, 0xe8, 0xa0, 
	0x04, 0xb4, 0x85, 0x4a, 0xf6, 0x13, 0x54, 0xb6, 
	0xdf, 0x0c, 0x1a, 0x8e, 0xde, 0xe0, 0x39, 0xfc, 
	0x20, 0x9b, 0x24, 0x4e, 0xa9, 0x98, 0x9e, 0xab, 
	0xf2, 0x60, 0xd0, 0x6c, 0xea, 0xfa, 0xc7, 0xd9, 
	0x00, 0xd4, 0x1f, 0x6e, 0x43, 0xbc, 0xec, 0x53, 
	0x89, 0xfe, 0x7a, 0x5d, 0x49, 0xc9, 0x32, 0xc2, 
	0xf9, 0x9a, 0xf8, 0x6d, 0x16, 0xdb, 0x59, 0x96, 
	0x44, 0xe9, 0xcd, 0xe6, 0x46, 0x42, 0x8f, 0x0a, 
	0xc1, 0xcc, 0xb9, 0x65, 0xb0, 0xd2, 0xc6, 0xac, 
	0x1e, 0x41, 0x62, 0x29, 0x2e, 0x0e, 0x74, 0x50, 
	0x02, 0x5a, 0xc3, 0x25, 0x7b, 0x8a, 0x2a, 0x5b, 
	0xf0, 0x06, 0x0d, 0x47, 0x6f, 0x70, 0x9d, 0x7e, 
	0x10, 0xce, 0x12, 0x27, 0xd5, 0x4c, 0x4f, 0xd6, 
	0x79, 0x30, 0x68, 0x36, 0x75, 0x7d, 0xe4, 0xed, 
	0x80, 0x6a, 0x90, 0x37, 0xa2, 0x5e, 0x76, 0xaa, 
	0xc5, 0x7f, 0x3d, 0xaf, 0xa5, 0xe5, 0x19, 0x61, 
	0xfd, 0x4d, 0x7c, 0xb7, 0x0b, 0xee, 0xad, 0x4b, 
	0x22, 0xf5, 0xe7, 0x73, 0x23, 0x21, 0xc8, 0x05, 
	0xe1, 0x66, 0xdd, 0xb3, 0x58, 0x69, 0x63, 0x56, 
	0x0f, 0xa1, 0x31, 0x95, 0x17, 0x07, 0x3a, 0x28
};

static
unsigned char log_tab[TABLESIZE]={	
	0x80, 0x00, 0xb0, 0x09, 0x60, 0xef, 0xb9, 0xfd, 
	0x10, 0x12, 0x9f, 0xe4, 0x69, 0xba, 0xad, 0xf8, 
	0xc0, 0x38, 0xc2, 0x65, 0x4f, 0x06, 0x94, 0xfc, 
	0x19, 0xde, 0x6a, 0x1b, 0x5d, 0x4e, 0xa8, 0x82, 
	0x70, 0xed, 0xe8, 0xec, 0x72, 0xb3, 0x15, 0xc3, 
	0xff, 0xab, 0xb6, 0x47, 0x44, 0x01, 0xac, 0x25, 
	0xc9, 0xfa, 0x8e, 0x41, 0x1a, 0x21, 0xcb, 0xd3, 
	0x0d, 0x6e, 0xfe, 0x26, 0x58, 0xda, 0x32, 0x0f, 
	0x20, 0xa9, 0x9d, 0x84, 0x98, 0x05, 0x9c, 0xbb, 
	0x22, 0x8c, 0x63, 0xe7, 0xc5, 0xe1, 0x73, 0xc6, 
	0xaf, 0x24, 0x5b, 0x87, 0x66, 0x27, 0xf7, 0x57, 
	0xf4, 0x96, 0xb1, 0xb7, 0x5c, 0x8b, 0xd5, 0x54, 
	0x79, 0xdf, 0xaa, 0xf6, 0x3e, 0xa3, 0xf1, 0x11, 
	0xca, 0xf5, 0xd1, 0x17, 0x7b, 0x93, 0x83, 0xbc, 
	0xbd, 0x52, 0x1e, 0xeb, 0xae, 0xcc, 0xd6, 0x35, 
	0x08, 0xc8, 0x8a, 0xb4, 0xe2, 0xcd, 0xbf, 0xd9, 
	0xd0, 0x50, 0x59, 0x3f, 0x4d, 0x62, 0x34, 0x0a, 
	0x48, 0x88, 0xb5, 0x56, 0x4c, 0x2e, 0x6b, 0x9e, 
	0xd2, 0x3d, 0x3c, 0x03, 0x13, 0xfb, 0x97, 0x51, 
	0x75, 0x4a, 0x91, 0x71, 0x23, 0xbe, 0x76, 0x2a, 
	0x5f, 0xf9, 0xd4, 0x55, 0x0b, 0xdc, 0x37, 0x31, 
	0x16, 0x74, 0xd7, 0x77, 0xa7, 0xe6, 0x07, 0xdb, 
	0xa4, 0x2f, 0x46, 0xf3, 0x61, 0x45, 0x67, 0xe3, 
	0x0c, 0xa2, 0x3b, 0x1c, 0x85, 0x18, 0x04, 0x1d, 
	0x29, 0xa0, 0x8f, 0xb2, 0x5a, 0xd8, 0xa6, 0x7e, 
	0xee, 0x8d, 0x53, 0x4b, 0xa1, 0x9a, 0xc1, 0x0e, 
	0x7a, 0x49, 0xa5, 0x2c, 0x81, 0xc4, 0xc7, 0x36, 
	0x2b, 0x7f, 0x43, 0x95, 0x33, 0xf2, 0x6c, 0x68, 
	0x6d, 0xf0, 0x02, 0x28, 0xce, 0xdd, 0x9b, 0xea, 
	0x5e, 0x99, 0x7c, 0x14, 0x86, 0xcf, 0xe5, 0x42, 
	0xb8, 0x40, 0x78, 0x2d, 0x3a, 0xe9, 0x64, 0x1f, 
	0x92, 0x90, 0x7d, 0x39, 0x6f, 0xe0, 0x89, 0x30
};


/*******************************************************************************
* void Safer_Expand_Userkey(safer_block_t userkey_1,
*                           safer_block_t userkey_2,
*                           unsigned int nof_rounds,
*                           int strengthened,
*                           safer_key_t key)
*
*   expands a user-selected key of length 64 bits or 128 bits to a encryption /
*   decryption key. If your user-selected key is of length 64 bits, then give
*   this key to both arguments 'userkey_1' and 'userkey_2', e.g.
*   'Safer_Expand_Userkey(z, z, key)'. Note: SAFER K-64 and SAFER SK-64 with a
*   user-selected key 'z' of length 64 bits are identical to SAFER K-128 and
*   SAFER SK-128 with a user-selected key 'z z' of length 128 bits,
*   respectively.
*   pre:  'userkey_1'  contains the first 64 bits of user key.
*         'userkey_2'  contains the second 64 bits of user key.
*         'nof_rounds' contains the number of encryption rounds
*                      'nof_rounds' <= 'SAFER_MAX_NOF_ROUNDS'
*         'strengthened' is non-zero if the strengthened key schedule should be
*                      used and zero if the original key schedule should be
*                      used.
*   post: 'key'        contains the expanded key.
*
*******************************************************************************/
static
void Safer_Expand_Userkey(    safer_block_t userkey_1,
                              safer_block_t userkey_2,
                              unsigned int nof_rounds,
                              int strengthened,
                              safer_key_t key)
{   unsigned int i, j;
#ifndef SMALL_KERNEL_STACK
    unsigned char ka[SAFER_BLOCK_LEN + 1];
    unsigned char kb[SAFER_BLOCK_LEN + 1];
#else /* SMALL_KERNEL_STACK */
    unsigned char *ka;
    unsigned char *kb;

    if ((ka = KALLOC(SAFER_BLOCK_LEN + 1)) == NULL)
      return;
    if ((kb = KALLOC(SAFER_BLOCK_LEN + 1)) == NULL) {
      KFREE(ka, SAFER_BLOCK_LEN + 1);
      return;
    }
#endif /* SMALL_KERNEL_STACK */

    if (SAFER_MAX_NOF_ROUNDS < nof_rounds)
        nof_rounds = SAFER_MAX_NOF_ROUNDS;
    *key++ = (unsigned char)nof_rounds;
    ka[SAFER_BLOCK_LEN] = 0;
    kb[SAFER_BLOCK_LEN] = 0;
    for (j = 0; j < SAFER_BLOCK_LEN; j++)
    {
        ka[SAFER_BLOCK_LEN] ^= ka[j] = ROL(userkey_1[j], 5);
        kb[SAFER_BLOCK_LEN] ^= kb[j] = *key++ = userkey_2[j];
    }
    for (i = 1; i <= nof_rounds; i++)
    {
        for (j = 0; j < SAFER_BLOCK_LEN + 1; j++)
        {
            ka[j] = ROL(ka[j], 6);
            kb[j] = ROL(kb[j], 6);
        }
        for (j = 0; j < SAFER_BLOCK_LEN; j++)
            if (strengthened)
                *key++ = (ka[(j + 2 * i - 1) % (SAFER_BLOCK_LEN + 1)]
                                + exp_tab[exp_tab[18 * i + j + 1]]) & 0xFF;
            else
                *key++ = (ka[j] + exp_tab[exp_tab[18 * i + j + 1]]) & 0xFF;
        for (j = 0; j < SAFER_BLOCK_LEN; j++)
            if (strengthened)
                *key++ = (kb[(j + 2 * i) % (SAFER_BLOCK_LEN + 1)]
                                + exp_tab[exp_tab[18 * i + j + 10]]) & 0xFF;
            else
                *key++ = (kb[j] + exp_tab[exp_tab[18 * i + j + 10]]) & 0xFF;
    }
    for (j = 0; j < SAFER_BLOCK_LEN + 1; j++)
        ka[j] = kb[j] = 0;

#ifdef SMALL_KERNEL_STACK
    KFREE(kb, SAFER_BLOCK_LEN + 1);
    KFREE(ka, SAFER_BLOCK_LEN + 1);
#endif /* SMALL_KERNEL_STACK */
} /* Safer_Expand_Userkey */


/*******************************************************************************
* void Safer_Encrypt_Block(safer_block_t block_in, safer_key_t key, 
*                          safer_block_t block_out)
*
*   encryption algorithm.
*   pre:  'block_in'  contains the plain-text block.
*         'key'       contains the expanded key.
*   post: 'block_out' contains the cipher-text block.
*
*******************************************************************************/
static
void Safer_Encrypt_Block(safer_block_t block_in, safer_key_t key, 
		    safer_block_t block_out )
{   
    unsigned char a, b, c, d, e, f, g, h, t;
    unsigned int round;

    a = block_in[0]; b = block_in[1]; c = block_in[2]; d = block_in[3];
    e = block_in[4]; f = block_in[5]; g = block_in[6]; h = block_in[7];
    if (SAFER_MAX_NOF_ROUNDS < (round = *key)) round = SAFER_MAX_NOF_ROUNDS;
    while(round--)
    {
        a ^= *++key; b += *++key; c += *++key; d ^= *++key;
        e ^= *++key; f += *++key; g += *++key; h ^= *++key;
        a = EXP(a) + *++key; b = LOG(b) ^ *++key;
        c = LOG(c) ^ *++key; d = EXP(d) + *++key;
        e = EXP(e) + *++key; f = LOG(f) ^ *++key;
        g = LOG(g) ^ *++key; h = EXP(h) + *++key;
        PHT(a, b); PHT(c, d); PHT(e, f); PHT(g, h);
        PHT(a, c); PHT(e, g); PHT(b, d); PHT(f, h);
        PHT(a, e); PHT(b, f); PHT(c, g); PHT(d, h);
        t = b; b = e; e = c; c = t; t = d; d = f; f = g; g = t;
    }
    a ^= *++key; b += *++key; c += *++key; d ^= *++key;
    e ^= *++key; f += *++key; g += *++key; h ^= *++key;
    block_out[0] = a & 0xFF; block_out[1] = b & 0xFF;
    block_out[2] = c & 0xFF; block_out[3] = d & 0xFF;
    block_out[4] = e & 0xFF; block_out[5] = f & 0xFF;
    block_out[6] = g & 0xFF; block_out[7] = h & 0xFF;
} /* Safer_Encrypt_Block */


/*******************************************************************************
* void Safer_Decrypt_Block(safer_block_t block_in, safer_key_t key,
*                          safer_block_t block_out)
*
*   decryption algorithm.
*   pre:  'block_in'  contains the cipher-text block.
*         'key'       contains the expanded key.
*   post: 'block_out' contains the plain-text block.
*
*******************************************************************************/
static
void Safer_Decrypt_Block(safer_block_t block_in, safer_key_t key, 
		         safer_block_t block_out)
{   unsigned char a, b, c, d, e, f, g, h, t;
    unsigned int round;

    a = block_in[0]; b = block_in[1]; c = block_in[2]; d = block_in[3];
    e = block_in[4]; f = block_in[5]; g = block_in[6]; h = block_in[7];
    if (SAFER_MAX_NOF_ROUNDS < (round = *key)) round = SAFER_MAX_NOF_ROUNDS;
    key += SAFER_BLOCK_LEN * (1 + 2 * round);
    h ^= *key; g -= *--key; f -= *--key; e ^= *--key;
    d ^= *--key; c -= *--key; b -= *--key; a ^= *--key;
    while (round--)
    {
        t = e; e = b; b = c; c = t; t = f; f = d; d = g; g = t;
        IPHT(a, e); IPHT(b, f); IPHT(c, g); IPHT(d, h);
        IPHT(a, c); IPHT(e, g); IPHT(b, d); IPHT(f, h);
        IPHT(a, b); IPHT(c, d); IPHT(e, f); IPHT(g, h);
        h -= *--key; g ^= *--key; f ^= *--key; e -= *--key;
        d -= *--key; c ^= *--key; b ^= *--key; a -= *--key;
        h = LOG(h) ^ *--key; g = EXP(g) - *--key;
        f = EXP(f) - *--key; e = LOG(e) ^ *--key;
        d = LOG(d) ^ *--key; c = EXP(c) - *--key;
        b = EXP(b) - *--key; a = LOG(a) ^ *--key;
    }
    block_out[0] = a & 0xFF; block_out[1] = b & 0xFF;
    block_out[2] = c & 0xFF; block_out[3] = d & 0xFF;
    block_out[4] = e & 0xFF; block_out[5] = f & 0xFF;
    block_out[6] = g & 0xFF; block_out[7] = h & 0xFF;
} /* Safer_Decrypt_Block */

/*************************************************************************
 * interface for SKIP                                                    *
 *************************************************************************/

#define KEYLEN_128	16	

struct statesk128
{
  int alg;
  safer_key_t key;
};

static int cryptsk128_init(void)
{
  return 0;
}

static int cryptsk128_exit(void)
{
  return 0;
}

static int cryptsk128_statelen(void)
{
  return sizeof(struct statesk128);
}

static int cryptsk128_blocklen(void)
{
  return sizeof(safer_block_t);
}

static int cryptsk128_keylen(void)
{
  return KEYLEN_128;
}

static int cryptsk128_MIlen(void)
{
  return sizeof(safer_block_t);
}

static int cryptsk128_setkey(u_char *key, u_char *stateptr)
{
  struct statesk128 *state = (struct statesk128 *)stateptr;

  Safer_Expand_Userkey(key, key+8, SAFER_K128_DEFAULT_NOF_ROUNDS, 1, state->key);
  return 0;
}

static int cryptsk128_crypt(u_char *in, u_char *out, int len, u_char *stateptr)
{
  struct statesk128 *state = (struct statesk128 *)stateptr;

  Safer_Encrypt_Block(in, state->key, out);
  return 0;
}

static int cryptsk128_decrypt(u_char *in, u_char *out, int len, u_char *stateptr)
{
  struct statesk128 *state = (struct statesk128 *)stateptr;

  Safer_Decrypt_Block(in, state->key, out);
  return 0;
}

struct crypt_algorithm cryptsafer_sk128 =
{
  cryptsk128_init, cryptsk128_exit,
  cryptsk128_statelen, cryptsk128_statelen, 
  cryptsk128_blocklen, cryptsk128_keylen, cryptsk128_MIlen,
  cryptsk128_setkey, cryptsk128_setkey,
  cryptsk128_crypt, cryptsk128_decrypt,
  NULL
};

