/*
 * Copyright (C) 1995,1997 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */
#include "config.h"
#include "skip_defs.h"
#include "memblk.h"
#include "dynamic.h"
#include "skipcache.h"
#include "ipsp.h"
#include "crypt.h"
#include "random.h"
#include "esp.h"

#ifdef __GNUC__
#ident "$Id: esp.c,v 1.3 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: esp.c,v 1.3 1996/04/25 15:05:30 cschneid Exp $";
#endif

struct esp_hdr
{
  u_int32 skip_spi;
};

int esp_maxheadergrowth(void)
{
  return sizeof(struct esp_hdr) + crypt_MIlen(CRYPTALG_MAX) +
    crypt_blocklen(CRYPTALG_MAX) - 1 + 2;
}

int esp_encrypt(struct skipcache *c, struct memblk *old, struct memblk *new,
                int nextproto)
{
  int result;
  struct esp_hdr *hdr = (struct esp_hdr *)BLKSTART(new);
  u_char *IV;
  int blocklen = crypt_blocklen(c->enskip_Crypt_alg);
  int keylen = crypt_keylen(c->enskip_Crypt_alg);
#ifndef SMALL_KERNEL_STACK
  u_char padding[MAXBLOCKLEN * 2];
  u_char *pad = padding;
#else /* SMALL_KERNEL_STACK */
  u_char *padding, *pad;
#endif /* SMALL_KERNEL_STACK */

  if ((blocklen <= 0) || (keylen <= 0))
  {
    ipsp_stat.badKpalg++;
    return -1;
  }

#ifdef SMALL_KERNEL_STACK
  if ((pad = padding = KALLOC(MAXBLOCKLEN * 2)) == NULL)
    return -1;
#endif /* SMALL_KERNEL_STACK */

  hdr->skip_spi = SKIP_HTONL(SKIP_SPI);

  BLKINC(new, sizeof(*hdr));
  IV = BLKSTART(new);
  BLKINC(new, c->enskip_MI.len);

  if (blocklen > 1)  /* Block cipher? */
  {
    int padlen;
    padlen = old->len % blocklen;

    if (padlen < blocklen - 2)
      padlen = blocklen - 2 - padlen;
    else if (padlen  > blocklen - 2)
      padlen = blocklen * 2 - 2 - padlen;
    else
      padlen = 0;

    random_random(padding, padlen);

    pad += padlen;
    *pad++ = padlen;
  }

  /* Our transforms always contain a next proto after the actual data */
  *pad++ = nextproto;

  /* Note: Append padding to memblk describing input data */
  old->freems->ptr = padding;
  old->freems->len = pad - padding;
  old->len += pad - padding;

  result = crypt_encrypt(c->enskip_Crypt_alg, c->data + c->encryptstate.offset,
			 c->data + c->enskip_E_Kp.offset + c->enskip_E_Kp.len -
			 keylen, IV, c->data + c->enskip_MI.offset, old, new);

  /* Remove padding again */
  old->len -= old->freems->len;

#ifdef SMALL_KERNEL_STACK
  KFREE(padding, MAXBLOCKLEN * 2);
#endif /* SMALL_KERNEL_STACK */

  return result;
}

int esp_decrypt(struct skipcache *c, int alg, struct memblk *old, 
		struct memblk *new)
{
  int result = -1;
  struct esp_hdr *hdr = (struct esp_hdr *)BLKSTART(old);
  int ivlen = crypt_MIlen(alg);
  int blocklen = crypt_blocklen(alg);
  int keylen = crypt_keylen(alg);

  if ((ivlen > 0) && (blocklen > 0) && (keylen > 0))
  {
    u_char *IV = (u_char *)(hdr + 1);

    if (old->len < sizeof(*hdr) + ivlen)
    {
      ipsp_stat.badpktlen++;
      return -1;
    }

    if (hdr->skip_spi != SKIP_HTONL(SKIP_SPI))
    {
      ipsp_stat.badKpalg++;
      return -1;
    }

    BLKINC(old, sizeof(*hdr) + ivlen);


    if (crypt_decrypt(alg, c->data + c->payload_decryptstate.offset,
		      c->data + c->deskip_E_Kp.offset + c->deskip_E_Kp.len -
		      keylen, IV, old, new) == 0)
    {
      /* Decrypted, remove payload type and padding */
      /* NOTE: last few (9) bytes are assumed to be in last segment! */


      BLKDEC(new, 1);
      result = *BLKSTART(new);  /* Read payload type */

      if (blocklen > 1)         /* Remove padding for block ciphers */
      {
	int padlen;

	BLKDEC(new, 1);
	padlen = *BLKSTART(new);
	if (padlen > blocklen) {
    	  ipsp_stat.badKpalg++;
	  return -1;
	}
	BLKDEC(new, padlen);
      }
    }
  }
  else
    ipsp_stat.badKpalg++;

  return result;
}
