/*
 * Copyright (C) 1995 M. Hauber, Ch. Schneider, G. Caronni
 * See COPYING for more details
 */
#include "config.h"
#include "skip_defs.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#include "ipreass.h"

#ifdef __GNUC__
#ident "$Id: ipreass.c,v 1.4 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: ipreass.c,v 1.4 1996/04/25 15:05:30 cschneid Exp $";
#endif

#define IP_FQSIZE	10	/* max number of frag queues */
#define IP_MAXNF	10	/* max number of frags/ datagram */
#define IP_FTTL		60	/* time to live */

#define IPFF_VALID	1	/* contents are valid */
#define IPFF_BOGUS	2	/* drop frags that match */
#define IPFF_FREE	3	/* this queue is free to be allocated */

#define TIMERINTERVAL	5	/* expire interval in seconds */
#define IP_FRAGOFF	0x1fff	/* offset mask */

struct ipfq
{
  u_char ipf_state;		/* VALID, FREE or BOGUS */
  u_char ipf_src[IPADDRSIZE];	/* IP address of source */
  u_short ipf_id;		/* datagram id */
  int ipf_ttl;			/* countdown to disposal */
  QUEUE ipf_q;			/* frag queue */
};

static SEMTYPE ipfmutex;	/* semaphore */
static struct ipfq ipfqt[IP_FQSIZE];	/* IP frag queue table */

static PKTENT *ipfjoin(struct ipfq *piq)
{
  int off = 0, packoff;
  struct ip *ip;

  if (piq->ipf_state == IPFF_BOGUS)
    return NULL;

  /* see if we have the whole datagram */
  while (ip = SKIP_FRAGGET(&piq->ipf_q))
  {
    packoff = (SKIP_NTOHS(ip->ip_off) & IP_FRAGOFF) << 3;
    if (off < packoff)
      return NULL;

    off = packoff + SKIP_NTOHS(ip->ip_len) - (ip->ip_hl * 4);
  }

  return ((SKIP_NTOHS(ip->ip_off) & IP_MF) == 0) ? SKIP_FRAGCONS(&piq->ipf_q) : NULL;
}

static int ipfadd(struct ipfq *piq, PKTENT *pkt)
{
  if (piq->ipf_state != IPFF_VALID)
    return -1;

  if (SKIP_FRAGENQ(&piq->ipf_q, pkt) < 0)
  {
    SKIP_FREEQ(&piq->ipf_q);
    return -1;
  }

  piq->ipf_ttl = IP_FTTL;
  return 0;
}

int ipreass(PKTENT **pkt, struct ip *ip)
{
  int firstfree = -1, walk;
  void *ret;

  /*
   * is it a fragment?
   */
  if ((SKIP_NTOHS(ip->ip_off) & (IP_FRAGOFF | IP_MF)) == 0)
    return IPF_PROCESS;

  SEMLOCK(ipfmutex);

  for (walk = 0; walk < IP_FQSIZE; walk++)
  {
    struct ipfq *piq = &ipfqt[walk];

    if (piq->ipf_state == IPFF_FREE)
    {
      if (firstfree == -1)
        firstfree = walk;
      continue;
    }
    /* id are compared in network byte order */
    if (piq->ipf_id != ip->ip_id)
      continue;
    if (MEMCMP(piq->ipf_src, &ip->ip_src, sizeof(piq->ipf_src)))
      continue;

    /* found a match */
    if (ipfadd(piq, *pkt) < 0)
    {
      SEMUNLOCK(ipfmutex);
      return IPF_FREE;
    }

    ret = ipfjoin(piq);
    SEMUNLOCK(ipfmutex);
    if (ret)
    {
      *pkt = ret;
      return IPF_PROCESS;
    }
    return IPF_QUEUED;
  }

  /* no match */
  if (firstfree < 0)
  {
    /* no room - drop */
    SEMUNLOCK(ipfmutex);
    return IPF_FREE;
  }

  MEMCPY(ipfqt[firstfree].ipf_src, &ip->ip_src, sizeof(ipfqt[firstfree].ipf_src));
  ipfqt[firstfree].ipf_id = ip->ip_id;
  ipfqt[firstfree].ipf_ttl = IP_FTTL;
  ipfqt[firstfree].ipf_state = IPFF_VALID;

  ipfadd(&ipfqt[firstfree], *pkt);
  SEMUNLOCK(ipfmutex);
  return IPF_QUEUED;
}

static int ipftimer(int gran)
{
  int walk;

  for (walk = 0; walk < IP_FQSIZE; walk++)
  {
    struct ipfq *piq = &ipfqt[walk];

    if (piq->ipf_state == IPFF_FREE)
      continue;

    if ((piq->ipf_ttl -= gran) < 0)
    {
      if (piq->ipf_state == IPFF_BOGUS)
      {
        piq->ipf_state = IPFF_FREE;
        continue;
      }
      SKIP_FREEQ(&piq->ipf_q);
      piq->ipf_state = IPFF_FREE;
    }
  }

  return 0;
}

static int ipreass_timeout(void *arg)
{
  static int times = 1;

  /*
   * this function should not sleep
   * so we just try to lock the semaphore
   * if we do not succeed, we expire the entries
   * another time.
   */
  if (SEMLOCKTRY(ipfmutex) == 0)
  {
    ipftimer(times);
    SEMUNLOCK(ipfmutex);
    times = 1;
  }
  else
    times++;

  TIMEOUT(ipreass_timeout, NULL, TIMERINTERVAL);
  return 0;
}

int ipreass_init()
{
  int walk;

  SEMALLOC(ipfmutex);
  SEMINIT(ipfmutex);

  for (walk = 0; walk < IP_FQSIZE; walk++)
  {
    MEMZERO(&ipfqt[walk], sizeof(ipfqt[walk]));
    ipfqt[walk].ipf_state = IPFF_FREE;
  }

  TIMEOUT(ipreass_timeout, NULL, TIMERINTERVAL);
  return 0;
}

int ipreass_exit()
{
  int walk;

  UNTIMEOUT(ipreass_timeout, NULL);
  SEMFREE(ipfmutex);

  for (walk = 0; walk < IP_FQSIZE; walk++)
  {
    ipfqt[walk].ipf_state = IPFF_FREE;
    SKIP_FREEQ(&ipfqt[walk].ipf_q);
  }

  return 0;
}
