/*
 * Copyright (C) 1995,1996 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */

#include "config.h"

#include <sys/types.h>
#include "skip_defs.h"

#ifndef KERNEL

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "inet_aton.h"

#endif

#include "id.h"


#ifdef __GNUC__
#ident "$Id: id.c,v 1.8 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: id.c,v 1.8 1996/04/25 15:05:30 cschneid Exp $";
#endif

int nsid_len[SKIP_NUM_NSID] =
{
  0,    /* Invalid */
  4,    /* IPv4 */
  4,    /* Posix */
  16,   /* IPv6 */
  16,   /* DNS */
  16,   /* ISO */
  16,   /* arbitrary ASCII */
  6,    /* MAC */
  16,   /* DH */
  16,   /* Mail */
  16,   /* Bank */
  16,   /* NIS */
  4,    /* Ephemeral DH Master Keyid (EMKID) */
};




#ifndef KERNEL


int id_printhex(FILE *f, u_char *val, int len)
{
  int i;

  for (i = 0; i < len; i++)
    fprintf(f, "%02x", *val++);

  return 0;
}

int id_sprinthex(char *s, u_char *val, int len)
{
  int i;

  for (i = 0; i < len; i++, s += 2)
    sprintf(s, "%02x", *val++);

  return 0;
}

int id_printnicehex(FILE *f, u_char *val, int len)
{
  fprintf(f, "0x");
  return id_printhex(f, val, len);
}

int id_scanhex(char *val, char *buf, int l)
{
  int used;

  for (used = 0; used < l; used++, val += 2)
  {
    static char *hexdigits = "0123456789abcdefABCDEF", *hd1, *hd2;

    if ((hd1 = strchr(hexdigits, val[0])) && (hd2 = strchr(hexdigits, val[1])))
    {
      int d1 = hd1 - hexdigits;
      int d2 = hd2 - hexdigits;
      if (d1 > 15)  /* Uppercase */
	d1 -= 6;
      if (d2 > 15)  /* Uppercase */
	d2 -= 6;
      *buf++ = (d1 << 4) | d2;
    }
    else
    {
      fprintf(stderr, "skipd: malformed hex number at '%s'\n", val);
      used = -1;
      break;
    }
  }

  return used;
}

int id_saveids(FILE *f, skip_id_t *id)
{
  fprintf(f, "%u-", id->type[0]);
  id_printhex(f, id->id[0], nsid_len[id->type[0]]);
  fprintf(f, ",%u-", id->type[1]);
  id_printhex(f, id->id[1], nsid_len[id->type[1]]);
  fprintf(f, ":\n");

  return 0;
}

/**
 * Get an ID string into buffer with buflen == sizeof(buffer)
 */
int id_sprintids(char *buf, int buflen, skip_id_t *id)
{

  /* It may not be representable in this buffer,
   * return an error
   */
  if (buflen < 2 * (2 + 1 + 2*SKIP_ID_MAXLEN) + 1 + 1)
    return -1;
  
  sprintf(     buf,               "%u-", id->type[0]);
  id_sprinthex(buf + strlen(buf), id->id[0], nsid_len[id->type[0]]);
  sprintf(     buf + strlen(buf), ",%u-", id->type[1]);
  id_sprinthex(buf + strlen(buf), id->id[1], nsid_len[id->type[1]]);

  return 0;
  
}


int id_parseids(char *name, skip_id_t *id)
{
  int t0, t1, l0, l1;
  char *id1, *id2, *tmp;

  if ((id1 = strtok(name, "-")) == NULL)
    return -1;
  t0 = atoi(id1);

  if ((id1 = strtok(NULL, ", ")) == NULL)
    return -1;

  tmp = id1 + strlen(id1) + 1;

  while ((*tmp == ' ') || (*tmp == ',')) tmp++;

  if ((id2 = strtok(tmp, "-")) == NULL)
    return -1;
  t1 = atoi(id2);

  if ((id2 = strtok(NULL, ":")) == NULL)
    return -1;

  l0 = strlen(id1);
  l1 = strlen(id2);

  id->type[0] = t0, id->type[1] = t1;

  /* be smart and support standard IP notation in name space 1 */
  if (t0 == SKIP_NSID_IPV4 && strchr(id1, '.')) {
    /* IP address in NS 1 */
    if (!inet_aton(id1, (struct in_addr *) &id->id[0])) {
      fprintf(stderr, "skipd: %s is not an IP address\n", id1);
      return -1;
    }
  }
  else {
    if (l0 & 1) {
      fprintf(stderr, "skipd: malformed hex number, length is odd\n");
      return -1;
    }

    l0 /= 2;

    if (l0 > nsid_len[t0]) {
      fprintf(stderr, "skipd: master key id too large\n");
      return -1;
    }

    if (id_scanhex(id1, id->id[0], l0) < 0) 
      return -1;
  }
    
  if (t1 == SKIP_NSID_IPV4 && strchr(id2, '.')) {
    /* IP address in NS 1 */
    if (!inet_aton(id2, (struct in_addr *) &id->id[1])) {
      fprintf(stderr, "skipd: %s is not an IP address\n", id2);
      return -1;
    }
  }
  else {
    if (l1 & 1) {
      fprintf(stderr, "skipd: malformed hex number, length is odd\n");
      return -1;
    }

    l1 /= 2;

    if (l1 > nsid_len[t1]) {
      fprintf(stderr, "skipd: master key id too large\n");
      return -1;
    }

    if (id_scanhex(id2, id->id[1], l1) < 0) 
      return -1;
  }
    
  return 0;
}

#endif
