/*
 * Copyright (C) 1995-1997 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */
/*
 * TODO:
 *  - compression
 *  - ip reassembly on input
 */

#include "config.h"

#include <errno.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#include "skip_defs.h"
#include "id.h"
#include "dynamic.h"
#include "memblk.h"
#include "skip.h"
#include "skipcache.h"
#include "ipsp.h"
#include "ipsum.h"
#include "crypt.h"
#include "sign.h"
#include "signmd5.h"
#include "random.h"
#include "ah.h"
#include "esp.h"

#ifdef __GNUC__
#ident "$Id: ipsp.c,v 1.9 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: ipsp.c,v 1.9 1996/04/25 15:05:30 cschneid Exp $";
#endif

#define ALIGN4(len) (((len) + 3) & ~3)

 
struct ipsp_hdr
{
  u_char version, srcnsid, dstnsid, nextproto;
  u_int32 n;
  u_char Kij_alg, Crypt_alg, MAC_alg, Comp_alg;
};  
 
ipsp_stat_t ipsp_stat;

/*
 * the first segment must contain the complete ip header + start of payload
 * and there must be ipsp_maxheadergrowth() bytes of memory in front
 * of the original ip header, ipsp_maxtrailergrowth() behind packet
 */
int ipsp_ip2ipsp(struct skipcache *c, struct memblk *old, struct memblk *new)
{
  struct ip *ip = (struct ip *)BLKSTART(old);
  struct ipsp_hdr *hdr;
  int tunnel;		/* tunnel or transport mode */
#ifndef SMALL_KERNEL_STACK
  struct memblk payload;
  struct memblk ah;	/* pointer to beginning of authenticated data */
  struct memblk data;
#else /* SMALL_KERNEL_STACK */
  struct memblk *payload;
  struct memblk *ah;	/* pointer to beginning of authenticated data */
  struct memblk *data;
#endif /* SMALL_KERNEL_STACK */
  int result = SKIP_PROCESSED;

  /* Ensure we got valid keys for enskipping, otherwise queue packet */
  if ((c->enskip_MAC_alg || c->enskip_Crypt_alg) &&
      ((c->enskip_Kp_maxbytes && (c->enskip_Kp_bytes >= c->enskip_Kp_maxbytes))
       || (c->enskip_Kp_ttl < 0) || (c->enskip_Kp_Kijn.n != skip_n)))
  {
    /* Enough data to calculate A_Kp and E_Kp for n? */
    if (c->Kijn[skip_n % 3].n == skip_n)
      random_randomKp(c);
    else
    {
      c->flags |= SKIP_ENSKIP_INVALID;
      return SKIP_QUEUED;
    }
  }

  c->enskip_Kp_bytes += new->len;

#ifndef SMALL_KERNEL_STACK
  data = *new;
#else /* SMALL_KERNEL_STACK */
  if ((ah = KALLOC(sizeof(*ah))) == NULL)
    return SKIP_DISCARD;
  if ((payload = KALLOC(sizeof(*payload))) == NULL) {
    KFREE(ah, sizeof(*ah));
    return SKIP_DISCARD;
  }
  if ((data = KALLOC(sizeof(*data))) == NULL) {
    KFREE(payload, sizeof(*payload));
    KFREE(ah, sizeof(*ah));
    return SKIP_DISCARD;
  }
  MEMCPY(data, new, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */

  /* Copy old IP header */
#ifndef SMALL_KERNEL_STACK
  BLKADD(&data, ip, sizeof(*ip));

  hdr = (struct ipsp_hdr *)BLKSTART(&data);
#else /* SMALL_KERNEL_STACK */
  BLKADD(data, ip, sizeof(*ip));

  hdr = (struct ipsp_hdr *)BLKSTART(data);
#endif /* SMALL_KERNEL_STACK */

  MEMZERO(hdr, sizeof(*hdr));
  /* we send it in tunnel mode if that's configured or if it's a fragment */
  tunnel = (c->flags & SKIP_TUNNEL) || (ip->ip_off & ~SKIP_HTONS(IP_DF|IP_CE));

  hdr->version = SKIP_VERSION;
  hdr->n = SKIP_HTONL(skip_n);
  hdr->Kij_alg = (u_char)c->enskip_Kij_alg;
  hdr->Crypt_alg = (u_char)c->enskip_Crypt_alg;
  hdr->MAC_alg = (u_char)c->enskip_MAC_alg;
  hdr->Comp_alg = (u_char)c->enskip_Comp_alg;

  if (c->enskip_MAC_alg)
    hdr->nextproto = AH_PROTO;
  else if (c->enskip_Crypt_alg)
    hdr->nextproto = ESP_PROTO;
  else if (tunnel)
    hdr->nextproto = IPPROTO_ENCAP;
  else
    hdr->nextproto = ip->ip_p;

#ifndef SMALL_KERNEL_STACK
  BLKINC(&data, sizeof(*hdr));
#else /* SMALL_KERNEL_STACK */
  BLKINC(data, sizeof(*hdr));
#endif /* SMALL_KERNEL_STACK */

  /* Insert encrypted packet key if encrypting or authenticating */
  if (c->enskip_Crypt_alg || c->enskip_MAC_alg)
  {
#ifndef SMALL_KERNEL_STACK
    BLKADD(&data, c->data + c->enskip_Kp_Kijn.key.offset, c->enskip_Kp_Kijn.key.len);
#else /* SMALL_KERNEL_STACK */
    BLKADD(data, c->data + c->enskip_Kp_Kijn.key.offset, c->enskip_Kp_Kijn.key.len);
#endif /* SMALL_KERNEL_STACK */
  }

  /* Need to add src MKID? */
  if ((c->enskip_mkid[0] == ENSKIPMKID_YES) ||
      ((c->enskip_mkid[0] == ENSKIPMKID_AUTO) && 
       ((c->id.type[0] != SKIP_NSID_IPV4) ||
        MEMCMP(c->id.id[0], &ip->ip_src, sizeof(ip->ip_src)))))
  {
    int nsidlen = nsid_len[c->id.type[0]];

    hdr->srcnsid = c->id.type[0];
#ifndef SMALL_KERNEL_STACK
    BLKADD(&data, c->id.id[0], nsidlen);
#else /* SMALL_KERNEL_STACK */
    BLKADD(data, c->id.id[0], nsidlen);
#endif /* SMALL_KERNEL_STACK */
  }

  /* Need to add dst MKID? */
  if ((c->enskip_mkid[1] == ENSKIPMKID_YES) ||
      ((c->enskip_mkid[1] == ENSKIPMKID_AUTO) && 
       ((c->id.type[1] != SKIP_NSID_IPV4) ||
        MEMCMP(c->id.id[1], &ip->ip_dst, sizeof(ip->ip_dst)))))
  {
    int nsidlen = nsid_len[c->id.type[1]];

    hdr->dstnsid = c->id.type[1];
#ifndef SMALL_KERNEL_STACK
    BLKADD(&data, c->id.id[1], nsidlen);
#else /* SMALL_KERNEL_STACK */
    BLKADD(data, c->id.id[1], nsidlen);
#endif /* SMALL_KERNEL_STACK */
  }

  /* Prepare calculation of AH MAC, reserve space */
  if (c->enskip_MAC_alg)
  {
    int nextproto;

    /* Next protocol inside AH */
    if (c->enskip_Crypt_alg)
      nextproto = ESP_PROTO;
    else if (tunnel)
      nextproto = IPPROTO_ENCAP;
    else
      nextproto = ip->ip_p;

#ifndef SMALL_KERNEL_STACK
    ah = data;
    if (ah_prepare(c, &data, nextproto) != 0) {
#else /* SMALL_KERNEL_STACK */
    MEMCPY(ah, data, sizeof(*ah));
    if (ah_prepare(c, data, nextproto) != 0) {
      KFREE(data, sizeof(*data));
      KFREE(payload, sizeof(*payload));
      KFREE(ah, sizeof(*ah));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;
    }
  }

#ifndef SMALL_KERNEL_STACK
  payload = *old;
#else /* SMALL_KERNEL_STACK */
  MEMCPY(payload, old, sizeof(*payload));
#endif /* SMALL_KERNEL_STACK */
  if (tunnel) {		/* Include ip header only in tunnel mode */
    ipsp_stat.tunnel_out++;
    result |= SKIP_P_TUNNEL;
  } 
  else
#ifndef SMALL_KERNEL_STACK
    BLKINC(&payload, ip->ip_hl * 4);
#else /* SMALL_KERNEL_STACK */
    BLKINC(payload, ip->ip_hl * 4);
#endif /* SMALL_KERNEL_STACK */

  if (c->enskip_Crypt_alg)
  {
    /* Next protocol inside ESP */
    int nextproto = tunnel ? IPPROTO_ENCAP : ip->ip_p;

#ifndef SMALL_KERNEL_STACK
    if (esp_encrypt(c, &payload, &data, nextproto) == 0) {
#else /* SMALL_KERNEL_STACK */
    if (esp_encrypt(c, payload, data, nextproto) == 0) {
#endif /* SMALL_KERNEL_STACK */
      ipsp_stat.crypt_out++;
      result |= SKIP_P_ENCRYPT;
    }
    else {
#ifdef SMALL_KERNEL_STACK
      KFREE(data, sizeof(*data));
      KFREE(payload, sizeof(*payload));
      KFREE(ah, sizeof(*ah));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;
    }
  }
  else
  {
#ifndef SMALL_KERNEL_STACK
    memblk_copy(&data, &payload);	/* Copy from payload to new data */
#else /* SMALL_KERNEL_STACK */
    memblk_copy(data, payload);	/* Copy from payload to new data */
#endif /* SMALL_KERNEL_STACK */
  }

  /* Adjust length of newly created packet to contain everything */
#ifndef SMALL_KERNEL_STACK
  new->len -= data.len;		/* Data.len is length left in buffer */
#else /* SMALL_KERNEL_STACK */
  new->len -= data->len;		/* Data.len is length left in buffer */
#endif /* SMALL_KERNEL_STACK */

  ip = (struct ip *)BLKSTART(new);

  /* Update IP header to reflect changes */
  ip->ip_p = IPPROTO_SKIP;
  ip->ip_off &= SKIP_HTONS(IP_DF|IP_CE); /* not fragmented, preserve DF/CE */
  ip->ip_hl = sizeof(*ip) / 4;
  ip->ip_len = SKIP_HTONS(new->len);

  /* Calculate AH MAC, data points to end of packet */
  if (c->enskip_MAC_alg)
  {
#ifndef SMALL_KERNEL_STACK
    if (ah_calc(c, &ah, new) == 0) {
#else /* SMALL_KERNEL_STACK */
    if (ah_calc(c, ah, new) == 0) {
#endif /* SMALL_KERNEL_STACK */
      ipsp_stat.auth_out++;
      result |= SKIP_P_AUTH;
    } 
    else {
#ifdef SMALL_KERNEL_STACK
      KFREE(data, sizeof(*data));
      KFREE(payload, sizeof(*payload));
      KFREE(ah, sizeof(*ah));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;
    }
  }

  ip->ip_sum = SKIP_HTONS(ipsum(ip));

  ipsp_stat.out++;

#ifdef SMALL_KERNEL_STACK
  KFREE(data, sizeof(*data));
  KFREE(payload, sizeof(*payload));
  KFREE(ah, sizeof(*ah));
#endif /* SMALL_KERNEL_STACK */

  return result;
}

/*
 * the first segment must contain the complete ipsp header + start of payload
 */
int ipsp_ipsp2ip(struct skipcache *c, struct memblk *old, struct memblk *new, int dir)
{
  struct ip *ip = (struct ip *)BLKSTART(old);
  struct ipsp_hdr *hdr;
#ifndef SMALL_KERNEL_STACK
  struct memblk data, newdata;
#else /* SMALL_KERNEL_STACK */
  struct memblk *data, *newdata;
#endif /* SMALL_KERNEL_STACK */
  u_int32 pkt_n;
  int Kij_alg, Kp_Kij_len;
  int nextproto;
  u_short ip_sum;
  int result = SKIP_PROCESSED;
  
  ipsp_stat.in++;

  if ((ip->ip_hl < sizeof(*ip) / 4) || (SKIP_NTOHS(ip->ip_len) != old->len))
  {
    ipsp_stat.badpktlen++;
    return SKIP_DISCARD;
  }

  ip_sum = SKIP_NTOHS(ip->ip_sum);

  /* check ip checksum */
  if (ip_sum && (ipsum(ip) != ip_sum))
  {
    ipsp_stat.badipsum++;
    return SKIP_DISCARD;
  }

  /* drop pkt if it is a fragment */
  if (ip->ip_off & ~SKIP_HTONS(IP_DF|IP_CE))
  {
    ipsp_stat.fragdrop++;
    return SKIP_DISCARD;
  }

#ifndef SMALL_KERNEL_STACK
  data = *old;

  BLKINC(&data, ip->ip_hl * 4);
  hdr = (struct ipsp_hdr *)BLKSTART(&data);
  BLKINC(&data, sizeof(struct ipsp_hdr));
#else /* SMALL_KERNEL_STACK */
  if ((data = KALLOC(sizeof(*data))) == NULL)
    return SKIP_DISCARD;

  MEMCPY(data, old, sizeof(*data));

  BLKINC(data, ip->ip_hl * 4);
  hdr = (struct ipsp_hdr *)BLKSTART(data);
  BLKINC(data, sizeof(struct ipsp_hdr));
#endif /* SMALL_KERNEL_STACK */

  /* is packet too short? (header excl. next proto field, reserved) */
#ifndef SMALL_KERNEL_STACK
  if (data.len < 0)
#else /* SMALL_KERNEL_STACK */
  if (data->len < 0)
#endif /* SMALL_KERNEL_STACK */
  {
    ipsp_stat.badpktlen++;
#ifdef SMALL_KERNEL_STACK
    KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
    return SKIP_DISCARD;
  }

  if (hdr->version != SKIP_VERSION)  /* XXX Also checks if Rsvd is 0! */
  {
    ipsp_stat.badversion++;
#ifdef SMALL_KERNEL_STACK
    KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
    return SKIP_DISCARD;
  }

  pkt_n = SKIP_NTOHL(hdr->n);

  /* Be graceful: Only check validity of pkt n if encrypting or auth. */
  if (hdr->Crypt_alg || hdr->MAC_alg)
  {
    if ((pkt_n < skip_n - 1) || (pkt_n > skip_n + 1))
    {
      ipsp_stat.badn++;
#ifdef SMALL_KERNEL_STACK
      KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;
    }
  }

  if (hdr->Comp_alg)  /* Currently, no compression is supported */
  {
    ipsp_stat.badCompalg++;
#ifdef SMALL_KERNEL_STACK
    KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
    return SKIP_DISCARD;
  }

  /* Calculate size of encrypted packet key Kp_Kijn */
  Kp_Kij_len = 0;

  if (hdr->Crypt_alg)
  {
    if ((Kp_Kij_len = crypt_keylen(hdr->Crypt_alg)) < 0)
    {
      ipsp_stat.badKpalg++;
#ifdef SMALL_KERNEL_STACK
      KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;
    }
  }
  else if (c->deskip_policy[dir] & SAID_CRYPT)
  {
    ipsp_stat.badpolicy++;  /* Packets not encrypted are not allowed */
#ifdef SMALL_KERNEL_STACK
    KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
    return SKIP_DISCARD;
  }

  if (hdr->MAC_alg)
  {
    if (Kp_Kij_len < SIGN_KEYLEN)
      Kp_Kij_len = SIGN_KEYLEN;
  }
  else if (c->deskip_policy[dir] & SAID_AUTH)
  {
    ipsp_stat.badpolicy++;  /* Packets not authenticated are not allowed */
#ifdef SMALL_KERNEL_STACK
    KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
    return SKIP_DISCARD;
  }

  Kij_alg = hdr->Kij_alg;

  if (Kp_Kij_len)
  {
    int blocklen = crypt_blocklen(Kij_alg);

    if (blocklen > 1)  /* Kij alg valid and not a stream cipher? */
    {
      if (Kp_Kij_len % blocklen)
	Kp_Kij_len += blocklen - (Kp_Kij_len % blocklen);
    }
    else
    {
      ipsp_stat.badKijalg++;
#ifdef SMALL_KERNEL_STACK
      KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;
    }

#ifndef SMALL_KERNEL_STACK
    if (data.len < Kp_Kij_len)
#else /* SMALL_KERNEL_STACK */
    if (data->len < Kp_Kij_len)
#endif /* SMALL_KERNEL_STACK */
    {
      ipsp_stat.badpktlen++;
#ifdef SMALL_KERNEL_STACK
      KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;
    }

    /* New encrypted packet key Kp_Kijn? Yes -> Decrypt if possible */
    if ((c->deskip_Kp_Kijn.n != pkt_n) ||
	(c->deskip_Kij_alg != Kij_alg) ||
	(c->deskip_Kp_Kijn_len != Kp_Kij_len) ||
#ifndef SMALL_KERNEL_STACK
	MEMCMP(BLKSTART(&data), c->data + c->deskip_Kp_Kijn.key.offset, Kp_Kij_len))
#else /* SMALL_KERNEL_STACK */
	MEMCMP(BLKSTART(data), c->data + c->deskip_Kp_Kijn.key.offset, Kp_Kij_len))
#endif /* SMALL_KERNEL_STACK */
    {
      if (c->Kijn[pkt_n % 3].n == pkt_n)  /* Got valid Kijn for pkt_n? */
      {
#ifndef SMALL_KERNEL_STACK
	u_char Kp[MAXKEYLEN];
#else /* SMALL_KERNEL_STACK */
	u_char *Kp;
#endif /* SMALL_KERNEL_STACK */
	u_char MI[MAXBLOCKLEN];
	struct memblk inmb, outmb;
	struct memseg inms, outms;
	int keylen = crypt_keylen(Kij_alg);

#ifdef SMALL_KERNEL_STACK
	if ((Kp = KALLOC(MAXKEYLEN)) == NULL) {
          KFREE(data, sizeof(*data));
	  return SKIP_DISCARD;
	}
#endif /* SMALL_KERNEL_STACK */

	MEMZERO(&inmb, sizeof(inmb));
	inmb.len = Kp_Kij_len;
	inmb.ms = &inms;
#ifndef SMALL_KERNEL_STACK
	inms.ptr = BLKSTART(&data);
#else /* SMALL_KERNEL_STACK */
	inms.ptr = BLKSTART(data);
#endif /* SMALL_KERNEL_STACK */
	inms.len = Kp_Kij_len;

	MEMZERO(&outmb, sizeof(outmb));
	outmb.len = Kp_Kij_len;
	outmb.ms = &outms;
	outms.ptr = Kp;
	outms.len = Kp_Kij_len;

	MEMZERO(MI, sizeof(MI));

	
	if (crypt_decrypt(Kij_alg, c->data + c->Kp_decryptstate.offset,
			  c->data + c->Kijn[pkt_n % 3].key.offset +
			  c->Kijn[pkt_n % 3].key.len - keylen, MI,
			  &inmb, &outmb) < 0)
	{
	  ipsp_stat.badKijalg++;
#ifdef SMALL_KERNEL_STACK
	  KFREE(Kp, MAXKEYLEN);
          KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
	  return SKIP_DISCARD;
	}
	
	/* Copy new packet key we got and decrypted */
	c->deskip_Kp_Kijn.n = pkt_n;
	c->deskip_Kij_alg = Kij_alg;
	c->deskip_Kp_Kijn_len = Kp_Kij_len;
#ifndef SMALL_KERNEL_STACK
	MEMCPY(c->data + c->deskip_Kp_Kijn.key.offset, BLKSTART(&data), Kp_Kij_len);
#else /* SMALL_KERNEL_STACK */
	MEMCPY(c->data + c->deskip_Kp_Kijn.key.offset, BLKSTART(data), Kp_Kij_len);
#endif /* SMALL_KERNEL_STACK */
	
	/*
	 * We always calc the whole length of both A_Kp and E_Kp, simple...
	 */
	random_expandKp(Kp, Kp_Kij_len,
			c->data + c->deskip_E_Kp.offset, c->deskip_E_Kp.len,
			c->data + c->deskip_A_Kp.offset, c->deskip_A_Kp.len,
			hdr->Crypt_alg, hdr->MAC_alg);
#ifdef SMALL_KERNEL_STACK
	KFREE(Kp, MAXKEYLEN);
#endif /* SMALL_KERNEL_STACK */
      }
      else
      {
	c->flags |= SKIP_DESKIP_INVALID;
#ifdef SMALL_KERNEL_STACK
        KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
	return SKIP_QUEUED;
      }
    }

#ifndef SMALL_KERNEL_STACK
    BLKINC(&data, Kp_Kij_len);
#else /* SMALL_KERNEL_STACK */
    BLKINC(data, Kp_Kij_len);
#endif /* SMALL_KERNEL_STACK */
  }

  /*
   * Skip MKIDs. Note: They are assumed to be valid since ipsp_getid
   * (used to get the cache entry) didn't complain about this packet
   */
#ifndef SMALL_KERNEL_STACK
  BLKINC(&data, nsid_len[hdr->srcnsid] + nsid_len[hdr->dstnsid]);
#else /* SMALL_KERNEL_STACK */
  BLKINC(data, nsid_len[hdr->srcnsid] + nsid_len[hdr->dstnsid]);
#endif /* SMALL_KERNEL_STACK */

#ifndef SMALL_KERNEL_STACK
  if (data.len < 0)
#else /* SMALL_KERNEL_STACK */
  if (data->len < 0)
#endif /* SMALL_KERNEL_STACK */
  {
    ipsp_stat.badpktlen++;
#ifdef SMALL_KERNEL_STACK
    KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
    return SKIP_DISCARD;
  }

  /* Now we got valid keys A_Kp and E_Kp for this packet, process it */
  nextproto = hdr->nextproto;

  /* We rely on the order AH, ESP, payload (IP or TCP/UDP/...) */
  if (nextproto == AH_PROTO)
  {
#ifndef SMALL_KERNEL_STACK
    if ((nextproto = ah_check(c, hdr->MAC_alg, &data, old)) < 0) {
#else /* SMALL_KERNEL_STACK */
    if ((nextproto = ah_check(c, hdr->MAC_alg, data, old)) < 0) {
      KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;  /* IPSP stats are updated by ah */
    }

    ipsp_stat.auth_in++;
    result |= SKIP_P_AUTH;
  }

#ifndef SMALL_KERNEL_STACK
  newdata = *new;
#else /* SMALL_KERNEL_STACK */
  if ((newdata = KALLOC(sizeof(*newdata))) == NULL) {
    KFREE(data, sizeof(*data));
    return SKIP_DISCARD;
  }
  MEMCPY(newdata, new, sizeof(*newdata));
#endif /* SMALL_KERNEL_STACK */

  if (nextproto == ESP_PROTO)
  {
    /* Copy IP header in transport mode, will be ignored in tunnel mode */
#ifndef SMALL_KERNEL_STACK
    BLKADD(&newdata, ip, ip->ip_hl * 4);
#else /* SMALL_KERNEL_STACK */
    BLKADD(newdata, ip, ip->ip_hl * 4);
#endif /* SMALL_KERNEL_STACK */

#ifndef SMALL_KERNEL_STACK
    if ((nextproto = esp_decrypt(c, hdr->Crypt_alg, &data, &newdata)) < 0) {
#else /* SMALL_KERNEL_STACK */
    if ((nextproto = esp_decrypt(c, hdr->Crypt_alg, data, newdata)) < 0) {
      KFREE(newdata, sizeof(*newdata));
      KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */
      return SKIP_DISCARD;  /* IPSP stats are updated by esp */
    }

    ipsp_stat.crypt_in++;
    result |= SKIP_P_DECRYPT;

    if (nextproto == IPPROTO_ENCAP)
      BLKINC(new, ip->ip_hl * 4);  /* Ignore copied IP header */
  }
  else
  {
    if (nextproto != IPPROTO_ENCAP)  /* Copy old IP hdr unless in tunnel mode */
#ifndef SMALL_KERNEL_STACK
      BLKADD(&newdata, ip, ip->ip_hl * 4);
#else /* SMALL_KERNEL_STACK */
      BLKADD(newdata, ip, ip->ip_hl * 4);
#endif /* SMALL_KERNEL_STACK */

#ifndef SMALL_KERNEL_STACK
    memblk_copy(&newdata, &data);
#else /* SMALL_KERNEL_STACK */
    memblk_copy(newdata, data);
#endif /* SMALL_KERNEL_STACK */
  }

  /* Adjust length of newly created packet to contain just packet */
#ifndef SMALL_KERNEL_STACK
  new->len -= newdata.len;   /* Newdata.len is length left in buffer */
#else /* SMALL_KERNEL_STACK */
  new->len -= newdata->len;   /* Newdata.len is length left in buffer */
#endif /* SMALL_KERNEL_STACK */

  ip = (struct ip *)BLKSTART(new);

  if (nextproto != IPPROTO_ENCAP)
  {
    ip->ip_p = nextproto;
    ip->ip_len = SKIP_HTONS(new->len);
    ip->ip_sum = SKIP_HTONS(ipsum(ip));
  }
  else
  {
    ipsp_stat.tunnel_in++;
    result |= SKIP_P_TUNNEL;

    /* Minimal sanity checks */
    if ((ip->ip_v != IPVERSION) || (ipsum(ip) != SKIP_NTOHS(ip->ip_sum)))
    {
      ipsp_stat.badipsum++;
#ifdef SMALL_KERNEL_STACK
      KFREE(newdata, sizeof(*newdata));
      KFREE(data, sizeof(*data));
#endif
      return SKIP_DISCARD;  /* Minimal sanity checks failed, drop */
    }
  }

#ifdef SMALL_KERNEL_STACK
  KFREE(newdata, sizeof(*newdata));
  KFREE(data, sizeof(*data));
#endif /* SMALL_KERNEL_STACK */

  return result;
}

int ipsp_maxheadergrowth()
{
  return ALIGN4(sizeof(struct ip) + sizeof(struct ipsp_hdr) +
                MAXKEYLEN + 2 * SKIP_ID_MAXLEN +
                ah_maxheadergrowth() + esp_maxheadergrowth());
}

/* the getid function is only used by skip.c; 
   hence assert ip_p == IPPROTO_SKIP and skip the protocol check */
int ipsp_getid(struct memblk *m, skip_id_t *id)
{
  int result = 0;
  struct ip *ip = (struct ip *)BLKSTART(m);
  struct ipsp_hdr *hdr = (struct ipsp_hdr *)(((u_char *)ip) + ip->ip_hl * 4);
  int kplen = 0;
  int blocklen = crypt_blocklen(hdr->Kij_alg);
 
  /* Fill in src and dst IPs */
  MEMZERO(id, sizeof(*id));
  /* Default to IPv4 NSIDs */
  id->type[0] = id->type[1] = SKIP_NSID_IPV4;
  MEMCPY(id->id[0], &ip->ip_src, sizeof(ip->ip_src));
  MEMCPY(id->id[1], &ip->ip_dst, sizeof(ip->ip_dst));
 

  /* Calc length of Kp_Kijn in packet */
  if (hdr->Crypt_alg)
    kplen = crypt_keylen(hdr->Crypt_alg);	/* May fail... */

  if ((kplen >= 0) && (blocklen > 0))
  {
    u_char *nsid = (u_char *)(hdr + 1);
 
    /* If authenticating, packet may contain longer Kp_Kijn */
    if (hdr->MAC_alg && (kplen < SIGN_KEYLEN))
      kplen = SIGN_KEYLEN;

    if (kplen % blocklen)     /* Calc padded length of Kp_Kij */
      kplen += blocklen - (kplen % blocklen);
 
    nsid += kplen;    /* Skip past Kp_Kij */

    if (hdr->srcnsid)
    {
      if (hdr->srcnsid < SKIP_NUM_NSID)
      {
        id->type[0] = hdr->srcnsid;
        MEMCPY(id->id[0], nsid, nsid_len[hdr->srcnsid]);
        nsid += nsid_len[hdr->srcnsid];
      }
      else
        result = -1;		/* Invalid src NSID */
    }
 
    if (hdr->dstnsid)
    {
      if (hdr->dstnsid < SKIP_NUM_NSID)
      {
        id->type[1] = hdr->dstnsid;
        MEMCPY(id->id[1], nsid, nsid_len[hdr->dstnsid]);
      }
      else
        result = -1;		/* Invalid src NSID */
    }
  }
  else
    result = -1;
 
  return result;
}
 
int ipsp_getheadersize(void *packetptr)
{
  struct ip *ip = (struct ip *)packetptr;
  int len = ip->ip_hl * 4 + 2;  /* Size of current ip header + start of data */
 
  if (ip->ip_p == IPPROTO_SKIP)
    len += ipsp_maxheadergrowth();
 
  return len;
}

int ipsp_init()
{
  MEMZERO(&ipsp_stat, sizeof(ipsp_stat));
  return 0;
}

int ipsp_exit()
{
  return 0;
}
