/*
 * Copyright (C) 1995,1996 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */

/* #define DEBUG_SKIPC */

#include "config.h"
#include <errno.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/udp.h>

#include "skip_defs.h"
#include "memblk.h"
#include "dynamic.h"
#include "skip.h"
#include "skipcache.h"
#include "ipsp.h"
#include "queue.h"
#include "com.h"
#include "interface.h"

#ifdef __GNUC__
#ident "$Id: skip.c,v 1.5 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: skip.c,v 1.5 1996/04/25 15:05:30 cschneid Exp $";
#endif

static int skip_queue(struct skipcache *c, skip_id_t *id, int dir,
		      void *interface, void *header, void *pkt)
{
  int ret = SKIP_DISCARD;
  struct skipcache *l;

  if (c == NULL)
  {
    if ((l = KALLOC(sizeof(*l) + sizeof(struct _kernel_queue))) == NULL)
      return SKIP_DISCARD;

    MEMZERO(l, sizeof(*l) + sizeof(struct _kernel_queue));
    MEMCPY(&l->id, id, sizeof(*id));
    l->ttl = l->maxttl = QUEUETTL;
    l->q.len = l->datalen = sizeof(struct _kernel_queue);
    l->data = (u_char *)(l + 1);

    queue_initqueues(l);

    if (skipcache_update(l) < 0)
    {
      KFREE(l, sizeof(*l) + sizeof(struct _kernel_queue));
      return SKIP_DISCARD;
    }
    /* inc reference counter since we use entry to enqueue */
    skipcache_lookup(&l->id);
  }
  else
    l = c;

#ifdef DEBUG_SKIPC
  printf(">>skip_queue: dir %d, id = %u-%u.%u.%u.%u/%u-%u.%u.%u.%u\n", dir,
	 id->type[0], id->id[0][0], id->id[0][1], id->id[0][2], id->id[0][3],
	 id->type[1], id->id[1][0], id->id[1][1], id->id[1][2], id->id[1][3]);
#endif

  if (dir == SKIP_OUTPUT)
  {
    if (queue_enqueueout(l, interface, header, pkt) == 0)
    {
      ipsp_stat.enqueued_out++;
      ret = SKIP_QUEUED;
    }
    else
      ipsp_stat.queuelimit++;
  }
  else
  {
    /* Queue input */
    if (queue_enqueuein(l, interface, header, pkt) == 0)
    {
      ipsp_stat.enqueued_in++;
      ret = SKIP_QUEUED;
    }
    else
      ipsp_stat.queuelimit++;
  }

  if ((ret == SKIP_QUEUED) && ((l->flags & SKIP_SIGNALED) == 0))
  {
#ifdef DEBUG_SKIPC
    printf(">>skip_queue: signal sent!\n");
#endif
    if (com_signal(SIGURG) >= 0)
      l->flags |= SKIP_SIGNALED;
  }

  /* release newly allocated q entry */
  if (c == NULL)
    skipcache_release(l);

  return ret;
}

static int skip_lookup(struct skipcache **c, int dir, void *interface,
		       void *header, void **pkt, struct memblk *m)
{
  int ret = SKIP_PROCESSED;
  skip_id_t *id;
  struct ip *ip = (struct ip *)BLKSTART(m);

  if ((id = KALLOC(sizeof(*id))) == NULL)
    return -1;

  /* Packet may contain skip_id_t if protocol is SKIP */
  if (ip->ip_p == IPPROTO_SKIP)
  {
    ipsp_getid(m, id);

    if (dir == SKIP_INPUT)  /* Swap src & dst if input */
    {
      skip_id_t t;

      t.type[0] = id->type[0];
      id->type[0] = id->type[1];
      id->type[1] = t.type[0];
      MEMCPY(t.id[0], id->id[0], sizeof(t.id[0]));
      MEMCPY(id->id[0], id->id[1], sizeof(id->id[0]));
      MEMCPY(id->id[1], t.id[0], sizeof(id->id[1]));
    }
  }
  else /* protocol is not SKIP (->enskip) */
  {
    MEMZERO(id, sizeof(*id));

    id->type[0] = id->type[1] = SKIP_NSID_IPV4;

    if (dir == SKIP_INPUT)  /* Swap src & dst if input */
    {
      MEMCPY(id->id[1], &ip->ip_src, sizeof(ip->ip_src));
      MEMCPY(id->id[0], &ip->ip_dst, sizeof(ip->ip_dst));
    }
    else
    {
      MEMCPY(id->id[0], &ip->ip_src, sizeof(ip->ip_src));
      MEMCPY(id->id[1], &ip->ip_dst, sizeof(ip->ip_dst));
    }
  }

  *c = skipcache_lookup(id);

  /*
   * enqueue packet if no valid key was found, else process it
   */
  if ((*c == NULL) || (((*c)->flags & (SKIP_VALIDKEY | SKIP_NOKEY)) == 0))
  {
    ret = skip_queue(*c, id, dir, interface, header, *pkt);
    skipcache_release(*c);
  }

  KFREE(id, sizeof(*id));

  return ret;
}

static int skip_enskip(struct skipcache *c, int dir, void **pkt,
		       struct memblk *old, struct memblk *new)
{
  int result = SKIP_PROCESSED;
  struct ip *ip = (struct ip *)BLKSTART(old);
  int mode;

  switch (ip->ip_p)
  {
    case IPPROTO_TCP:  mode = ENSKIPMODE_TCP;   break;
    case IPPROTO_UDP:  mode = ENSKIPMODE_UDP;   break;
    case IPPROTO_ICMP: mode = ENSKIPMODE_ICMP;  break;
    default:           mode = ENSKIPMODE_OTHER; break;
  }

  if (c->enskip_mode[dir] & mode)
  {
    int growth = ipsp_maxheadergrowth();

    if (interface_getbuf(new, pkt, growth + old->len) == 0)
      result = ipsp_ip2ipsp(c, old, new);
    else
      result = SKIP_DISCARD;
  }

  return result;
}

static int skip_deskip(struct skipcache *c, int dir, void **pkt,
		       struct memblk *old, struct memblk *new)
{
  int result = SKIP_PROCESSED;

  if (c->deskip_mode[dir] == DESKIPMODE_DESKIP)
  {
    int growth = ipsp_maxheadergrowth();

    if (interface_getbuf(new, pkt, growth + old->len) == 0)
      result = ipsp_ipsp2ip(c, old, new, dir);
    else
      result = SKIP_DISCARD;
  }

  return result;
}

static int skip_convert(struct skipcache *c, int dir,
			void *interface, void *header,
			void **pkt, struct memblk *old, struct memblk *new)
{
  int ret = SKIP_PROCESSED;

  if (c->flags & SKIP_VALIDKEY)
  {
    struct ip *ip = (struct ip *)BLKSTART(old);
    void *oldpkt = *pkt;

    ret = SKIP_DISCARD;

    ret = (ip->ip_p == IPPROTO_SKIP) ? 
          skip_deskip(c, dir, pkt, old, new) :
	  skip_enskip(c, dir, pkt, old, new);

    if (ret == SKIP_QUEUED)
      ret = skip_queue(c, &c->id, dir, interface, header, oldpkt);
  }

  return ret;
}

/*
 * Input: (system dependent) packet (will be replaced with new if processed),
 *        packet, (system-dependent) packet header & interface, direction
 */
int skip_process(int dir, void *interface, void *header, 
		 void **pkt, struct memblk *old, struct memblk *new)
{
  int ret = SKIP_PROCESSED;
  struct ip *ip = (struct ip *)BLKSTART(old);
  struct skipcache *c;
  
#ifdef DEBUG_SKIPC
  printf(">> skip_process: ip len before %d\n", SKIP_NTOHS(ip->ip_len));
#endif
  /*
   * XXX TODO: reassembly depending on dir!
   */

#ifndef TRANSPARENT_UDP_IN_INTERFACE
  /* Check for transparent UDP ports for certificate discovery */
  if ((ip->ip_p == IPPROTO_UDP) &&
      (SKIP_NTOHS(ip->ip_len) >= ip->ip_hl * 4 + sizeof(struct udphdr)))
  {
    struct udphdr *udp = (struct udphdr *)(((u_char *)ip) + ip->ip_hl * 4); 
    u_short sport = SKIP_NTOHS(udp->uh_sport);
    u_short dport = SKIP_NTOHS(udp->uh_dport);

    if ((sport == SKIP_UDP_RECV_PORT) || (sport == SKIP_UDP_SEND_PORT) ||
	(dport == SKIP_UDP_RECV_PORT) || (dport == SKIP_UDP_SEND_PORT))
    {
      return SKIP_PROCESSED;
    }
  }
#endif
  
  /* Look up cache entry, cache if necessary */
  if ((ret = skip_lookup(&c, dir, interface, header, pkt, old)) <
      SKIP_PROCESSED)
    return ret;

  if (FILTER_CHECK(c->filter_before[dir], ip->ip_p))
  {
    skipcache_release(c);
    ipsp_stat.badfilter++;
    return SKIP_DISCARD;
  }

#if 0  /* De-/ and enskipping in one step... not supported any more */
  /* Gateway de- & enskipping on output? XXX: Should be tested */
  if ((dir == SKIP_OUTPUT) && (c->flags & SKIP_VALIDKEY) &&
      (c->deskip_mode[dir] == DESKIPMODE_DESKIP))
  {
    struct skipcache *c2;

    /* Look up cache entry, cache if necessary */
    if ((ret = skip_lookup(&c2, dir, interface, header, pkt, old)) <
	SKIP_PROCESSED)
      return ret;
    
    ret = skip_convert(c2, dir, pkt, old, new);  /* XXX */

    if (ret >= SKIP_PROCESSED)
    {
      ip = (struct ip *)BLKSTART(old);
      if (FILTER_CHECK(c2->filter_after[dir], ip->ip_p))
      {
	ipsp_stat.badfilter++;
	ret = SKIP_DISCARD;
      }
    }
  }
#endif

  /* Convert for delivery */
  if (ret >= SKIP_PROCESSED)
  {
    if ((ret = skip_convert(c, dir, interface, header, pkt, old, new)) >=
	SKIP_PROCESSED)
    {
      /* new memblk is valid if length is not zero (set in interface_getbuf) */
      if (new->len > 0)
	ip = (struct ip *)BLKSTART(new);
      if (FILTER_CHECK(c->filter_after[dir], ip->ip_p))
      {
	ipsp_stat.badfilter++;
	ret = SKIP_DISCARD;
      }
    }
  }

#ifdef DEBUG_SKIPC
  printf(">> skip_process: ip len after %d\n", SKIP_NTOHS(ip->ip_len));
#endif

  skipcache_release(c);

  return ret;
}
