/*

Modified for ENskip 12/1996, 01/1997 Robert Muchsel

Changed keyInit to support "espionage enabled" effective key sizes
smaller than 1024 bit.

-------------------------------------------------------------------------------

RC2 implementation from Peter Gutmann (cryptlib)

CRYPTLIB copyright follows:

P o r t a b l e  E n c r y p t i o n  L i b r a r y   Version 2.00

Copyright Peter Gutmann, Eric Young, and Colin Plumb  1992 - 1996

Author Contact Information:

Peter Gutmann, pgut001@cs.auckland.ac.nz

The cryptlib home page, http://www.cs.auckland.ac.nz/~pgut001/cryptlib.html,
contains more information on the latest cryptlib developments.


Usage Conditions
----------------

This software is distributed as copyrighted freeware, with copyrights on
individual encryption modules being held by the contributing authors.  You are
free to use the code in any way you want, with the following restrictions:

- If you make any changes to the code, you should send a copy of the changes to
  the author or authors to allow them to integrate the changes into the code.
  This is to allow a central consistent version to be maintained.

- If you use the library as part of a product, you should offer a copy to the
  authors of the library.  This is to let the authors know that their work is
  being usefully applied.  You should also give the authors credit in your
  software and/or documentation.  This is to let others know that the authors
  work is being usefully applied :-).

- Any commercial software you create with this code may not be merely a set or
  subset of the encryption library, with or without minor added functionality.
  In particular you can't sell the library (or any modified form of it) as
  "your" encryption product.  You can sell your own product which utilizes the
  encryption library, but you can't charge for the library itself or claim it
  as yours.  This is to stop people adding their own wrappers and selling it as
  "their" encryption product.
*/

/* #define TEST_RC2 */

/****************************************************************************
* RC2 Encryption Algorithm 
* Copyright Peter Gutmann 1996
****************************************************************************/

#ifndef TEST_RC2
#include "config.h"
#include "memblk.h"
#include "crypt.h"
#else
#include <stdio.h>
#include <string.h>
#endif

/* The size of the key in bytes and 16-bit words */

#define RC2_MAX_KEY_BITS	1024
#define RC2_KEY_SIZE		128
#define RC2_KEY_SIZE_WORDS	( RC2_KEY_SIZE / 2 )

/* The RC2 block size */

#define RC2_BLOCKSIZE		8

/* The RC2 key */

typedef struct RC2_KEY {
  unsigned int key[ RC2_KEY_SIZE_WORDS ];
} RC2_KEY;

#ifndef TEST_RC2
#define BYTE u_char
#define WORD u_short
#else
#define BYTE unsigned char
#define WORD unsigned short
#define MEMZERO(a,s) memset(a, 0, s)
#endif

#define mgetLWord(memPtr) \
	( ( WORD ) memPtr[ 0 ] ) | ( ( WORD ) memPtr[ 1 ] << 8 ); \
	memPtr += 2

#define mputLWord(memPtr,data)  \
	memPtr[ 0 ] = ( BYTE ) ( ( data ) & 0xFF ); \
	memPtr[ 1 ] = ( BYTE ) ( ( ( data ) >> 8 ) & 0xFF ); \
	memPtr += 2

#define MASK16( x )   ( ( x ) & 0xFFFFUL )

/* "It feels morally wrong to use a 32-bit processor and have only half the
	gates flapping" - Chris Wedgwood */

/* The following code uses unsigned int rather than WORD since many 32-bit
   compilers generate awful code when working with 16-bit data.  We know
   when a result will overflow 16 bits so we use ints and manually mask off
   the extra bits rather than have the compiler do it after every operation
   on a WORD */

/* ROTATE_LEFT/RIGHT rotates x by n bits */

#define ROTATE_LEFT(x,n)	( ( ( x ) << n ) | ( ( x ) >> ( 16 - n ) ) )
#define ROTATE_RIGHT(x,n)	( ( ( x ) >> n ) | ( ( x ) << ( 16 - n ) ) )

/* The basic en/decryption operations */

#define enc(A,B,C,D,S,round,rotAmount) \
	MASK16( ROTATE_LEFT( MASK16( A + ( B & ~C ) + ( D & C ) + S[ round ] ), rotAmount ) )

#define dec(A,B,C,D,S,round,rotAmount) \
	MASK16( MASK16( ROTATE_RIGHT( A, rotAmount ) ) - ( B & ~D ) - ( C & D ) - S[ round ] )

/* One round of en/decryption */

#define encRound(A,B,C,D,S,round) \
	A = enc( A, B, D, C, S, ( round * 4 ) + 0, 1 ); \
	B = enc( B, C, A, D, S, ( round * 4 ) + 1, 2 ); \
	C = enc( C, D, B, A, S, ( round * 4 ) + 2, 3 ); \
	D = enc( D, A, C, B, S, ( round * 4 ) + 3, 5 )

#define decRound(A,B,C,D,S,round) \
	D = dec( D, A, B, C, S, ( round * 4 ) + 3, 5 ); \
	C = dec( C, D, A, B, S, ( round * 4 ) + 2, 3 ); \
	B = dec( B, C, D, A, S, ( round * 4 ) + 1, 2 ); \
	A = dec( A, B, C, D, S, ( round * 4 ) + 0, 1 )

/* The addition/subtraction of the S-boxes which occurs on the 4th and 10th
   rounds.  The addition will overflow the 16-bit limit, but this isn't a
   problem since the next round of en/decryption gets things back down to 16
   bits (unfortunately this doesn't work for subtraction) */

#define addSboxes(A,B,C,D,S) \
	A += S[ D & (RC2_KEY_SIZE_WORDS - 1) ]; \
	B += S[ A & (RC2_KEY_SIZE_WORDS - 1) ]; \
	C += S[ B & (RC2_KEY_SIZE_WORDS - 1) ]; \
	D += S[ C & (RC2_KEY_SIZE_WORDS - 1) ]

#define subSboxes(A,B,C,D,S) \
	D -= S[ C & (RC2_KEY_SIZE_WORDS - 1) ]; \
	D = MASK16( D ); \
	C -= S[ B & (RC2_KEY_SIZE_WORDS - 1) ]; \
	C = MASK16( C ); \
	B -= S[ A & (RC2_KEY_SIZE_WORDS - 1) ]; \
	B = MASK16( B ); \
	A -= S[ D & (RC2_KEY_SIZE_WORDS - 1) ]; \
	A = MASK16( A )

/* The permutation table used for the RC2 key setup */

static BYTE sBox[] = {
	0xD9, 0x78, 0xF9, 0xC4, 0x19, 0xDD, 0xB5, 0xED,
	0x28, 0xE9, 0xFD, 0x79, 0x4A, 0xA0, 0xD8, 0x9D,
	0xC6, 0x7E, 0x37, 0x83, 0x2B, 0x76, 0x53, 0x8E,
	0x62, 0x4C, 0x64, 0x88, 0x44, 0x8B, 0xFB, 0xA2,
	0x17, 0x9A, 0x59, 0xF5, 0x87, 0xB3, 0x4F, 0x13,
	0x61, 0x45, 0x6D, 0x8D, 0x09, 0x81, 0x7D, 0x32,
	0xBD, 0x8F, 0x40, 0xEB, 0x86, 0xB7, 0x7B, 0x0B,
	0xF0, 0x95, 0x21, 0x22, 0x5C, 0x6B, 0x4E, 0x82,
	0x54, 0xD6, 0x65, 0x93, 0xCE, 0x60, 0xB2, 0x1C,
	0x73, 0x56, 0xC0, 0x14, 0xA7, 0x8C, 0xF1, 0xDC,
	0x12, 0x75, 0xCA, 0x1F, 0x3B, 0xBE, 0xE4, 0xD1,
	0x42, 0x3D, 0xD4, 0x30, 0xA3, 0x3C, 0xB6, 0x26,
	0x6F, 0xBF, 0x0E, 0xDA, 0x46, 0x69, 0x07, 0x57,
	0x27, 0xF2, 0x1D, 0x9B, 0xBC, 0x94, 0x43, 0x03,
	0xF8, 0x11, 0xC7, 0xF6, 0x90, 0xEF, 0x3E, 0xE7,
	0x06, 0xC3, 0xD5, 0x2F, 0xC8, 0x66, 0x1E, 0xD7,
	0x08, 0xE8, 0xEA, 0xDE, 0x80, 0x52, 0xEE, 0xF7,
	0x84, 0xAA, 0x72, 0xAC, 0x35, 0x4D, 0x6A, 0x2A,
	0x96, 0x1A, 0xD2, 0x71, 0x5A, 0x15, 0x49, 0x74,
	0x4B, 0x9F, 0xD0, 0x5E, 0x04, 0x18, 0xA4, 0xEC,
	0xC2, 0xE0, 0x41, 0x6E, 0x0F, 0x51, 0xCB, 0xCC,
	0x24, 0x91, 0xAF, 0x50, 0xA1, 0xF4, 0x70, 0x39,
	0x99, 0x7C, 0x3A, 0x85, 0x23, 0xB8, 0xB4, 0x7A,
	0xFC, 0x02, 0x36, 0x5B, 0x25, 0x55, 0x97, 0x31,
	0x2D, 0x5D, 0xFA, 0x98, 0xE3, 0x8A, 0x92, 0xAE,
	0x05, 0xDF, 0x29, 0x10, 0x67, 0x6C, 0xBA, 0xC9,
	0xD3, 0x00, 0xE6, 0xCF, 0xE1, 0x9E, 0xA8, 0x2C,
	0x63, 0x16, 0x01, 0x3F, 0x58, 0xE2, 0x89, 0xA9,
	0x0D, 0x38, 0x34, 0x1B, 0xAB, 0x33, 0xFF, 0xB0,
	0xBB, 0x48, 0x0C, 0x5F, 0xB9, 0xB1, 0xCD, 0x2E,
	0xC5, 0xF3, 0xDB, 0x47, 0xE5, 0xA5, 0x9C, 0x77,
	0x0A, 0xA6, 0x20, 0x68, 0xFE, 0x7F, 0xC1, 0xAD
	};

/* Perform an RC2 key schedule */

static void rc2keyInit( RC2_KEY *rc2key, BYTE *userKey, int length, int bits )
{
#ifndef SMALL_KERNEL_STACK
	BYTE keyTemp[ RC2_KEY_SIZE ], *keyTempPtr = keyTemp, x;
#else /* SMALL_KERNEL_STACK */
	BYTE *keyTemp, *keyTempPtr, x;
#endif /* SMALL_KERNEL_STACK */
	int i;

#ifdef SMALL_KERNEL_STACK
	if ((keyTempPtr = keyTemp = KALLOC(RC2_KEY_SIZE)) == NULL)
	  return;
#endif /* SMALL_KERNEL_STACK */

	/* Expand the secret key to 128 bytes by taking the sum of the first
	   and last bytes of the current key and appending the S-box entry this
	   corresponds to to the current key */
	MEMCPY( keyTemp, userKey, length );
	for( i = length; i < RC2_KEY_SIZE; i++ )
		keyTemp[ i ] = sBox[ ( keyTemp[ i - length ] + \
							   keyTemp[ i - 1 ] ) & 0xFF ];

	/* Reduce effective key size to "bits" (1024 -- no reduction) */
        length = ( bits + 7 ) >> 3;
        i = RC2_KEY_SIZE - length;
        x = sBox [ keyTemp [ i ] & ( 0xFF >> ( -bits & 0x07 ) ) ];
        keyTemp [ i ] = x;

        while ( i-- ) 
		{
                x = sBox [ x ^ keyTemp [ i + length ] ];
                keyTemp [ i ] = x;
        	}

	/* Copy the scheduled key to the RC2 key structure and erase it */
	for( i = 0; i < RC2_KEY_SIZE_WORDS; i++ )
		{
		rc2key->key[ i ] = mgetLWord( keyTempPtr );
		}
	MEMZERO(keyTemp, sizeof(keyTemp));

#ifdef SMALL_KERNEL_STACK
	KFREE(keyTemp, RC2_KEY_SIZE);
#endif /* SMALL_KERNEL_STACK */
}

/* Encrypt a block of data with RC2 */

static void rc2encrypt( RC2_KEY *rc2key, BYTE *inbuf, BYTE *outbuf )
{
	unsigned int word0, word1, word2, word3;
	unsigned int *key = rc2key->key;

	/* Extract the data from the buffer */
	word0 = mgetLWord( inbuf );
	word1 = mgetLWord( inbuf );
	word2 = mgetLWord( inbuf );
	word3 = mgetLWord( inbuf );

	/* Perform 16 rounds of encryption */
	encRound( word0, word1, word2, word3, key, 0 );
	encRound( word0, word1, word2, word3, key, 1 );
	encRound( word0, word1, word2, word3, key, 2 );
	encRound( word0, word1, word2, word3, key, 3 );
	encRound( word0, word1, word2, word3, key, 4 );
	addSboxes( word0, word1, word2, word3, key );
	encRound( word0, word1, word2, word3, key, 5 );
	encRound( word0, word1, word2, word3, key, 6 );
	encRound( word0, word1, word2, word3, key, 7 );
	encRound( word0, word1, word2, word3, key, 8 );
	encRound( word0, word1, word2, word3, key, 9 );
	encRound( word0, word1, word2, word3, key, 10 );
	addSboxes( word0, word1, word2, word3, key );
	encRound( word0, word1, word2, word3, key, 11 );
	encRound( word0, word1, word2, word3, key, 12 );
	encRound( word0, word1, word2, word3, key, 13 );
	encRound( word0, word1, word2, word3, key, 14 );
	encRound( word0, word1, word2, word3, key, 15 );

	/* Deposit the data back in the buffer */
	mputLWord( outbuf, word0 );
	mputLWord( outbuf, word1 );
	mputLWord( outbuf, word2 );
	mputLWord( outbuf, word3 );
}

/* Decrypt a block of data with RC2 */

static void rc2decrypt( RC2_KEY *rc2key, BYTE *inbuf, BYTE *outbuf )
{
	unsigned int word0, word1, word2, word3;
	unsigned int *key = rc2key->key;

	/* Extract the data from the buffer */
	word0 = mgetLWord( inbuf );
	word1 = mgetLWord( inbuf );
	word2 = mgetLWord( inbuf );
	word3 = mgetLWord( inbuf );

	/* Perform 16 rounds of decryption */
	decRound( word0, word1, word2, word3, key, 15 );
	decRound( word0, word1, word2, word3, key, 14 );
	decRound( word0, word1, word2, word3, key, 13 );
	decRound( word0, word1, word2, word3, key, 12 );
	decRound( word0, word1, word2, word3, key, 11 );
	subSboxes( word0, word1, word2, word3, key );
	decRound( word0, word1, word2, word3, key, 10 );
	decRound( word0, word1, word2, word3, key, 9 );
	decRound( word0, word1, word2, word3, key, 8 );
	decRound( word0, word1, word2, word3, key, 7 );
	decRound( word0, word1, word2, word3, key, 6 );
	decRound( word0, word1, word2, word3, key, 5 );
	subSboxes( word0, word1, word2, word3, key );
	decRound( word0, word1, word2, word3, key, 4 );
	decRound( word0, word1, word2, word3, key, 3 );
	decRound( word0, word1, word2, word3, key, 2 );
	decRound( word0, word1, word2, word3, key, 1 );
	decRound( word0, word1, word2, word3, key, 0 );

	/* Deposit the data back in the buffer */
	mputLWord( outbuf, word0 );
	mputLWord( outbuf, word1 );
	mputLWord( outbuf, word2 );
	mputLWord( outbuf, word3 );
}

#ifdef TEST_RC2

/* Test routines */

void main( void )
	{
	BYTE key1[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	BYTE plain1[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	BYTE cipher1[] = { 0x1C, 0x19, 0x8A, 0x83, 0x8D, 0xF0, 0x28, 0xB7 };
	BYTE key2[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 };
	BYTE plain2[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	BYTE cipher2[] = { 0x21, 0x82, 0x9C, 0x78, 0xA9, 0xF9, 0xC0, 0x74 };
	BYTE key3[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	BYTE plain3[] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
	BYTE cipher3[] = { 0x13, 0xDB, 0x35, 0x17, 0xD3, 0x21, 0x86, 0x9E };
	BYTE key4[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F };
	BYTE plain4[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	BYTE cipher4[] = { 0x50, 0xDC, 0x01, 0x62, 0xBD, 0x75, 0x7F, 0x31 };
	BYTE buffer[ 8 ];
	RC2_KEY rc2key;

	MEMCPY( buffer, plain1, 8 );
	rc2keyInit( &rc2key, key1, 16, RC2_MAX_KEY_BITS );
	rc2encrypt( &rc2key, buffer, buffer );
	if( memcmp( buffer, cipher1, 8 ) )
		puts( "Bang" );
	rc2decrypt( &rc2key, buffer, buffer );
	MEMCPY( buffer, plain2, 8 );
	rc2keyInit( &rc2key, key2, 16, RC2_MAX_KEY_BITS );
	rc2encrypt( &rc2key, buffer, buffer );
	if( memcmp( buffer, cipher2, 8 ) )
		puts( "Bang" );
	rc2decrypt( &rc2key, buffer, buffer );
	MEMCPY( buffer, plain3, 8 );
	rc2keyInit( &rc2key, key3, 16, RC2_MAX_KEY_BITS );
	rc2encrypt( &rc2key, buffer, buffer );
	if( memcmp( buffer, cipher3, 8 ) )
		puts( "Bang" );
	rc2decrypt( &rc2key, buffer, buffer );
	MEMCPY( buffer, plain4, 8 );
	rc2keyInit( &rc2key, key4, 16, RC2_MAX_KEY_BITS );
	rc2encrypt( &rc2key, buffer, buffer );
	if( memcmp( buffer, cipher4, 8 ) )
		puts( "Bang" );
	rc2decrypt( &rc2key, buffer, buffer );
	}
#else /* TEST_RC2 */

/*************************************************************************
 * interface for SKIP                                                    *
 *************************************************************************/

/* Espionage enabled version */
#define KEYBITS_40	40
/* The bsafe library reportedly sets the following value to 128 and we want 
   to be interoperable (I cannot think of any other reason why anyone would 
   use RC2). The cipher might be better with the full 1024 bits here. */
#define KEYBITS_128	128

/* Pick a key size - we use 64 bit to be compatible */
#define KEYLEN_40	8
#define KEYLEN_128	16	

struct staterc2_40
{
  int alg;
  u_char key[KEYLEN_40];
  RC2_KEY schedule;
};

struct staterc2_128
{
  int alg;
  u_char key[KEYLEN_128];
  RC2_KEY schedule;
};

static int cryptrc2_init(void)
{
  return 0;
}

static int cryptrc2_exit(void)
{
  return 0;
}

static int cryptrc2_40_statelen(void)
{
  return sizeof(struct staterc2_40);
}

static int cryptrc2_128_statelen(void)
{
  return sizeof(struct staterc2_128);
}

static int cryptrc2_blocklen(void)
{
  return RC2_BLOCKSIZE;
}

static int cryptrc2_40_keylen(void)
{
  return KEYLEN_40;
}

static int cryptrc2_128_keylen(void)
{
  return KEYLEN_128;
}

static int cryptrc2_MIlen(void)
{
  return RC2_BLOCKSIZE;
}

static int cryptrc2_40_setkey(u_char *key, u_char *stateptr)
{
  struct staterc2_40 *state = (struct staterc2_40 *)stateptr;

  rc2keyInit(&(state->schedule), key, KEYLEN_40, KEYBITS_40);
  return 0;
}

static int cryptrc2_128_setkey(u_char *key, u_char *stateptr)
{
  struct staterc2_128 *state = (struct staterc2_128 *)stateptr;

  rc2keyInit(&(state->schedule), key, KEYLEN_128, KEYBITS_128);
  return 0;
}

static int cryptrc2_40_crypt(u_char *in, u_char *out, int len, u_char *stateptr)
{
  struct staterc2_40 *state = (struct staterc2_40 *)stateptr;

  rc2encrypt(&(state->schedule), in, out);
  return 0;
}

static int cryptrc2_128_crypt(u_char *in, u_char *out, int len, u_char *stateptr)
{
  struct staterc2_128 *state = (struct staterc2_128 *)stateptr;

  rc2encrypt(&(state->schedule), in, out);
  return 0;
}

static int cryptrc2_40_decrypt(u_char *in, u_char *out, int len, u_char *stateptr)
{
  struct staterc2_40 *state = (struct staterc2_40 *)stateptr;
  rc2decrypt(&(state->schedule), in, out);
  return 0;
}

static int cryptrc2_128_decrypt(u_char *in, u_char *out, int len, u_char *stateptr)
{
  struct staterc2_128 *state = (struct staterc2_128 *)stateptr;
  rc2decrypt(&(state->schedule), in, out);
  return 0;
}

struct crypt_algorithm cryptrc2_40 =
{
  cryptrc2_init, cryptrc2_exit,
  cryptrc2_40_statelen, cryptrc2_40_statelen, 
  cryptrc2_blocklen, cryptrc2_40_keylen, cryptrc2_MIlen,
  cryptrc2_40_setkey, cryptrc2_40_setkey,
  cryptrc2_40_crypt, cryptrc2_40_decrypt,
  NULL
};

struct crypt_algorithm cryptrc2_128 =
{
  cryptrc2_init, cryptrc2_exit,
  cryptrc2_128_statelen, cryptrc2_128_statelen, 
  cryptrc2_blocklen, cryptrc2_128_keylen, cryptrc2_MIlen,
  cryptrc2_128_setkey, cryptrc2_128_setkey,
  cryptrc2_128_crypt, cryptrc2_128_decrypt,
  NULL
};

#endif /* !TEST_RC2 */
