/*
 * Copyright (C) 1995-1997 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */

/* #define DEBUG_ALLOC */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "skip_defs.h"
#include "memblk.h"
#include "sign.h"
#include "crypt.h"
#include "dynamic.h"
#include "skipcache.h"
#include "random.h"
#include "inet_aton.h"

#ifdef __GNUC__
#ident "$Id: parse.c,v 1.8 1996/06/07 11:59:08 skip Exp $"
#else
static char rcsid[] = "$Id: parse.c,v 1.8 1996/06/07 11:59:08 skip Exp $";
#endif

#define PARSE_COMMENT	"##\n" \
			"## SKIP Daemon configuration file\n" \
			"##\n\n"
#undef PARSE_COMMENT

#define ENTRYSEPARATOR	\
"#########################################################################\n\n"

#define IGNORE			"ignore"

#define IS_EOL(x)		(((x) == '\n') || ((x) == '\r'))
#define IS_BLANK(x)		(((x) == ' ') || ((x) == '\t'))
#define IS_SEPARATOR(x)		(IS_BLANK(x) || ((x) == ':') || ((x) == ';') || ((x) == ','))

#define ALIGN4(len) (((len) + 3) & ~3)

static skip_id_t _defaultid = { { 0, 0 }, 
			       { { 0xff, 0xff, 0xff, 0xff,
				   0xff, 0xff, 0xff, 0xff,
				   0xff, 0xff, 0xff, 0xff,
				   0xff, 0xff, 0xff, 0xff },
				 { 0xff, 0xff, 0xff, 0xff,
				   0xff, 0xff, 0xff, 0xff,
				   0xff, 0xff, 0xff, 0xff,
				   0xff, 0xff, 0xff, 0xff } } };
static skip_id_t defaultid;

static skip_id_t _ignoreid;
static skip_id_t ignoreid;

static u_char data[16384], *dataptr;

struct parseentry
{
  char *keyword;
  int (*req)(struct skipcache *cnf, char *val);
  int (*add)(struct skipcache *cnf, char *optname, FILE *f);
};

static u_char defaultdata[16384];

static struct skipcache _defaultconfig =
{
  { { 0, 0 } },                                        /* id */
  NULL, NULL,                                          /* next, prev */
  60, 60, 0,                                           /* ttl,maxttl,keyttl */
  0,                                                   /* ref */
  0,		                                       /* flags */

  { 0, 0 }, { 0, 0 },                                  /* src/ dst ip */
  NULL,

  { 0, 0 },                                            /* DH secret key */
  { 0, 0 },                                            /* lookup hosts */
  { { 0, { 0, 0 }}, { 0, { 0, 0 }}, { 0, { 0, 0 }}},   /* Kijn */

  { FILTER_NONE, FILTER_NONE },                        /* filter_before */
  { FILTER_NONE, FILTER_NONE },                        /* filter_after */

  { ENSKIPMODE_ALL, ENSKIPMODE_ALL },                  /* enskip_mode */
  { ENSKIPMKID_AUTO, ENSKIPMKID_AUTO },                /* enskip_mkid */
  1, 1, 1, 0,                                          /* alg */

  0, 512*1024,					       /* Kp_bytes, maxbytes */
  0, 30,					       /* Kp_ttl, maxttl */

  { 0, 0 }, { 0, 0 },                                  /* enskip A_Kp, E_Kp */
  { 0, { 0, 0 }},                                      /* enskip_Kp_Kijn */
  { 0, 0 },                                            /* enskip_MI */

  { DESKIPMODE_DESKIP, DESKIPMODE_DESKIP },            /* deskip_mode */
  { 0x00, 0x00 },                                      /* deskip_policy */
  0, 0,                                                /* Kij,Kp alg */

  { 0, 0 }, { 0, 0 },                                  /* deskip A_Kp, E_Kp */
  { 0, { 0, 0 }},                                      /* deskip_Kp_Kijn */

  { 0, 0 }, { 0, 0 }, { 0, 0 },                        /* state offsets */

  { 0, 0 },                                            /* q */

  defaultdata, 0,                                      /* data ptr, len */
};
static struct skipcache defaultconfig;

static int linecnt;

static char *cryptalgs[] = { "NONE", "DES", "3DES-3", "IDEA", "RC2-40", "RC4-40", "RC4-128", "3DES-2", "SIMPLE", "RC2-128", "SAFER-SK128" };
static char *said[] = { "NONE", "CRYPT", "AUTH", "SEQ", "COMP" };
static char *signalgs[] = { "NONE", "MD5" };
static char *filter[] = { "NONE", "IP", "SKIP" };
static char *enskipmode[] = { "NONE", "TCP", "UDP", "ICMP", "OTHER", "ALL" };
static char *enskipmkid[] = { "AUTO", "YES", "NO" };
static char *deskipmode[] = { "NONE", "DESKIP" };
static char *flags[] = { "NONE", "NO_KEY", "TUNNEL" , "VALID_KEY" };

static int allocdata(struct skipcache *cnf, skip_dynamic_t *d, u_char **buf, int len)
{
  int result = 0;

  if (len)
  {
    if (buf)
      *buf = dataptr;
    MEMZERO(dataptr, ALIGN4(len));
    result = dataptr - cnf->data;
    dataptr += ALIGN4(len);
  }

  if (d)
  {
    d->offset = result;
    d->len = len;
  }

  return result;
}

static int scannicehex(struct skipcache *cnf, skip_dynamic_t *d, char *val)
{
  int result = 0, used;
  u_char *buf;
  int l;

  val += 2;
  l = strlen(val);
  if (l & 1)
  {
    fprintf(stderr, "skipd: malformed hex number, length is odd\n");
    l = 0;
  }
  l /= 2;

  allocdata(cnf, d, &buf, l);

  used = id_scanhex(val, buf, l);

  if (used <= 0)
  {
    /* XXX: we do not free allocated data here */
    d->offset = 0;
    d->len = 0;
    l = 0;
    result = -1;
  }

  d->len = l;

  return result;
}

static int idx(char *str, char **arr, int entries)
{
  int walk;

  for (walk = 0; walk < entries; walk++)
  {
    if (arr[walk])
    {
      if (strcmp(str, arr[walk]) == 0)
        return walk;
    }
  }
  return -1;
}

static char *separator(char *val)
{
  char *ret = NULL, *tmp = val;

  while (tmp && *tmp)
  {
    if (IS_SEPARATOR(*tmp))
    {
      *tmp = '\0';
      ret = tmp;
    }
    else
    {
      if (ret)
      {
	ret = tmp;
	break;
      }
    }

    tmp++;
  }

  return (ret && *ret) ? ret : NULL;
}

static int dynamic_alloc(struct skipcache *cnf)
{
  int maxdslen = crypt_decryptstatelen(CRYPTALG_MAX);
  int maxkeylen = crypt_keylen(CRYPTALG_MAX);
  int maxblocklen = crypt_blocklen(CRYPTALG_MAX);
  int maxdsblocklen = crypt_decryptstatelen(CRYPTALG_MAXBLOCK);

  /* Kij not set (no manual keying) */
  if (cnf->sharedkey.len == 0)
  {
#ifdef DEBUG_ALLOC
    fprintf(stderr, ">> dynamic_alloc: allocating %d for shared key\n", SKIP_KIJ_MAXLEN);
#endif
    allocdata(cnf, &cnf->sharedkey, NULL, SKIP_KIJ_MAXLEN);
  }
#ifdef DEBUG_ALLOC
  else
    fprintf(stderr, ">> dynamic_alloc: no space allocated for shared key\n");
#endif

  /* Kijn */
  allocdata(cnf, &cnf->Kijn[0].key, NULL, maxkeylen);
  allocdata(cnf, &cnf->Kijn[1].key, NULL, maxkeylen);
  allocdata(cnf, &cnf->Kijn[2].key, NULL, maxkeylen);
  cnf->Kijn[0].n = cnf->Kijn[1].n = cnf->Kijn[2].n = 0;

  /* deskip */
  allocdata(cnf, &cnf->payload_decryptstate, NULL, maxdslen);
  allocdata(cnf, &cnf->deskip_A_Kp, NULL, SKIP_KP_LEN);
  allocdata(cnf, &cnf->deskip_E_Kp, NULL, SKIP_KP_LEN);
  allocdata(cnf, &cnf->deskip_Kp_Kijn.key, NULL, maxkeylen + maxblocklen - 1);
  cnf->deskip_Kp_Kijn.n = 0;
  allocdata(cnf, &cnf->Kp_decryptstate, NULL, maxdsblocklen);

  if (cnf->enskip_Kij_alg)
  {
    /* do we need an enskip_Kp_Kijn */
    if (cnf->enskip_Crypt_alg || cnf->enskip_MAC_alg)
    {
      int keylen = 0;
      int blocklen = crypt_blocklen(cnf->enskip_Kij_alg);
      int padlen;
      
      if (cnf->enskip_Crypt_alg)
	keylen = crypt_keylen(cnf->enskip_Crypt_alg);
      if ((cnf->enskip_MAC_alg) && (SIGN_KEYLEN > keylen))
	keylen = SIGN_KEYLEN;

      if ((padlen = keylen % blocklen))
	padlen = blocklen - padlen;

      cnf->enskip_Kp_Kijn.n = 0;
      allocdata(cnf, &cnf->enskip_Kp_Kijn.key, NULL, keylen + padlen);
    }
    /* esp stuff */
    if (cnf->enskip_Crypt_alg)
    {
      int MI_len = crypt_MIlen(cnf->enskip_Crypt_alg);
      int keylen = crypt_keylen(cnf->enskip_Crypt_alg);
      int estatelen = crypt_encryptstatelen(cnf->enskip_Crypt_alg);
      u_char *buf;

      allocdata(cnf, &cnf->enskip_MI, &buf, MI_len); /* XXX should check MI_len for -1 XXX */
      MEMZERO(buf, MI_len);
      allocdata(cnf, &cnf->encryptstate, NULL, estatelen);
      allocdata(cnf, &cnf->enskip_E_Kp, NULL, keylen);
    }
    /* ah stuff */
    if (cnf->enskip_MAC_alg)
    {
      allocdata(cnf, &cnf->enskip_A_Kp, NULL, SIGN_KEYLEN);
    }
  }

  return 0;
}


static int outfilterbefore_req(struct skipcache *cnf, char *val)
{
  int f = idx(val, filter, sizeof(filter) / sizeof(filter[0]));

  if (f < 0)
  {
    fprintf(stderr, "skipd: unknown filter '%s'.\n", val);
    return -1;
  }
  else
  {
    cnf->filter_before[SKIP_OUTPUT] = f;
    return 0;
  }
}
static int outfilterbefore_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %s\n", optname, filter[cnf->filter_before[SKIP_OUTPUT]]);
  return 0;
}

static int infilterbefore_req(struct skipcache *cnf, char *val)
{
  int f = idx(val, filter, sizeof(filter) / sizeof(filter[0]));

  if (f < 0)
  {
    fprintf(stderr, "skipd: unknown filter '%s'.\n", val);
    return -1;
  }
  else
  {
    cnf->filter_before[SKIP_INPUT] = f;
    return 0;
  }
}
static int infilterbefore_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %s\n", optname, filter[cnf->filter_before[SKIP_INPUT]]);
  return 0;
}

static int outfilterafter_req(struct skipcache *cnf, char *val)
{
  int f = idx(val, filter, sizeof(filter) / sizeof(filter[0]));

  if (f < 0)
  {
    fprintf(stderr, "skipd: unknown filter '%s'.\n", val);
    return -1;
  }
  else
  {
    cnf->filter_after[SKIP_OUTPUT] = f;
    return 0;
  }
}
static int outfilterafter_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %s\n", optname, filter[cnf->filter_after[SKIP_OUTPUT]]);
  return 0;
}

static int infilterafter_req(struct skipcache *cnf, char *val)
{
  int f = idx(val, filter, sizeof(filter) / sizeof(filter[0]));

  if (f < 0)
  {
    fprintf(stderr, "skipd: unknown filter '%s'.\n", val);
    return -1;
  }
  else
  {
    cnf->filter_after[SKIP_INPUT] = f;
    return 0;
  }
}
static int infilterafter_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %s\n", optname, filter[cnf->filter_after[SKIP_INPUT]]);
  return 0;
}

static int parseenskipmode(char *val)
{
  char *walk, *next;
  int mode;

  mode = ENSKIPMODE_NONE;
  for (walk = val, next = separator(walk); walk; walk = next, next = separator(walk))
  {
    switch (idx(walk, enskipmode, sizeof(enskipmode) / sizeof(enskipmode[0])))
    {
      case -1:
        fprintf(stderr, "skipd: unknown enskip mode '%s'.\n", val);
	break;
      case 0:
	/* NONE */
	break;
      case 1:
	mode |= ENSKIPMODE_TCP;
	break;
      case 2:
	mode |= ENSKIPMODE_UDP;
	break;
      case 3:
	mode |= ENSKIPMODE_ICMP;
	break;
      case 4:
	mode |= ENSKIPMODE_OTHER;
	break;
      case 5:
	mode |= ENSKIPMODE_ALL;
	break;
    }
  }
  return mode;
}

static int saveenskipmode(int mode, char *optname, FILE *f)
{
  fprintf(f, "  %s = ", optname);

  if (mode == ENSKIPMODE_ALL)
    fprintf(f, "ALL");
  else if (mode == ENSKIPMODE_NONE)
    fprintf(f, "NONE");
  else
  {
    if (mode & ENSKIPMODE_TCP)
      fprintf(f, "TCP ");
    if (mode & ENSKIPMODE_UDP)
      fprintf(f, "UDP ");
    if (mode & ENSKIPMODE_ICMP)
      fprintf(f, "ICMP ");
    if (mode & ENSKIPMODE_OTHER)
      fprintf(f, "OTHER ");
  }
  fprintf(f, "\n");
  return 0;
}

static int outenskipmode_req(struct skipcache *cnf, char *val)
{
  cnf->enskip_mode[SKIP_OUTPUT] = parseenskipmode(val);
  return 0;
}
static int outenskipmode_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return saveenskipmode(cnf->enskip_mode[SKIP_OUTPUT], optname, f);
}
 
static int inenskipmode_req(struct skipcache *cnf, char *val)
{
  cnf->enskip_mode[SKIP_INPUT] = parseenskipmode(val);
  return 0;
}
static int inenskipmode_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return saveenskipmode(cnf->enskip_mode[SKIP_INPUT], optname, f);
}

static int parseenskipmkid(char *val)
{
  int mkid = idx(val, enskipmkid, sizeof(enskipmkid) / sizeof(enskipmkid[0])); 

  if (mkid < 0)
  {
    fprintf(stderr, "skipd: unknown mkid mode '%s'.\n", val);
    return -1;
  }
  else
  {
    switch (mkid)
    {
      case 0:
        mkid = ENSKIPMKID_AUTO;
        break;
      case 1:
        mkid = ENSKIPMKID_YES;
        break;
      case 2:
        mkid = ENSKIPMKID_NO;
        break;
    }
  }

  return mkid;
}

static int saveenskipmkid(int mkid, char *optname, FILE *f)
{
  fprintf(f, "  %s = ", optname);

  if (mkid == ENSKIPMKID_AUTO)
    fprintf(f, "AUTO");
  else if (mkid == ENSKIPMKID_YES)
    fprintf(f, "YES");
  else if (mkid == ENSKIPMKID_NO)
    fprintf(f, "NO");

  fprintf(f, "\n");
  return 0;
}

static int srcenskipmkid_req(struct skipcache *cnf, char *val)
{
  cnf->enskip_mkid[0] = parseenskipmkid(val);
  return 0;
}
static int srcenskipmkid_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return saveenskipmkid(cnf->enskip_mkid[0], optname, f);
}
 
static int dstenskipmkid_req(struct skipcache *cnf, char *val)
{
  cnf->enskip_mkid[1] = parseenskipmkid(val);
  return 0;
}
static int dstenskipmkid_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return saveenskipmkid(cnf->enskip_mkid[1], optname, f);
}

static u_long parsesaid(char *val)
{
  char *walk, *next;
  u_char res;

  res = 0;

  for (walk = val, next = separator(walk); walk; walk = next, next = separator(walk))
  {
    switch (idx(walk, said, sizeof(said) / sizeof(said[0])))
    {
      case -1:
        fprintf(stderr, "skipd: unknown mode '%s'.\n", val);
	break;
      case 0:
	break;
      case 1:
	res |= SAID_CRYPT;
	break;
      case 2:
	res |= SAID_AUTH;
	break;
      case 3:
	res |= SAID_SEQ;
	break;
      case 4:
	res |= SAID_COMP;
	break;
    }
  }
  return res;
}

static int kpalg_req(struct skipcache *cnf, char *val)
{
  int alg = idx(val, cryptalgs, sizeof(cryptalgs) / sizeof(cryptalgs[0])); 

  if (alg < 0)
  {
    fprintf(stderr, "skipd: unknown Crypt algorithm '%s'.\n", val);
    return -1;
  }
  else
  {
    switch (alg)
    {
      case 1:
        alg = CRYPTALG_DES;
        break;
      case 2:
        alg = CRYPTALG_3DES_3;
        break;
      case 3:
        alg = CRYPTALG_IDEA;
	break;
      case 4:
	alg = CRYPTALG_RC2_40;
        break;
      case 5:
	alg = CRYPTALG_RC4_40;
        break;
      case 6:
	alg = CRYPTALG_RC4_128;
	break;
      case 7:
	alg = CRYPTALG_3DES_2;
	break;
      case 8:
	alg = CRYPTALG_SIMPLE;
	break;
      case 9:
	alg = CRYPTALG_RC2_128;
	break;
      case 10:
	alg = CRYPTALG_SAFER_SK128;
	break;
    }

    cnf->enskip_Crypt_alg = alg;
    return 0;
  }
}
static int kpalg_add(struct skipcache *cnf, char *optname, FILE *f)
{
  char *str = "NONE";

  switch (cnf->enskip_Crypt_alg)
  {
    case CRYPTALG_DES:
      str = "DES";
      break;
    case CRYPTALG_3DES_2:
      str = "3DES-2";
      break;
    case CRYPTALG_3DES_3:
      str = "3DES-3";
      break;
    case CRYPTALG_IDEA:
      str = "IDEA";
      break;
    case CRYPTALG_RC4_40:
      str = "RC4-40";
      break;
    case CRYPTALG_RC4_128:
      str = "RC4-128";
      break;
    case CRYPTALG_SIMPLE:
      str = "SIMPLE";
      break;
    case CRYPTALG_RC2_40:
      str = "RC2-40";
      break;
    case CRYPTALG_RC2_128:
      str = "RC2-128";
      break;
    case CRYPTALG_SAFER_SK128:
      str = "SAFER-SK128";
      break;
  }

  fprintf(f, "  %s = %s\n", optname, str);
  return 0;
}

static int kijalg_req(struct skipcache *cnf, char *val)
{
  int alg = idx(val, cryptalgs, sizeof(cryptalgs) / sizeof(cryptalgs[0])); 

  if (alg <= 0)
  {
    fprintf(stderr, "skipd: unknown Kij algorithm '%s'.\n", val);
    return -1;
  }
  else
  {
    switch (alg)
    {
      case 1:
        alg = CRYPTALG_DES;
        break;
      case 2:
        alg = CRYPTALG_3DES_3;
        break;
      case 3:
        alg = CRYPTALG_IDEA;
	break;
      case 4:
	alg = CRYPTALG_RC2_40;
	break;
      case 5:
	alg = CRYPTALG_RC4_40;
	break;
      case 6:
	alg = CRYPTALG_RC4_128;
	break;
      case 7:
	alg = CRYPTALG_3DES_2; 
	break;
      case 8:
	alg = CRYPTALG_SIMPLE;
	break;
      case 9:
	alg = CRYPTALG_RC2_128;
	break;
      case 10:
	alg = CRYPTALG_SAFER_SK128;
	break;
    }

    if (crypt_blocklen(alg) != 1)
      cnf->enskip_Kij_alg = alg;
    else
    {
      fprintf(stderr, "skipd: stream ciphers cannot be used to encrypt packet keys\n");
      return -1;
    }
  }

  return 0;
}
static int kijalg_add(struct skipcache *cnf, char *optname, FILE *f)
{
  char *str = "NONE";

  switch (cnf->enskip_Kij_alg)
  {
    case CRYPTALG_DES:
      str = "DES";
      break;
    case CRYPTALG_3DES_2:
      str = "3DES-2";
      break;
    case CRYPTALG_3DES_3:
      str = "3DES-3";
      break;
    case CRYPTALG_IDEA:
      str = "IDEA";
      break;
    case CRYPTALG_RC4_40:
      str = "RC4-40";
      break;
    case CRYPTALG_RC4_128:
      str = "RC4-128";
      break;
    case CRYPTALG_SIMPLE:
      str = "SIMPLE";
      break;
    case CRYPTALG_RC2_40:
      str = "RC2-40";
      break;
    case CRYPTALG_RC2_128:
      str = "RC2-128";
      break;
    case CRYPTALG_SAFER_SK128:
      str = "SAFER-SK128";
      break;
  }

  fprintf(f, "  %s = %s\n", optname, str);
  return 0;
}

static int macalg_req(struct skipcache *cnf, char *val)
{
  int alg = idx(val, signalgs, sizeof(signalgs) / sizeof(signalgs[0])); 

  if (alg < 0)
  {
    fprintf(stderr, "skipd: unknown MAC algorithm '%s'.\n", val);
    return -1;
  }
  else
  {
    cnf->enskip_MAC_alg = alg;
    return 0;
  }
}
static int macalg_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %s\n", optname, signalgs[cnf->enskip_MAC_alg]);
  return 0;
}

static int outdeskipmode_req(struct skipcache *cnf, char *val)
{
  int m = idx(val, deskipmode, sizeof(deskipmode) / sizeof(deskipmode[0]));

  if (m < 0)
  {
    fprintf(stderr, "skipd: unknown deskip mode '%s'.\n", val);
    return -1;
  }
  else
    cnf->deskip_mode[SKIP_OUTPUT] = m;

  return 0;
}
static int outdeskipmode_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %s\n", optname, deskipmode[cnf->deskip_mode[SKIP_OUTPUT]]);
  return 0;
}

static int indeskipmode_req(struct skipcache *cnf, char *val)
{
  int m = idx(val, deskipmode, sizeof(deskipmode) / sizeof(deskipmode[0]));

  if (m < 0)
  {
    fprintf(stderr, "skipd: unknown deskip mode '%s'.\n", val);
    return -1;
  }
  else
    cnf->deskip_mode[SKIP_INPUT] = m;

  return 0;
}
static int indeskipmode_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %s\n", optname, deskipmode[cnf->deskip_mode[SKIP_INPUT]]);
  return 0;
}

static int savesaid(u_char said, char *optname, FILE *f)
{
  if (said)
  {
    fprintf(f, "  %s = ", optname);

    if (said & SAID_CRYPT)
      fprintf(f, "CRYPT ");
    if (said & SAID_AUTH)
      fprintf(f, "AUTH ");
    if (said & SAID_SEQ)
      fprintf(f, "SEQ ");
    if (said & SAID_COMP)
      fprintf(f, "COMP ");
    fprintf(f, "\n");
  }
  else
    fprintf(f, "  %s = NONE\n", optname);

  return 0;
}

static int outpolicy_req(struct skipcache *cnf, char *val)
{
  cnf->deskip_policy[SKIP_OUTPUT] = parsesaid(val);
  return 0;
}
static int outpolicy_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return savesaid(cnf->deskip_policy[SKIP_OUTPUT], optname, f);
}

static int inpolicy_req(struct skipcache *cnf, char *val)
{
  cnf->deskip_policy[SKIP_INPUT] = parsesaid(val);
  return 0;
}
static int inpolicy_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return savesaid(cnf->deskip_policy[SKIP_INPUT], optname, f);
}

static int ttl_req(struct skipcache *cnf, char *val)
{
  cnf->ttl = atoi(val);
  return 0;
}

static int ttl_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %d\n", optname, cnf->ttl);
  return 0;
}

static int maxttl_req(struct skipcache *cnf, char *val)
{
  cnf->maxttl = atoi(val);
  return 0;
}

static int maxttl_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %d\n", optname, cnf->maxttl);
  return 0;
}

static int keyttl_req(struct skipcache *cnf, char *val)
{
  cnf->keyttl = atoi(val);
  return 0;
}

static int keyttl_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %d\n", optname, cnf->keyttl);
  return 0;
}

static int keychangebytes_req(struct skipcache *cnf, char *val)
{
  cnf->enskip_Kp_maxbytes = atoi(val);
  return 0;
}

static int keychangebytes_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %d\n", optname, cnf->enskip_Kp_maxbytes);
  return 0;
}

static int keychangetime_req(struct skipcache *cnf, char *val)
{
  cnf->enskip_Kp_maxttl = atoi(val);
  return 0;
}

static int keychangetime_add(struct skipcache *cnf, char *optname, FILE *f)
{
  fprintf(f, "  %s = %d\n", optname, cnf->enskip_Kp_maxttl);
  return 0;
}

static int manualsecret_req(struct skipcache *cnf, char *val)
{
  int result = -1;
  u_char *buf;
  int len;

  /* already parsed? */
  if (cnf->sharedkey.len)
  {
    fprintf(stderr, "skipd: shared key is already specified.\n");
    return -1;
  }

  if (*val == '"')  /* Manual secret format "...." */
  {
    val++;
    for (len = 0; val[len] && (val[len] != '"'); len++)
      ;

    if (len)
    {
      int maxkeylen = 256/8 ; /*crypt_keylen(CRYPTALG_MAXBLOCK);*/

      cnf->flags |= (SKIP_VALIDKEY | SKIP_MANUALKEY);
      cnf->flags &= ~SKIP_NOKEY;

      allocdata(cnf, &cnf->sharedkey, &buf, maxkeylen);
      random_hash(buf, maxkeylen, val, len);

      result = 0;
    }
    else
      fprintf(stderr, "skipd: empty 'manual secret' value\n");
  }
  else if ((val[0] == '0') && (val[1] == 'x'))  /* Secret format 0x... */
  {
    if (scannicehex(cnf, &cnf->sharedkey, val) == 0) {
      result = 0;
      cnf->flags |= (SKIP_VALIDKEY | SKIP_MANUALKEY);
      cnf->flags &= ~SKIP_NOKEY;
    }
    else
      fprintf(stderr, "skipd: malformed 'manual secret' hex value\n");
  }
  else
    fprintf(stderr, "skipd: malformed 'manual secret' value\n");

  return result;
}

static int manualsecret_add(struct skipcache *cnf, char *optname, FILE *f)
{
  if (cnf->sharedkey.len && (cnf->flags & SKIP_MANUALKEY))
  {
    fprintf(stderr, "skipd: manual secret cannot be saved!\n");
  }

  return 0;
}

static int hosts_req(struct skipcache *cnf, skip_dynamic_t *d, char *val)
{
  int result = -1;

  d->offset = dataptr - cnf->data;
  d->len = 0;

  while (val && *val)
  {
    char *next = separator(val);

    MEMZERO(dataptr, IPADDRSIZE);
   
    if (inet_aton(val, (struct in_addr *) dataptr))
    {
      d->len  += IPADDRSIZE;
      dataptr += IPADDRSIZE;
    }
    else
      fprintf(stderr, "skipd: malformed entry in hosts list, value at %s\n", val);

    val = next;
  }

  if (d->len)
    result = 0;
  else
  {
    d->offset = 0;
    fprintf(stderr, "skipd: empty host list\n");
  }

  return result;
}

static int hosts_add(struct skipcache *cnf, skip_dynamic_t *d, char *optname, FILE *f)
{
  int i;

  if (d->len)
  {
    fprintf(f, "  %s =", optname);

    for (i = 0; i < d->len; i += IPADDRSIZE)
    {
      fprintf(f, " %u.%u.%u.%u",
	      cnf->data[d->offset + i],
	      cnf->data[d->offset + i + 1],
	      cnf->data[d->offset + i + 2],
	      cnf->data[d->offset + i + 3]);
    }

    fprintf(f, "\n");
  }

  return 0;
}
static int lookuphosts_req(struct skipcache *cnf, char *val)
{
  return hosts_req(cnf, &cnf->lookuphosts, val);
}

static int lookuphosts_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return hosts_add(cnf, &cnf->lookuphosts, optname, f);
}

static int flags_req(struct skipcache *cnf, char *val)
{
  char *walk, *next;

  cnf->flags &= ~(SKIP_VALIDKEY | SKIP_NOKEY | SKIP_TUNNEL);
  for (walk = val, next = separator(walk); walk; walk = next, next = separator(walk))
  {
    switch (idx(walk, flags, sizeof(flags) / sizeof(flags[0])))
    {
      case -1:
        fprintf(stderr, "skipd: unknown flag '%s'.\n", val);
	break;
      case 0:
	break;
      case 1:
	cnf->flags |= SKIP_NOKEY;
	cnf->flags &= ~SKIP_VALIDKEY;
	break;
      case 2:
	cnf->flags |= SKIP_TUNNEL;
	break;
      case 3:
	if (cnf->sharedkey.len == 0)
	{
	  fprintf(stderr, "skipd: no valid key specified.\n");
	  return -1;
	}
	cnf->flags |= SKIP_VALIDKEY;
	cnf->flags &= ~SKIP_NOKEY;
	break;
    }
  }
  
  /* hmm... we ignore flags if manual secret was entered */
  if (cnf->flags & SKIP_MANUALKEY)
  {
    cnf->flags |= SKIP_VALIDKEY;
    cnf->flags &= ~SKIP_NOKEY;
  }
  
  return 0;
}

static int flags_add(struct skipcache *cnf, char *optname, FILE *f)
{
  if ((cnf->flags & (SKIP_NOKEY | SKIP_VALIDKEY | SKIP_TUNNEL)) == 0)
    fprintf(f, "  %s = NONE\n", optname);
  else
  {
    fprintf(f, "  %s = ", optname);
    if (cnf->flags & SKIP_NOKEY)
      fprintf(f, "NO_KEY ");
    if (cnf->flags & SKIP_VALIDKEY)
      fprintf(f, "VALID_KEY ");
    if (cnf->flags & SKIP_TUNNEL)
      fprintf(f, "TUNNEL ");
    fprintf(f, "\n");
  }
  return 0; 
}

static int srcip_req(struct skipcache *cnf, char *val)
{
  return hosts_req(cnf, &cnf->srcip, val);
}

static int srcip_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return hosts_add(cnf, &cnf->srcip, optname, f);
}

static int dstip_req(struct skipcache *cnf, char *val)
{
  return hosts_req(cnf, &cnf->dstip, val);
}

static int dstip_add(struct skipcache *cnf, char *optname, FILE *f)
{
  return hosts_add(cnf, &cnf->dstip, optname, f);
}

/*
 * parse core
 */
static struct parseentry pt[] =
{
  { "output filter before", outfilterbefore_req, outfilterbefore_add },
  { "input filter before", infilterbefore_req, infilterbefore_add },
  { "output filter after", outfilterafter_req, outfilterafter_add },
  { "input filter after", infilterafter_req, infilterafter_add },
  { "output enskip mode", outenskipmode_req, outenskipmode_add },
  { "input enskip mode", inenskipmode_req, inenskipmode_add },
  { "src mkid", srcenskipmkid_req, srcenskipmkid_add },
  { "dst mkid", dstenskipmkid_req, dstenskipmkid_add },
  { "Crypt algorithm", kpalg_req, kpalg_add },
  { "Kij algorithm", kijalg_req, kijalg_add },
  { "MAC algorithm", macalg_req, macalg_add },
  { "output deskip mode", outdeskipmode_req, outdeskipmode_add },
  { "input deskip mode", indeskipmode_req, indeskipmode_add },
  { "output deskip policy", outpolicy_req, outpolicy_add },
  { "input deskip policy", inpolicy_req, inpolicy_add },
  { "ttl", ttl_req, ttl_add },
  { "maxttl", maxttl_req, maxttl_add },
  { "keyttl", keyttl_req, keyttl_add },
  { "key change bytes", keychangebytes_req, keychangebytes_add },
  { "key change time", keychangetime_req, keychangetime_add },
  { "manual secret", manualsecret_req, manualsecret_add },
  { "lookuphosts", lookuphosts_req, lookuphosts_add },
  { "flags", flags_req, flags_add },
  { "src ip", srcip_req, srcip_add },
  { "dst ip", dstip_req, dstip_add },
};
static struct skipcache *first, *last;

static int parse_addlist(struct skipcache *cnf)
{
  struct skipcache *new;
  
  if (MEMCMP(&cnf->id, &ignoreid, sizeof(ignoreid)) == 0)
    return 0;
    
  dynamic_alloc(cnf);

  cnf->datalen = dataptr - cnf->data;

  if ((new = malloc(sizeof(*new) + cnf->datalen)) == NULL)
  {
    fprintf(stderr, "skipd: not enough memory (malloc failed).");
    return -1;
  }

  *new = *cnf;
  new->data = (u_char *)(new + 1);
  new->next = NULL;
  MEMCPY(new->data, cnf->data, cnf->datalen);

  if (first == NULL)
  {
    first = last = new;
  }
  else
  {
    last->next = new;
    last = new;
  }
  return 0;
}

struct skipcache *parse_parse(char *path)
{
  FILE *f;
  char line[1024];
  struct skipcache entry, *lastentry = NULL;

  first = last = NULL;
  MEMCPY(&defaultconfig.id, &defaultid, sizeof(defaultid));

  if ((f = fopen(path, "r")) == NULL)
  {
    fprintf(stderr, "skipd: cannot open file '%s'\n", path);
    return NULL;
  }

  for (line[0] = '\0', linecnt = 1; fgets(line, sizeof(line), f); line[0] = '\0', linecnt++)
  {
    char *walk, name[128], var[128], val[128];
    int r;

    for (walk = line; *walk && IS_BLANK(*walk); walk++) ;

    if (IS_EOL(walk[0]) || (walk[0] == '\0') || (walk[0] == '#'))
      continue;

    if (strchr(walk, ':') && (strchr(walk, '=') == 0) && (sscanf(walk, "%[^:]:%s\n", name, var) == 1))
    {
      skip_id_t id;

      if (lastentry)
      {
	if (MEMCMP(&lastentry->id, &defaultid, sizeof(defaultid)))
          parse_addlist(lastentry);
        else
	{
	  entry.datalen = dataptr - data; 
          defaultconfig = *lastentry;
	  defaultconfig.data = defaultdata;
	  MEMCPY(defaultdata, data, lastentry->datalen);
	}
      }

      /* init default */
      lastentry = &entry;
      entry = defaultconfig;
      MEMCPY(data, defaultdata, defaultconfig.datalen);
      entry.data =  data;
      if (defaultconfig.datalen)
	dataptr = data + defaultconfig.datalen;
      else
	dataptr = data + ALIGN4(1);

      if (!strcmp(name, IGNORE))  /* Ignore entry */
	MEMCPY(&entry.id, &ignoreid, sizeof(ignoreid));
      else if (strcmp(name, "default"))  /* Not default entry */
      {
	MEMZERO(&id, sizeof(id));
        if (id_parseids(name, &id) < 0)
	{
	  fprintf(stderr, "skipd: bad id declaration on line %d\n", linecnt);
	  continue;
	}
        MEMCPY(&entry.id, &id, sizeof(id));
      }
    }
    else if ((r = sscanf(walk," %[^=] = %[^\n]\n", var, val)) == 2)
    {
      int walk, notfound;

      for (walk = 0, notfound = 1; walk < (sizeof(pt) / sizeof(struct parseentry)); walk++)
      {
        if (pt[walk].keyword && pt[walk].req && ((notfound = strncasecmp(pt[walk].keyword, var, strlen(pt[walk].keyword))) == 0))
        {
	  if (pt[walk].req(&entry, val) != 0)
	    fprintf(stderr, "skipd: error in configuration file '%s' on line %d.\n", path, linecnt);
          break;
        }
      }

      if (notfound)
        fprintf(stderr, "skipd: unknown keyword '%s' in configuration file '%s' on line %d.\n", var, path, linecnt);
    }
    else if (r != 1)
      fprintf(stderr, "skipd: error in configuration file '%s' on line %d.\n", path, linecnt);
  }

  if (lastentry)
  {
    if (MEMCMP(&lastentry->id, &defaultid, sizeof(defaultid)))
      parse_addlist(lastentry);
    else
    {
      entry.datalen = dataptr - data; 
      defaultconfig = *lastentry;
      defaultconfig.data = defaultdata;
      MEMCPY(defaultdata, data, lastentry->datalen);
    }
  }

  dataptr = defaultconfig.data + defaultconfig.datalen;
  parse_addlist(&defaultconfig);

  return first;
}

int parse_save(struct skipcache *cnf, FILE *f)
{
  struct skipcache *cnfwalk;

#ifdef PARSE_COMMENT
  if (fwrite(PARSE_COMMENT, 1, strlen(PARSE_COMMENT), f) != strlen(PARSE_COMMENT
))
  {
    return -1;
  }
#endif

  for (cnfwalk = cnf; cnfwalk; cnfwalk = cnfwalk->next)
  {
    int walk;

    if (MEMCMP(&cnfwalk->id, &defaultid, sizeof(defaultid)) == 0)
      fprintf(f, "default:\n");
    else
      id_saveids(f, &cnfwalk->id);

    for (walk = 0; walk < (sizeof(pt) / sizeof(struct parseentry)); walk++)
      if (pt[walk].add)
        pt[walk].add(cnfwalk, pt[walk].keyword, f);
      else if (pt[walk].req == NULL)
        fprintf(f, pt[walk].keyword);

    fprintf(f, "\n");
    if (cnfwalk->next)
      fprintf(f, ENTRYSEPARATOR);
  }

  return 0;
}

int parse_free(struct skipcache *cnf)
{
  struct skipcache *next;

  for (; cnf; cnf = next)
  {
    next = cnf->next;
    free(cnf);
  }
  return 0;
}

int parse_init()
{
  /* initialize all data (the parser might be called multiple times) */
  MEMCPY(&defaultconfig, &_defaultconfig, sizeof(defaultconfig));
  MEMCPY(&defaultid, &_defaultid, sizeof(defaultid));
  MEMCPY(&ignoreid, &_ignoreid, sizeof(ignoreid));
  MEMZERO(data, sizeof(data));
  MEMZERO(defaultdata, sizeof(defaultdata));
  dataptr = NULL;
  first = last = NULL;

  defaultconfig.data = defaultdata;
  defaultconfig.datalen = 0;

  return 0;
}
