/*
 * Copyright (C) 1995,1997 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */

/* #define DEBUG_REQ */

#include "config.h"

#include "skip_defs.h"
#include "memblk.h"
#include "dynamic.h"
#include "skipcache.h"
#include "interface.h"
#include "ipsp.h"
#include "queue.h"
#include "crypt.h"
#include "com.h"
#include "req.h"

#ifdef __GNUC__
#ident "$Id: req.c,v 1.5 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: req.c,v 1.5 1996/04/25 15:05:30 cschneid Exp $";
#endif

static int req_cacheinsert(struct skipreq_updateentry *r, int len)
{
  struct skipcache *new;
  int res = -1;

  if (len >= (sizeof(r->entry) + r->entry.datalen))
  {
    if ((new = KALLOC(sizeof(*new) + r->entry.datalen +
		      sizeof(struct _kernel_queue))))
    {
      u_char *q;
      
      MEMCPY(new, &r->entry, (sizeof(*new) + r->entry.datalen));
      new->data = (u_char *)(new + 1);

      /* init kernel q */
      q = new->data + new->datalen;
      MEMZERO(q, sizeof(struct _kernel_queue));
      new->q.offset = new->datalen;
      new->datalen += sizeof(struct _kernel_queue);
      new->q.len = sizeof(struct _kernel_queue);
      queue_initqueues(new);

      if (skipcache_update(new) == 0)
	res = 0;
      else
	KFREE(new, sizeof(*new) + r->entry.datalen);
    }
  }

  return res;
}

int req_handle(u_char *req, int len)
{
  struct skipreq_hdr *hdr = (struct skipreq_hdr *)req;
  int reqlen = -1;

  if (len >= sizeof(*hdr))
  {

    switch (hdr->req)
    {
      case SKIPREQ_UPDATEENTRY:
      {
	struct skipreq_updateentry *r = (struct skipreq_updateentry *)hdr;
        struct skipcache *l1, *l2;

	if (MEMCMP(&r->hdr.id, &r->oldhdr.id, sizeof(r->hdr.id)))
	{
	  /* ids differ */
	  l1 = skipcache_lookup(&r->oldhdr.id);
	  l2 = skipcache_lookup(&r->hdr.id);
#ifdef DEBUG_REQ
          printf("UPDATEENTRY: ids differ; l1=%p l2=%p\n", l1, l2);
#endif

	  /* no new entry -> insert */
	  if (l2 == NULL)
	  {
	    if (req_cacheinsert(r, len) == 0)
	      reqlen = 0;
	  }
	  else
	  {
	    /* cache entry already exists, only add mapping */
	    if (skipcache_addmapping(l2, r->oldhdr.srcip, r->oldhdr.dstip) == 0)
	      reqlen = 0;
	    skipcache_release(l2);
	  }

	  /* feed old entry and release/ remove */
	  if (l1)
	  {
	    queue_feed(l1);
	    skipcache_release(l1);
	    skipcache_remove(&l1->id);
	  }
	}
	else
	{
          /* Work-around for master key id change: check for cache entries
             of linked IP packets when receiving a new key */
          if ((r->hdr.id.type[0] != SKIP_NSID_IPV4) ||
              (r->hdr.id.type[1] != SKIP_NSID_IPV4)) {
#ifdef SMALL_KERNEL_STACK
            static
#endif
	    skip_id_t id;

            MEMZERO(&id, sizeof(id));
            id.type[0] = SKIP_NSID_IPV4;
            id.type[1] = SKIP_NSID_IPV4;
            MEMCPY(id.id[0], r->hdr.srcip, IPADDRSIZE);
            MEMCPY(id.id[1], r->hdr.dstip, IPADDRSIZE);
            l2 = skipcache_lookup(&id);
          }
          else
            l2 = NULL;
	  l1 = skipcache_lookup(&r->hdr.id);

#ifdef DEBUG_REQ
          printf("UPDATEENTRY: ids equal, lookup: l1=%p l2=%p\n", l1, l2);
#endif

	  if (req_cacheinsert(r, len) == 0)
	    reqlen = sizeof(r->entry) + r->entry.datalen -
	             sizeof(struct _kernel_queue);

	  if (l1)
	  {
	    queue_feed(l1);
	    skipcache_release(l1);           /* -> free! */
	  }

          /* Work-around for master key change (see above) */ 
          if (l2)
          {
	    if (l2 != l1)
              queue_feed(l2);
            skipcache_release(l2);
            if (l2 != l1)
	      skipcache_remove(&l2->id);
          }
	}

	break;
      }

      case SKIPREQ_UPDATEENSKIPKP:
      {
	struct skipreq_updateKp *r = (struct skipreq_updateKp *)hdr;
	struct skipcache *l = skipcache_lookup(&hdr->id);

#ifdef DEBUG_REQ
        printf("UPDATEENSKIPKP: lookup: l=%p\n", l);
#endif

        if (l)
        {
          if (len >= sizeof(*r))
          {
	    dynamic_copybuf2dyn(l->data, &l->enskip_A_Kp, 
				r->A_Kp, r->A_Kplen);
	    dynamic_copybuf2dyn(l->data, &l->enskip_E_Kp, 
				r->E_Kp, r->E_Kplen);
	    dynamic_copybuf2dyn(l->data, &l->enskip_Kp_Kijn.key, 
				r->Kp_Kijn, r->Kp_Kijnlen);
            l->enskip_Kp_Kijn.n = r->n;
            l->enskip_Kij_alg = r->Kij_alg;
            MEMZERO(l->data + l->enskip_MI.offset, l->enskip_MI.len);
	    reqlen = sizeof(*r);
          }
          skipcache_release(l);
        }
	break;
      }

      case SKIPREQ_UPDATEDESKIPKP:
      {
	struct skipreq_updateKp *r = (struct skipreq_updateKp *)hdr;
	struct skipcache *l = skipcache_lookup(&hdr->id);

#ifdef DEBUG_REQ
        printf("UPDATEDESKIPKP: lookup: l=%p\n", l);
#endif

        if (l)
        {
          if (len >= sizeof(*r))
          {
	    dynamic_copybuf2dyn(l->data, &l->deskip_A_Kp, 
				r->A_Kp, r->A_Kplen);
	    dynamic_copybuf2dyn(l->data, &l->deskip_E_Kp, 
				r->E_Kp, r->E_Kplen);
	    dynamic_copybuf2dyn(l->data, &l->deskip_Kp_Kijn.key, 
				r->Kp_Kijn, r->Kp_Kijnlen);
            l->deskip_Kp_Kijn.n = r->n;
            l->deskip_Kij_alg = r->Kij_alg;
	    reqlen = sizeof(*r);
          }
          skipcache_release(l);
        }
	break;
      }

      case SKIPREQ_UPDATEKIJN:
      {
	struct skipreq_updateKijn *r = (struct skipreq_updateKijn *)hdr;
	struct skipcache *l1, *l2;

        l1 = skipcache_lookup(&r->oldhdr.id);
        l2 = skipcache_lookup(&r->hdr.id);

#ifdef DEBUG_REQ
        printf("UPDATEKIJN: r->n=%i lookup: l1=%p l2=%p\n", r->n, l1, l2);
#endif

        if (l1 && l2)
        {
          /*
           * add mapping to new entry
           * we do not remove the old entry since it can still be used
           * by other connections. the entry will timeout
           * if it is no longer used
           */
          if ((l1 != l2) &&
             (skipcache_addmapping(l2, r->oldhdr.srcip, r->oldhdr.dstip) < 0))
          {
             skipcache_release(l1);
             skipcache_release(l2);
             break;
          }
        }

	if (l2)
	{
          if (r->n)
          {
            if (dynamic_copybuf2dyn(l2->data, &l2->Kijn[(r->n - 1) % 3].key,
				    r->Kijn[(r->n - 1) % 3], MAXKEYLEN) == 0)
            {
              l2->Kijn[(r->n - 1) % 3].n = (r->n - 1);
            }
            if (dynamic_copybuf2dyn(l2->data, &l2->Kijn[r->n % 3].key,
				    r->Kijn[r->n % 3], MAXKEYLEN) == 0)
            {
              l2->Kijn[r->n % 3].n = r->n;
            }
            if (dynamic_copybuf2dyn(l2->data, &l2->Kijn[(r->n + 1) % 3].key,
				    r->Kijn[(r->n + 1) % 3], MAXKEYLEN) == 0)
            {
              l2->Kijn[(r->n + 1) % 3].n = (r->n + 1);
            }
            l2->keyttl = r->keyttl;
          }
          else
            l2->flags |= SKIP_NOKEY;

	  reqlen = sizeof(*r);
          if (l1)
            queue_feed(l1);
	}

        skipcache_release(l1);
        skipcache_release(l2);
	break;
      }

      case SKIPREQ_GETREQUESTS:
      {
	reqlen = skipcache_getrequests(req, len);
	break;
      }

      case SKIPREQ_REMOVEENTRY:
      {
	if (skipcache_remove(&hdr->id) == 0)
	  reqlen = sizeof(*hdr);
	break;
      }

      case SKIPREQ_STATCACHE:
      {
	struct skipreq_statcache *r = (struct skipreq_statcache *)hdr;

	if (len >= sizeof(*r))
	{
	  MEMCPY(&r->stat, &skipcache_stat, sizeof(skipcache_stat));
	  reqlen = sizeof(*r);
	}
	break;
      }

      case SKIPREQ_STATIPSP:
      {
	struct skipreq_statipsp *r = (struct skipreq_statipsp *)hdr;

	if (len >= sizeof(*r))
	{
	  MEMCPY(&r->stat, &ipsp_stat, sizeof(ipsp_stat));
	  reqlen = sizeof(*r);
	}
	break;
      }
      
      case SKIPREQ_FLUSH:
      {
	if (skipcache_flush() == 0)
	  reqlen = sizeof(*hdr);
	break;
      }

      case SKIPREQ_ATTACH:
      {
	if (interface_attach(NULL, hdr->srcip) == 0)
	  reqlen = sizeof(*hdr);
	break;
      }

      case SKIPREQ_DETACH:
      {
	/* first free all pending paket queues -mh */
	skipcache_freeallqueues();

	if (interface_detach(NULL, hdr->srcip) == 0) 
	  reqlen = sizeof(*hdr);
	break;
      }

      case SKIPREQ_REGISTER:
      {
	struct skipreq_register *r = (struct skipreq_register *)hdr;

	if (com_register(r->pid) == 0)
	  reqlen = sizeof(*r);
	break;
      }

      case SKIPREQ_DUMP:
      {
        struct skipcache *c = skipcache_lookup(&hdr->id);

	if (c)
        {
          int l = (c->datalen + sizeof(*c));

          if (l <= len)
          {
            MEMCPY(req, c, l);
	    reqlen = l;
          }
          skipcache_release(c);
        }
	break;
      }

      /* reads the hashqueue and all IDs of the cache */
      case SKIPREQ_LISTCACHE:
      {
        reqlen = skipcache_listcache(req,len);

	break;
      }

  /* allocates a queue entry and forces the daemon to handle a getentry req */
      case SKIPREQ_TEST:
      {
	struct skipcache *new;

	if ((new = KALLOC(sizeof(*new) + sizeof(struct _kernel_queue))))
	{
	  MEMZERO(new, sizeof(*new));
	  MEMCPY(&new->id, &hdr->id, sizeof(new->id));
	  new->data = (u_char *)(new + 1);
	  new->datalen = 0;
	  new->ttl = 60;
	  new->q.offset = new->datalen;
	  new->q.len = sizeof(struct _kernel_queue);
	  new->datalen += sizeof(struct _kernel_queue);

	  MEMZERO(new->data + new->q.offset, new->q.len);
          queue_initqueues(new);

	  if (skipcache_update(new) == 0)
	  {
	    com_signal(SIGURG);
	    reqlen = sizeof(*hdr);	  
	  }
	}
	break;
      }

    } /* switch */
  }

  return reqlen;
}
