/*
 * Copyright (C) 1995,1997 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */

/* #define DEBUG_CACHE */
/* #define DEBUG_CACHE_TIMER */

#include "config.h"
#include <errno.h>

#include "skip_defs.h"
#include "dynamic.h"
#include "memblk.h"
#include "skipcache.h"
#include "crypt.h"
#include "random.h"
#include "ipsp.h"
#include "queue.h"
#include "req.h"

#ifdef __GNUC__
#ident "$Id: skipcache.c,v 1.8 1996/04/25 15:05:30 cschneid Exp $"
#else
static char rcsid[] = "$Id: skipcache.c,v 1.8 1996/04/25 15:05:30 cschneid Exp $";
#endif

#define TIMERINTERVAL	5	/* expire interval in seconds */
#define INCTTL		1	/* value ttl is incremented after use */

#define ALIGN4(len) (((len) + 3) & ~3)

#ifdef DEBUG_CACHE
#define PRINTIDS(str, x) printf(">> %s %02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x\n", str, x[0], x[1], x[2], x[3], x[4], x[5], x[6], x[7], x[8], x[9], x[10], x[11], x[12], x[13], x[14], x[15]);
#define PRINTID(a, b) PRINTIDS((a), ((unsigned char *) &(b)))
#else
#define PRINTID(str, x)	
#endif

skipcache_stat_t skipcache_stat;
u_int32 skip_n;
static struct skipcache *tbl[HASHTBLSIZE];
static SEMTYPE semaphore;
static int maxent, maxttl;

static inline int skipcache_index(skip_id_t *id)
{
  u_char *d = (u_char *)id;
  int walk, sum;

  for (walk = sum = 0; walk < sizeof(*id); walk++)
    sum += d[walk];

  return (sum % HASHTBLSIZE);
}

/*********************************************************************
 * Everything concerning mapping from src & dst IP to skipcache entry
 */
struct skipcache_mapping
{
  struct skipcache_mapping **prev, *next;	/* Src IP: Hash table link */
  skip_id_t id;					/* Always contains 2 IPs */
  struct skipcache *entry;
};

struct skipcache_map
{
  struct skipcache_map *next;
  int numsrc, numdst;    	/* # of skipcache_mappings after this struct */
};

static struct skipcache_mapping *mapping_tbl[HASHTBLSIZE];

/* Src and dst IPs are arrays containing the ip lists */
static int skipcache_insertmap(struct skipcache *c)
{
  struct skipcache_map *map;
  struct skipcache_mapping *mappings;
  int numsrc, numdst;
  u_char *srcips, *dstips;

  numsrc = c->srcip.len / IPADDRSIZE;
  numdst = c->dstip.len / IPADDRSIZE;
  srcips = c->data + c->srcip.offset;
  dstips = c->data + c->dstip.offset;

#ifdef DEBUG_CACHE
  printf("skipcache_insertmap(), numsrc=%i, numdst=%i\n", numsrc, numdst);
#endif

  if ((map = KALLOC(sizeof(*map) + numsrc * numdst * sizeof(*mappings))))
  {
    int i, j;

    map->next = (struct skipcache_map *)c->map;
    c->map = (void *)map;
    map->numsrc = numsrc;
    map->numdst = numdst;

    mappings = (struct skipcache_mapping *)(map + 1);

    for (i = 0; i < numsrc * IPADDRSIZE; i += IPADDRSIZE)
    {
      for (j = 0; j < numdst * IPADDRSIZE; j += IPADDRSIZE)
      {
	int iphash;

	mappings->entry = c;
	MEMZERO(&mappings->id, sizeof(mappings->id));
	mappings->id.type[0] = mappings->id.type[1] = SKIP_NSID_IPV4;
	MEMCPY(mappings->id.id[0], &srcips[i], IPADDRSIZE);
	MEMCPY(mappings->id.id[1], &dstips[j], IPADDRSIZE);

	PRINTID("mapping [0]", mappings->id.id[0]);
        PRINTID("mapping [1]", mappings->id.id[1]);

	iphash = skipcache_index(&mappings->id);
	mappings->prev = &mapping_tbl[iphash];
        if ((mappings->next = mapping_tbl[iphash]))
	  mappings->next->prev = &mappings->next;
        mapping_tbl[iphash] = mappings;

	mappings++;
      }
    }
  }

  return map ? 0 : -1;
}

static void skipcache_removemap(struct skipcache *c)
{
  struct skipcache_map *map, *next;

#ifdef DEBUG_CACHE
  printf("skipcache_removemap()\n");
#endif

  for (map = (struct skipcache_map *)c->map; map; map = next)
  {
    int i;
    struct skipcache_mapping *mappings;

    next = map->next;
    mappings = (struct skipcache_mapping *)(map + 1);

    for (i = 0; i < (map->numsrc * map->numdst); i++)
    {
      *mappings->prev = mappings->next;
      if (mappings->next)
        mappings->next->prev = mappings->prev;

      /* this was missing in previous releases */
      mappings++;
    }

    KFREE(map, sizeof(*map) + map->numsrc * map->numdst * sizeof(*mappings));
  }

  c->map = NULL;
}

int skipcache_addmapping(struct skipcache *c, u_char *srcip, u_char *dstip)
{
  int result = -1;
  struct skipcache_map *map;
  struct skipcache_mapping *mappings;

#ifdef DEBUG_CACHE
  printf("skipcache_addmapping: srcip=%08x dstip=%08x\n",
         *((unsigned int *) srcip), *((unsigned int *) dstip));
#endif

  if ((map = KALLOC(sizeof(*map) + sizeof(*mappings))))
  {
    int iphash;

    SEMLOCK(semaphore);

    map->next = (struct skipcache_map *)c->map;
    c->map = (void *)map;
    map->numsrc = map->numdst = 1;

    mappings = (struct skipcache_mapping *)(map + 1);
    mappings->entry = c;

    MEMZERO(&mappings->id, sizeof(mappings->id));
    mappings->id.type[0] = mappings->id.type[1] = SKIP_NSID_IPV4;
    MEMCPY(mappings->id.id[0], srcip, IPADDRSIZE);
    MEMCPY(mappings->id.id[1], dstip, IPADDRSIZE);

    iphash = skipcache_index(&mappings->id);
    mappings->prev = &mapping_tbl[iphash];
    if ((mappings->next = mapping_tbl[iphash]))
      mappings->next->prev = &mappings->next;
    mapping_tbl[iphash] = mappings;

    SEMUNLOCK(semaphore);

    result = 0;
  }

  return result;
}

static int skipcache_getdstmappings(struct skipcache *c,
				    u_char *dstips, int dstips_len)
{
  struct skipcache_map *map;
  int len = 0;

#ifdef DEBUG_CACHE
  printf("skipcache_getdstmappings()\n");
#endif

  for (map = (struct skipcache_map *)c->map; map; map = map->next)
  {
    int i;
    struct skipcache_mapping *mapping;

    mapping = (struct skipcache_mapping *)(map + 1);

    /* add dst ips */
    for (i = 0; (i < map->numdst) && ((len + IPADDRSIZE) < dstips_len); i++)
    {
      MEMCPY(dstips + len, mapping[i].id.id[1], IPADDRSIZE);
      len += IPADDRSIZE;
    }
  }

  return len;
}

static int skipcache_getsrcmappings(struct skipcache *c,
				    u_char *srcips, int srcips_len)
{
  struct skipcache_map *map;
  int len = 0;
#ifdef DEBUG_CACHE
  printf("skipcache_getsrcmappings()\n");
#endif

  for (map = (struct skipcache_map *)c->map; map; map = map->next)
  {
    int i;
    struct skipcache_mapping *mapping;

    mapping = (struct skipcache_mapping *)(map + 1);

    /* add src ips */
    for (i = 0; (i < map->numsrc) && ((len + IPADDRSIZE) < srcips_len); i++)
    {
      MEMCPY(srcips + len, mapping[i * map->numdst].id.id[0], IPADDRSIZE);
      len += IPADDRSIZE;
    }
  }

  return len;
}

/*
 * id is of type SKIP_NSID_IPV4
 */
static inline struct skipcache *skipcache_lookupip(skip_id_t *id)
{
  struct skipcache_mapping *m;

  PRINTID("lookupip [0]", id->id[0]);
  PRINTID("lookupip [1]", id->id[1]);

  m = mapping_tbl[skipcache_index(id)];

  while (m && (MEMCMP(&m->id, id, sizeof(*id)) != 0))
    m = m->next;

  PRINTID("lookupip end [0]", id->id[0]);
  PRINTID("lookupip end [1]", id->id[1]);

  return m ? m->entry : NULL;
}

/*********************************************************************
 * Skipcache
 */

static int skipcache_freequeues(struct skipcache *c)
{
  SKIP_FREEQ(c);
  return 0;
}

int skipcache_freeallqueues()
{
  int walk;

  SEMLOCK(semaphore);

  for (walk = 0; walk < HASHTBLSIZE; walk++)
  {
    struct skipcache *c;

    for (c = tbl[walk]; c; c = c->next)
      skipcache_freequeues(c);
  }

  SEMUNLOCK(semaphore);
  return 0;
}

static int skipcache_freeentry(struct skipcache *c)
{

  skipcache_freequeues(c);
  KFREE(c, (c->datalen + sizeof(struct skipcache)));
  return 0;
}

static int skipcache_removeentry(struct skipcache *c)
{
  PRINTID("removeentry", c->id);

  /* rip out of hash chain */
  if (c->prev)
  {
    c->prev->next = c->next;
    if (c->next)
      c->next->prev = c->prev;
  }
  else
  {
    int idx = skipcache_index(&c->id);

    tbl[idx] = c->next;
    if (c->next)
      c->next->prev = NULL;
  }
  skipcache_stat.entries--;
  c->next = c->prev = NULL;

  return 0;
}

static int skipcache_replaceentry(struct skipcache *old, struct skipcache *new)
{
  new->next = old->next;
  new->prev = old->prev;

  if (old->prev)
    old->prev->next = new;
  else
    tbl[skipcache_index(&old->id)] = new;

  if (old->next)
    old->next->prev = new;

  old->next = old->prev = NULL;

  return 0;
}

static void skipcache_removetimedout(struct skipcache *c)
{

  skipcache_stat.timedout++;
  skipcache_removeentry(c);
  skipcache_removemap(c);
  if (--c->ref <= 0)
    skipcache_freeentry(c);
}

/*
 * the function decrements all entries' time to live value and
 * removes all entries with a time to live value equal or below zero.
 * the number of expired entries is returned.
 */
static int skipcache_age(int times)
{
  int walk, cnt = skipcache_stat.entries;


  for (walk = 0; walk < HASHTBLSIZE; walk++)
  {
    struct skipcache *c, *next;

    for (c = tbl[walk]; c; c = next)
    {
      next = c->next;


#ifdef KERNEL
      /* The following must be cleared or we will be able to fetch
         Kijn only once for each primary cache entry (if the cache entry
	 lasts longer than twice the update period)...
         It is safe do do it here, because it will be cleares at most
         every TIMEOUT, preventing endless request loops. */
      c->flags &= ~(SKIP_REQSENT_GETDESKIPKEY|SKIP_REQSENT_GETENSKIPKEY);
#endif

#ifdef DEBUG_CACHE_TIMER
      printf(">> checking entry with len %d and ttl %d\n", c->datalen, c->ttl);
#endif /* DEBUG */

      /* Absolute ttl, e.g. Public Key ttl */
      if (c->keyttl && ((c->keyttl -= times) <= 0))
      {
	skipcache_removetimedout(c);
      }
      else if ((c->flags & SKIP_DONTAGE) == 0)
      {
        /* is entry expired and not referenced? */
        if (c->ttl && (c->ref <= 1) && ((c->ttl -= times) <= 0))
        {
	  skipcache_removetimedout(c);
        }
        else  /* Check time for packet key change */
        {
          if ((c->enskip_Kp_ttl > 0) && ((c->enskip_Kp_ttl -= times) <= 0))
          {
            c->enskip_Kp_ttl = -1;
          }
        }
      }
    }
  }

  return (cnt - skipcache_stat.entries);
}

/*
 * remove entry with lowest time to live from cache
 */
static int skipcache_removelru(void)
{
  int walk;
  struct skipcache *lru = NULL;

  /* find entry with lowest ttl */
  for (walk = 0; walk < HASHTBLSIZE; walk++)
  {
    struct skipcache *c;

    for (c = tbl[walk]; c; c = c->next)
    {
      /* has entry lower ttl and is not (externally) referenced? */
      if ((c->ref <= 1) && c->ttl && ((lru == NULL) || (lru->ttl > c->ttl)))
        lru = c;
    }
  }

  if (lru)
  {
    skipcache_removeentry(lru);
    skipcache_removemap(lru);
    if (--lru->ref <= 0)
      skipcache_freeentry(lru);
    return 0;
  }
  else
    return -ENOENT;
}

static int skipcache_insertentry(struct skipcache *new)
{
  int idx = skipcache_index(&new->id);

  if (maxent && (skipcache_stat.entries > maxent) &&
      (skipcache_removelru() < 0))
    return -EAGAIN;

  /* link in hash list */
  new->next = tbl[idx];
  new->prev = NULL;

  if (tbl[idx])
    tbl[idx]->prev = new;
  tbl[idx] = new;
  skipcache_stat.entries++;

  return 0;
}

/*
 * returns a pointer to the cache structure for a given IP address
 * if no entry is found with a matching IP address, NULL is returned
 */
struct skipcache *skipcache_lookup(skip_id_t *id)
{
  struct skipcache *walk = NULL;
  int idx = skipcache_index(id);

  PRINTID("lookup [0]", id->id[0]);
  PRINTID("lookup [1]", id->id[1]);

  SEMLOCK(semaphore);

  if ((id->type[0] == SKIP_NSID_IPV4) && (id->type[1] == SKIP_NSID_IPV4))
    walk = skipcache_lookupip(id);

  if (walk == NULL)
  {
    for (walk = tbl[idx]; walk; walk = walk->next)
      if (MEMCMP(&walk->id, id, sizeof(*id)) == 0)
        break;
  }

  if (walk == NULL)
  {
    /* send request to daemon if no entry was found */
    skipcache_stat.lookuprequests++;
  }
  else
  {
    /* inc reference count of entry */
    walk->ref++;
    skipcache_stat.lookuphits++;
  }

  SEMUNLOCK(semaphore);

  PRINTID("lookup end [0]", id->id[0]);
  PRINTID("lookup end [1]", id->id[1]);

  return walk;
}

/*
 * release entry
 * if there are no more references, update and remove is checked.
 * if the entry is neither updated nor removed, its ttl is incremented
 */
int skipcache_release(struct skipcache *c)
{
  if (c == NULL)
    return -1;


  SEMLOCK(semaphore);

  if (--c->ref <= 0)
    skipcache_freeentry(c);
  else if (c->flags & (SKIP_VALIDKEY | SKIP_NOKEY))
  {
    c->ttl += INCTTL;
    if (c->ttl > c->maxttl)
      c->ttl = c->maxttl;
  }

  SEMUNLOCK(semaphore);

  return 0;
}

#ifdef KERNEL
int skipcache_getrequests(u_char *buf, int len)
{
  int walk, used;

  SEMLOCK(semaphore);

  for (walk = used = 0; walk < HASHTBLSIZE; walk++)
  {
    struct skipcache *c;

    for (c = tbl[walk]; c; c = c->next)
    {
      if ((c->flags & (SKIP_VALIDKEY | SKIP_NOKEY | SKIP_REQSENT_GETENTRY)) == 0)
      {
	struct skipreq_getentry *r = (struct skipreq_getentry *)buf;
	if ((used + sizeof(*r)) <= len)
	{
	  if (queue_getips(c, r->hdr.srcip, r->hdr.dstip) == 0)
	  {
	    r->hdr.req = SKIPREQ_GETENTRY;
	    MEMCPY(&r->hdr.id, &c->id, sizeof(c->id));
	    buf += sizeof(*r);
	    used += sizeof(*r);
	    c->flags |= SKIP_REQSENT_GETENTRY;
	  }
	}
	else
	  break;
      }
      if (((c->flags & SKIP_REQSENT_GETENSKIPKEY) == 0) &&
	  (c->flags & SKIP_ENSKIP_INVALID))
      {
	struct skipreq_getenskipkey *r = (struct skipreq_getenskipkey *)buf;
	if ((used + sizeof(*r)) <= len)
	{
          if (queue_getips(c, r->hdr.srcip, r->hdr.dstip) == 0)
          {
	    r->hdr.req = SKIPREQ_GETENSKIPKEY;
	    MEMCPY(&r->hdr.id, &c->id, sizeof(c->id));
	    r->Kij_alg = c->enskip_Kij_alg;
	    dynamic_copydyn2buf(c->data, &c->enskip_Kp_Kijn.key, r->Kp, sizeof(r->Kp));

	    buf += sizeof(*r);
	    used += sizeof(*r);
	    c->flags |= SKIP_REQSENT_GETENSKIPKEY;
          }
	}
	else
	  break;
      }
      if (((c->flags & SKIP_REQSENT_GETDESKIPKEY) == 0) &&
	  (c->flags & SKIP_DESKIP_INVALID))
      {
	struct skipreq_getdeskipkey *r = (struct skipreq_getdeskipkey *)buf;
	if ((used + sizeof(*r)) <= len)
	{
          if (queue_getips(c, r->hdr.srcip, r->hdr.dstip) == 0)
          {
	    r->hdr.req = SKIPREQ_GETDESKIPKEY;
	    MEMCPY(&r->hdr.id, &c->id, sizeof(c->id));
	    r->n = c->deskip_Kp_Kijn.n;
	    r->Kij_alg = c->deskip_Kij_alg;
	    /* Error! deskip_Kp_Kijn.len is always the maximum length plus
	       some, we cannot copy this into r->Kp_Kijn, it will overflow.
	       [ original code: 
                 dynamic_copydyn2buf(c->data, &c->deskip_Kp_Kijn.key, r->Kp_Kijn, sizeof(r->Kp_Kijn));
               ]
               So let's do it right: */
            if (c->deskip_Kp_Kijn_len <= sizeof(r->Kp_Kijn))
              MEMCPY(r->Kp_Kijn, c->deskip_Kp_Kijn.key.offset + c->data,
                     c->deskip_Kp_Kijn_len);
            else
              printf(">> skipcache_getrequests: overflow: src=%i > dst=%i\n", 
                     c->deskip_Kp_Kijn_len, sizeof(r->Kp_Kijn));

	    buf += sizeof(*r);
	    used += sizeof(*r);
	    c->flags |= SKIP_REQSENT_GETDESKIPKEY;
          }
	}
	else
	  break;
      }

      c->flags &= ~SKIP_SIGNALED;  /* Requests for this signal delivered */
    }
  }

  SEMUNLOCK(semaphore);

  return used;
}
#endif

/*
 * inserts the given entry in the hash table
 * if the entry already existed, the old entry's queues
 * are handed over to the IPSP object that treats them
 * with the updated keying information
 */
int skipcache_update(struct skipcache *new)
{
  int idx;
  struct skipcache *walk;

  /* no time to live? */
  if (new->ttl < 0)
    return -EINVAL;

  /* time to live to large */
  if (maxttl && (new->maxttl > maxttl))
    new->maxttl = maxttl;

  if (new->maxttl && (new->ttl > new->maxttl))
    new->ttl = new->maxttl;


  PRINTID("update", new->id);

  /* set starting ref count, init update pointer, map pointer */
  new->ref = 1;
  new->map = NULL;


  idx = skipcache_index(&new->id);

  SEMLOCK(semaphore);
  /*
   * search entry in hash table and remember last one to fix
   * the linked list afterwards.
   */
  for (walk = tbl[idx]; walk; walk = walk->next)
    if (MEMCMP(&walk->id, &new->id, sizeof(new->id)) == 0)
      break;

  /* already an entry in skipcache? */
  if (walk)
  {
    skipcache_stat.updated++;

    skipcache_replaceentry(walk, new);
    skipcache_removemap(walk);
    skipcache_insertmap(new);

    if (--walk->ref <= 0)
      skipcache_freeentry(walk);
  }
  else
  {
    skipcache_stat.inserted++;

    skipcache_insertentry(new);
    skipcache_insertmap(new);
  }

  SEMUNLOCK(semaphore);
  return 0;
}

/*
 * removes an entry with given IP address from cache
 * returns 0 on success, -1 on failure
 */
int skipcache_remove(skip_id_t *id)
{
  int idx;
  struct skipcache *walk, *last;

  idx = skipcache_index(id);


  SEMLOCK(semaphore);
  /*
   * search entry in hash table and remember last one to fix
   * the linked list afterwards.
   */
  for (walk = tbl[idx], last = NULL; walk; last = walk, walk = walk->next)
    if (MEMCMP(&walk->id, id, sizeof(*id)) == 0)
      break;

  if (walk)
  {
    skipcache_stat.removed++;
    skipcache_removeentry(walk);
    skipcache_removemap(walk);
    if (--walk->ref <= 0)
      skipcache_freeentry(walk);
    else
      walk->flags |= SKIP_REMOVE;

    SEMUNLOCK(semaphore);
    return 0;
  }

  SEMUNLOCK(semaphore);
  return -ENOENT;
}

/*
 * change the id of an entry (mapping will not be lost!)
 */
int skipcache_change(struct skipcache *c, skip_id_t *id)
{
  int res;


  SEMLOCK(semaphore);

  skipcache_removeentry(c);
  MEMCPY(&c->id, id, sizeof(*id));
  res = skipcache_insertentry(c);

  SEMUNLOCK(semaphore);

  return res;
}

static int skipcache_timeout(void *arg)
{
  static int times = 1;


  /*
   * this function should not sleep
   * so we just try to lock the semaphore
   * if we do not succeed, we expire the entries
   * another time.
   */
  if (SEMLOCKTRY(semaphore) == 0)
  {
    skipcache_age(times);
    SEMUNLOCK(semaphore);
    times = 0;
  }

  times += TIMERINTERVAL;
 

  /* update skip_n */
  skip_n = GMTSECONDS;
  skip_n /= 3600;
  skip_n -= (25 * 365 + 6) * 24;    /* 25 years + 6 leap years */

  TIMEOUT(skipcache_timeout, NULL, TIMERINTERVAL);
  return 0;
}

static int skipcache_reset(void)
{
  int walk;

  for (walk = 0; walk < HASHTBLSIZE; walk++)
    tbl[walk] = NULL;

  MEMZERO(&skipcache_stat, sizeof(skipcache_stat));

  return 0;
}

/* 
 * Flushes all entries from skipcache. Note: Also clears the cache statistics
 */
int skipcache_flush(void)
{
  int walk;

  SEMLOCK(semaphore);

  for (walk = 0; walk < HASHTBLSIZE; walk++)
  {
    struct skipcache *c, *next;

    for (c = tbl[walk]; c; c = next)
    {
      next = c->next;
      skipcache_removemap(c);
      skipcache_freeentry(c);
    }
  }

  skipcache_reset();
  SEMUNLOCK(semaphore);

  return 0;
}

static int copydynamic(struct skipcache *oc, struct skipcache *nc,
		       skip_dynamic_t *od, skip_dynamic_t *nd)
{
  u_char *ptr = nc->data + ALIGN4(nc->datalen);

  if (od->len <= 0)
    return -1;

  MEMCPY(ptr, oc->data + od->offset, od->len);
  nd->len = od->len;
  nd->offset = ptr - nc->data;
  nc->datalen = nd->offset + nd->len;

  return 0;
}

static int copydynamic_maxkeylen(struct skipcache *oc, struct skipcache *nc,
				 skip_dynamic_t *od, skip_dynamic_t *nd)
{
  u_char *ptr = nc->data + nc->datalen;

  if (od->len <= 0)
    return -1;

  MEMCPY(ptr, oc->data + od->offset + od->len - MAXKEYLEN, MAXKEYLEN);
  nd->len = MAXKEYLEN;
  nd->offset = nc->datalen;
  nc->datalen += nd->len;

  return 0;
}

int skipcache_compressentry(struct skipcache *c, u_char *buf, int len)
{
  struct skipcache *cnf = (struct skipcache *)buf;
  u_char *ptr;

  MEMCPY(cnf, c, sizeof(*cnf));

  /* clear user level stuff */
  cnf->sharedkey.len = 0;
  cnf->lookuphosts.len = 0;
  cnf->q.len = 0;

  /* copy dynamic part */
  cnf->data = (u_char *)(cnf + 1);
  cnf->datalen = 0;

  /* copy only last MAXKEYLEN bytes for Kijn */
  copydynamic_maxkeylen(c, cnf, &c->Kijn[0].key, &cnf->Kijn[0].key);
  copydynamic_maxkeylen(c, cnf, &c->Kijn[1].key, &cnf->Kijn[1].key);
  copydynamic_maxkeylen(c, cnf, &c->Kijn[2].key, &cnf->Kijn[2].key);

  copydynamic(c, cnf, &c->enskip_A_Kp, &cnf->enskip_A_Kp);
  copydynamic(c, cnf, &c->enskip_E_Kp, &cnf->enskip_E_Kp);
  copydynamic(c, cnf, &c->enskip_Kp_Kijn.key, &cnf->enskip_Kp_Kijn.key);
  copydynamic(c, cnf, &c->enskip_MI, &cnf->enskip_MI);

  copydynamic(c, cnf, &c->deskip_A_Kp, &cnf->deskip_A_Kp);
  copydynamic(c, cnf, &c->deskip_E_Kp, &cnf->deskip_E_Kp);
  copydynamic(c, cnf, &c->deskip_Kp_Kijn.key, &cnf->deskip_Kp_Kijn.key);

  copydynamic(c, cnf, &c->encryptstate, &cnf->encryptstate);
  copydynamic(c, cnf, &c->Kp_decryptstate, &cnf->Kp_decryptstate);
  copydynamic(c, cnf, &c->payload_decryptstate, &cnf->payload_decryptstate);

  /* copy ip addr mappings */
  ptr = cnf->data + ALIGN4(cnf->datalen);
  cnf->srcip.offset = (ptr - cnf->data);
  cnf->srcip.len = skipcache_getsrcmappings(c, ptr, len - (ptr - buf));
  cnf->datalen += cnf->srcip.len;

  ptr += cnf->srcip.len;
  cnf->dstip.offset = (ptr - cnf->data);
  cnf->dstip.len = skipcache_getdstmappings(c, ptr, len - (ptr - buf));
  cnf->datalen += cnf->dstip.len;

  /*
   * XXX: to reduce memory overhead the crypt states, all *Kp* containers
   * and the MI could be 'allocated' here.
   */

  return 0;
}

/*
 * initalize hash table
 */
int skipcache_init(int mttl, int ment)
{
  maxttl = mttl;
  maxent = ment;

  skipcache_reset();
  SEMALLOC(semaphore);
  SEMINIT(semaphore);
  TIMEOUT(skipcache_timeout, NULL, TIMERINTERVAL);

  skip_n = GMTSECONDS;
  skip_n /= 3600;
  skip_n -= (25 * 365 + 6) * 24;    /* 25 years + 6 leap years */

  return 0;
}

/*
 * free all entries in hash table
 */
int skipcache_exit(void)
{
  UNTIMEOUT(skipcache_timeout, NULL);
  skipcache_flush();
  SEMFREE(semaphore);

  return 0;
}

int skipcache_listcache(u_char *buf, int len)
{
  struct skipreq_listcache *l = (struct skipreq_listcache *)buf;
  struct skipcache *c;
  int i,n;

  if (len < sizeof(struct skipreq_listcache)) return -1;

  SEMLOCK(semaphore);

  l->cache_maxentries=maxent;
  for(i=0, n=0, l->cache_entries=0; i<HASHTBLSIZE; i++) {
      for(l->hash_depth[i]=0, c=tbl[i]; c; c=c->next) {
          l->hash_depth[i]++;
	  l->cache_entries++;
	  if (((u_char *)&(l->entry[n+1])) - buf <= len) {
	      MEMCPY(&(l->entry[n]),&(c->id), sizeof(skip_id_t));
	      n++;
	  }
      }
  }

  SEMUNLOCK(semaphore);

  return len;
}
