/*
   Copyright (C) 1996-1997 Robert Muchsel

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Please address all correspondence concerning the software to
   muchsel@acm.org.
*/

#include "config.h"
#include <asm/system.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/firewall.h>
#include <linux/ip.h>
#include "skip_defs.h"
#include "dynamic.h"
#include "memblk.h"
#include "skipcache.h"
#include "queue.h"
#include "interface.h"

/* #define DEBUG_LINUX_QUEUE */

/* Puts a packet into the ip address specific queue. We simply use
   a Linux skb queue. */
int queue_enqueuein(struct skipcache *c, void *interface, void *header,
		    void *pkt)
{
  struct _kernel_queue *q = (struct _kernel_queue *) (c->data + c->q.offset);
  QUEUE *Q = &q->inq;
  SEMTYPE s;
  struct queue *tmp;
  int len = 0;
  int retval = 0;

#ifdef DEBUG_LINUX_QUEUE
  struct iphdr *ip = ((struct sk_buff *) pkt)->ip_hdr;

  printf("queue %p: enqueue_in skb=%p ip_hdr=%p srcip=%08x dstip=%08x\n",
         q, pkt, ip, ip->saddr, ip->daddr);
#endif
  tmp = kmalloc(sizeof (*tmp), GFP_ATOMIC);
  tmp->next = NULL;
  tmp->pskb = (struct sk_buff *) pkt;
  SEMLOCK(s);
  if (Q->first == NULL) {
    Q->first = Q->last = tmp;
    Q->len = 1;
  }
  else {
    len = Q->len;
    if (len < MAXQLEN) {
      Q->last->next = tmp;
      Q->last = tmp;
      Q->len++;
    }
  }
  SEMUNLOCK(s);

  if (len >= MAXQLEN) {
#ifdef DEBUG_LINUX_QUEUE
    printf("queue overflow: free skb=%p queue=%p\n", pkt, tmp);
#endif
    kfree(tmp);
    kfree_skb((struct sk_buff *) pkt, FREE_WRITE);

    retval = -1;
  }

  return retval;
}

/* Same for the output queue */
int queue_enqueueout(struct skipcache *c, void *interface, void *header,
		     void *pkt)
{
  struct _kernel_queue *q = (struct _kernel_queue *) (c->data + c->q.offset);
  QUEUE *Q = &q->outq;
  SEMTYPE s;
  struct queue *tmp;
  int len = 0;
  int retval = 0;

#ifdef DEBUG_LINUX_QUEUE
  struct iphdr *ip = ((struct sk_buff *) pkt)->ip_hdr;

  printf("queue %p: enqueue_out skb=%p ip_hdr %p srcip=%08x dstip=%08x\n", 
         q, pkt, ip, ip->saddr, ip->daddr);
#endif

  tmp = kmalloc(sizeof (struct queue), GFP_ATOMIC);
  tmp->next = NULL;
  tmp->pskb = (struct sk_buff *) pkt;
  SEMLOCK(s);
  if (Q->first == NULL) {
    Q->first = Q->last = tmp;
    Q->len = 1;
  }
  else {
    len = Q->len;
    if (len < MAXQLEN) {
      Q->last->next = tmp;
      Q->last = tmp;
      Q->len++;
    }
  }
  SEMUNLOCK(s);

  if (len >= MAXQLEN) {
#ifdef DEBUG_LINUX_QUEUE
    printf("queue overflow: free skb=%p queue=%p\n", pkt, tmp);
#endif
    kfree(tmp);
    kfree_skb((struct sk_buff *) pkt, FREE_WRITE);

    retval = -1;
  }

  return retval;
}

/* Free the queue for a given IP address */
int queue_free(void *cptr)
{
  struct skipcache *c = (struct skipcache *) cptr;
  struct _kernel_queue *q = (struct _kernel_queue *) (c->data + c->q.offset);
  struct queue *tmp;
  QUEUE *Q = &q->outq;
  SEMTYPE s;

#ifdef DEBUG_LINUX_QUEUE
  printf("queue %p: queue_free\n", q);
#endif

  while (Q->first != NULL) {
    SEMLOCK(s);
    tmp = Q->first;
    Q->first = tmp->next;
    Q->len--;
    SEMUNLOCK(s);
#ifdef DEBUG_LINUX_QUEUE
    printf("queue_free: free skb=%p queue=%p\n", tmp->pskb, tmp);
#endif
    kfree_skb(tmp->pskb, FREE_WRITE);
    kfree(tmp);
  }

  Q = &q->inq;
  while (Q->first != NULL) {
    SEMLOCK(s);
    tmp = Q->first;
    Q->first = tmp->next;
    Q->len--;
    SEMUNLOCK(s);
#ifdef DEBUG_LINUX_QUEUE
    printf("queue_free: free skb=%p queue=%p\n", tmp->pskb, tmp);
#endif
    kfree_skb(tmp->pskb, FREE_WRITE);
    kfree(tmp);
  }

#ifdef DEBUG_LINUX_QUEUE
  printf("queue %p: queue_free ends\n", q);
#endif
 
  return 0;
}

/* Feed all skbs for a given IP address back into processing */
int queue_feed(void *cptr)
{
  struct skipcache *c = (struct skipcache *) cptr;
  struct _kernel_queue *q = (struct _kernel_queue *) (c->data + c->q.offset);
  struct queue *tmp;
  QUEUE *Q = &q->inq;
  SEMTYPE s;

#ifdef DEBUG_LINUX_QUEUE
  printf("queue %p: queue_feed\n", q);
#endif

  while (Q->first != NULL) {
    SEMLOCK(s);
    tmp = Q->first;
    Q->first = tmp->next;
    Q->len--;
    SEMUNLOCK(s);
#ifdef DEBUG_LINUX_QUEUE
    printf("queue_feed: feed_in %p free queue=%p\n", tmp->pskb, tmp);
#endif
    interface_feed_in(tmp->pskb);
    kfree(tmp);
  }

  Q = &q->outq;
  while (Q->first != NULL) {
    SEMLOCK(s);
    tmp = Q->first;
    Q->first = tmp->next;
    Q->len--;
    SEMUNLOCK(s);
#ifdef DEBUG_LINUX_QUEUE
    printf("queue_feed: feed_out %p free queue=%p\n", tmp->pskb, tmp);
#endif
    interface_feed_out(tmp->pskb);
    kfree(tmp);
  }

#ifdef DEBUG_LINUX_QUEUE
  printf("queue %p: queue_feed ends\n", q);
#endif

  return 0;
}

/* Get IP addresses for a specific queue */
int queue_getips(struct skipcache *c, u_char *srcip, u_char *dstip)
{
  int result = -1;
  struct _kernel_queue *q = (struct _kernel_queue *) (c->data + c->q.offset);
  struct sk_buff *m;
  SEMTYPE s;

#ifdef DEBUG_LINUX_QUEUE
  printf("queue %p: queue_getips", q);
#endif

  SEMLOCK(s);
  if ((&q->inq)->first != NULL)
    m = (&q->inq)->first->pskb;
  else
    if ((&q->outq)->first != NULL)
      m = (&q->outq)->first->pskb;
    else
      m = NULL;

  if (m && m->ip_hdr) {
    *((__u32 *) srcip) = m->ip_hdr->saddr;
    *((__u32 *) dstip) = m->ip_hdr->daddr;
    result = 0;
  }
  else {
    *((__u32 *) srcip) = 0;
    *((__u32 *) dstip) = 0;
  }
  SEMUNLOCK(s);

#ifdef DEBUG_LINUX_QUEUE
  printf(": skb=%p srcip=%08x dstip=%08x result=%i\n", 
         m, *((__u32 *) srcip), *((__u32 *) dstip), result);
#endif

  return result;
}


/* initialize input/output queues */
int queue_initqueues(struct skipcache *c)
{
  struct _kernel_queue *q = (struct _kernel_queue *) (c->data + c->q.offset);

#ifdef DEBUG_LINUX_QUEUE
  printf("queue %p: queue_init\n", q);
#endif

  MEMZERO(&q->outq, sizeof(QUEUE));
  MEMZERO(&q->inq, sizeof(QUEUE));

  return 0;
}

/* init the queue module */
int queue_init(void)
{
  return 0;
}

/* de-init the queue module */
int queue_exit(void)
{
  return 0;
}
