/*
   Copyright (C) 1996-1997 Robert Muchsel

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Please address all correspondence concerning the software to
   muchsel@acm.org.
*/

#include "config.h"
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/fcntl.h>
#include <linux/errno.h>
#include <linux/major.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/firewall.h>
#include <linux/ip.h>
#include <net/protocol.h>
#include <net/sock.h>
#include <linux/ipsec.h>
#include "skip_defs.h"
#include "dynamic.h"
#include "memblk.h"
#include "skipcache.h"
#include "random.h"
#include "ipsp.h"
#include "crypt.h"
#include "sign.h"
#include "interface.h"
#include "queue.h"
#include "com.h"
#include "device.h"
#include "timeout.h"

int init_module(void);
void cleanup_module(void);


/* call all init functions */
static void skipmod_init(void)
{
  crypt_init();
  sign_init();
  random_init();
  queue_init();
  skipcache_init(SKIPCACHE_MAXTTL, SKIPCACHE_MAXENTRIES);
  ipsp_init();
  interface_init();
}

/* call all de-init functions */
static void skipmod_exit(void)
{
  interface_exit();
  ipsp_exit();
  skipcache_exit();
  queue_exit();
  random_exit();
  sign_exit();
  crypt_exit();
  timeout_exit();
}

/* hook for incoming packets */
struct firewall_ops skip_in =
{
  NULL,
  packet_accept,
  input_packet,
  packet_accept,
  PF_INET,
  10 /* make sure filter is called before real firewalls */
};

/* hook for outgoing packets */
struct firewall_ops skip_out =
{
  NULL,
  forward_packet,
  packet_nop,
  output_packet,
  PF_IPSEC,
  2  /* allow other filters to hook before us */
};

/* device switching table */
static struct file_operations skip_fops =
{
  NULL,                         /* lseek */
  NULL,                     	/* read */
  NULL,                     	/* write */
  NULL,                         /* readdir */
  NULL,                     	/* select */
  ioctl_skip,                   /* ioctl */
  NULL,                         /* mmap */
  open_skip,                	/* open code */
  close_skip,               	/* release code */
  NULL                          /* fsync */
};


/* initialize the loadable module */
int init_module(void)
{
  if (register_chrdev(SKIP_DEV_MAJOR, SKIP_DEV_NAME, &skip_fops)) {
    printk("Unable to get major %d for ENskip device!\n", SKIP_DEV_MAJOR);
    return -EIO;
  }

  if (register_firewall(PF_INET, &skip_in) ||
      register_firewall(PF_IPSEC, &skip_out)) {
    printk("Unable to register ENskip as packet filter!\n");

    unregister_chrdev(SKIP_DEV_MAJOR, SKIP_DEV_NAME);
    return -EIO;
  }

  skipmod_init();

  printk(KERN_INFO "enskip: module loaded.\n");

  return 0;
}

/* unload the loadable module */
void cleanup_module(void)
{
  if (MOD_IN_USE)
    printk(KERN_INFO "ENskip: busy - remove delayed\n");
  else {
    skipmod_exit();

    unregister_chrdev(SKIP_DEV_MAJOR, SKIP_DEV_NAME);

    unregister_firewall(PF_IPSEC, &skip_out);
    unregister_firewall(PF_INET, &skip_in);

    printk(KERN_INFO "enskip: module unloaded.\n");
  }
}
