/*
 * Copyright (C) 1995 M. Hauber, Ch. Schneider, G. Caronni
 * See COPYING for more details
 */
#include "config.h"

#include <errno.h>
#include <sys/socket.h>
#include <machine/cpu.h>
#include <net/netisr.h>
#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <sys/domain.h>
#include <sys/protosw.h>

#include "skip_defs.h"
#include "dynamic.h"
#include "memblk.h"
#include "skipcache.h"
#include "ipsp.h"
#include "skip.h"
#include "ipsum.h"
#include "interface.h"

#ifdef __GNUC__
#ident "$Id: interface.c,v 1.7 1996/04/25 15:02:21 cschneid Exp $"
#else
static char rcsid[] = "$Id: interface.c,v 1.7 1996/04/25 15:02:21 cschneid Exp $";
#endif

/* Avoid unnecessary softints for ipintrq, ipintr is looping anyway */
#define AVOID_SOFTINTS

/* XXX: Duplicated in queue.c, should be moved to some .h */
struct interface_fbparam
{
  struct ifnet *ifp;
  struct sockaddr_in address;
  struct rtentry *rt;
};

static int interface_output(struct ifnet *ifp, struct mbuf *m,
                            struct sockaddr *address, struct rtentry *rt);

static void(*oldinput)(struct mbuf *, int);

struct interface
{
  u_char IPaddr[IPADDRSIZE];
  struct ifnet *ifp;			/* interface identifier */

  /* function handles */
  int (*output)(struct ifnet *ifp, struct mbuf *m, struct sockaddr *sa,
                struct rtentry *rt);

  struct interface *next, *prev;
};

static struct interface *head = NULL;

static struct ifnet *interface_find(u_char *IPaddr)
{
  struct in_ifaddr *in_ifa;

#if NetBSD < 199511     /* Release before NetBSD 1.1 */
  for (in_ifa = in_ifaddr; in_ifa; in_ifa = in_ifa->ia_next)
#else                   /* NetBSD 1.1 or newer */
  for (in_ifa = in_ifaddr.tqh_first; in_ifa; in_ifa = in_ifa->ia_list.tqe_next)
#endif
  {
    struct sockaddr_in *sin = (struct sockaddr_in *)&in_ifa->ia_addr;

    if (in_ifa->ia_ifp->if_output == interface_output)
      continue;  /* Only interfaces not already patched by SKIP */

    if ((sin->sin_family == AF_INET) &&
        (MEMCMP(&sin->sin_addr, IPaddr, 4) == 0))
    {
      break;
    }
  }

  return in_ifa ? in_ifa->ia_ifp : NULL;
}


static struct interface *interface_search(u_char *IPaddr)
{
  struct interface *walk;

  for (walk = head; walk; walk = walk->next)
    if (MEMCMP(walk->IPaddr, IPaddr, 4) == 0)
      break;

  return walk;
}

static struct interface *interface_searchifp(struct ifnet *ifp)
{
  struct interface *walk;

  for (walk = head; walk; walk = walk->next)
    if (ifp == walk->ifp)
      break;

  return walk;
}

/* Make first <size> bytes contiguous, only works for size < cluster size */
static struct mbuf *interface_pullup(struct mbuf *m, int size)
{
  struct mbuf *newm = m, *next;

  if (m->m_len < size)  /* Need to pullup? */
  {
    MGETHDR(newm, M_DONTWAIT, MT_HEADER);

    if (newm)
    {
      M_COPY_PKTHDR(newm, m);

      if (size > MHLEN)
        MCLGET(newm, M_DONTWAIT);

      if ((size <= MHLEN) || (newm->m_flags & M_EXT))
      {
        u_char *dst = mtod(newm, u_char *);

        newm->m_len = 0;

        while (m && (size > 0))
        {
          int len = (m->m_len < size) ? m->m_len : size;

          MEMCPY(dst, mtod(m, u_char *), len);
          dst += len;
          newm->m_len += len;
          size -= len;

          m_adj(m, len);

          if (m->m_len == 0)	/* Else size == 0, don't advance m */
          {
            MFREE(m, next);
            m = next;
          }
        }

        newm->m_next = m;
      }
      else
      {
        MFREE(newm, next);
        newm = NULL;
      }
    }
  }

  return newm;
}

static int mbuf2memblk(struct mbuf *m, struct memblk *mb, struct memseg *ms)
{
  int count = 0;

  /* Construct memblk description */
  mb->dynamic = NULL;
  mb->dynamiclen = 0;
  mb->ms = ms;
  mb->offset = 0;
  mb->len = 0;


  while (m)  /* More than one mbuf? */
  {
    int ms_len = m->m_len;

    if (ms_len > 0)  /* Ignore empty mbufs */
    {
      if (count == (STATIC_MEMSEGS - 2))  /* Need dynamic mseg :-\ */
      {
        struct mbuf *t;
        int len;

        for (len = STATIC_MEMSEGS * sizeof(*ms), t = m; t; t = t->m_next)
          len += sizeof(struct memseg);

        mb->dynamiclen = len;
        if ((mb->dynamic = KALLOC(len)) == NULL)
          return -1;

        MEMCPY(mb->dynamic, mb->ms, count * sizeof(struct memseg));
        mb->ms = mb->dynamic;
        ms = mb->dynamic + count;
      }

      ms->ptr = mtod(m, u_char *);
      ms->len = ms_len;
      mb->len += ms_len;
      ms++;
      count++;
    }

    m = m->m_next;
  }

  mb->freems = ms;      /* First free memseg, ESP uses this internally */

  return 0;
}


static void interface_doinput(struct mbuf *m)
{
  int result = SKIP_DISCARD;
  struct mbuf *newm = m;
  struct memblk oldmb, newmb;
  struct memseg oldms[STATIC_MEMSEGS], newms[STATIC_MEMSEGS];

  MEMZERO(&oldmb, sizeof(oldmb));
  MEMZERO(&newmb, sizeof(newmb));
  newmb.ms = newms;

  /* Ensure IP header contiguous, hopefully mostly a no-op */
  if ((newm = interface_pullup(m, sizeof(struct ip))))
  {
    m = newm;

    if ((newm = interface_pullup(m, ipsp_getheadersize(mtod(m, void *)))))
    {
      m = newm;

      if (mbuf2memblk(m, &oldmb, oldms) == 0)
      {
        result = skip_process(SKIP_INPUT, NULL, NULL, (void **)&newm, &oldmb,
                              &newmb);
      }
    }
  }

  switch (result)
  {
    case SKIP_DISCARD:
    {
      m_freem(m);
      if (newm != m)  /* Buffer already allocated before failure? */
        m_freem(newm);
      break;
    }

    case SKIP_PROCESSED:
    {
      /* Feed to IP again */
      struct ifqueue *ifq = &ipintrq;
      int old;

      if (newm != m)  /* Newly allocated buffer used? */
      {
        struct memseg *ms;

        m_freem(m);

        /* Adjust length fields of newly created packet */
        newm->m_pkthdr.len = newmb.len;

        m = newm;
        ms = newmb.ms;

        while (m)
        {
          m->m_data = ms->ptr + newmb.offset;
          m->m_len = (ms->len < newmb.len) ? ms->len : newmb.len;

          newmb.len -= m->m_len;
          newmb.offset = 0;
          ms++;
          m = m->m_next;
        }
      }

      old = splimp();
      if (IF_QFULL(ifq)) 
      {
        m_freem(newm);
        IF_DROP(ifq);
      } 
      else
      {
        IF_ENQUEUE(ifq, newm);
#ifndef AVOID_SOFTINTS
        schednetisr(NETISR_IP);
#endif
      }
      splx(old);
      break;
    }

    case SKIP_QUEUED:
    {
      if (newm != m)  /* Buffer already allocated before failure? */
        m_freem(newm);
      break;
    }
  }

  if (oldmb.dynamic)
    KFREE(oldmb.dynamic, oldmb.dynamiclen);
  if (newmb.dynamic)
    KFREE(newmb.dynamic, newmb.dynamiclen);
}

static void interface_input(struct mbuf *m, int hlen)
{
  struct ip *ip = mtod(m, struct ip *);

  if (ip->ip_p == IPPROTO_SKIP)
  {
    /* Undo changes ip_input did to our packet :-\ */
    ip->ip_len += hlen;
    ip->ip_len = SKIP_HTONS(ip->ip_len);
    ip->ip_id = SKIP_HTONS(ip->ip_id);
    ip->ip_off = SKIP_HTONS(ip->ip_off);
    ip->ip_sum = SKIP_HTONS(ipsum(ip));

#if 0
    {
    struct mbuf *t;

    printf(">>interface_input: pkt =");

    for (t = m; t; t = t->m_next)
    {
      int i;
      u_char *p = mtod(t, u_char *);

      for (i = 0; i < m->m_len; i++)
        printf(" %02x", p[i]);

      printf("\n");
    }
    }
#endif
    interface_doinput(m);
  }
  else
    oldinput(m, hlen);
}

static int interface_output(struct ifnet *ifp, struct mbuf *m,
                            struct sockaddr *address, struct rtentry *rt)
{
  int result;
  struct interface *i = interface_searchifp(ifp);
  struct sockaddr_in *sin = (struct sockaddr_in *)address;
  struct memblk oldmb, newmb;
  struct memseg oldms[STATIC_MEMSEGS], newms[STATIC_MEMSEGS];

  if (i == NULL)
  {
    m_freem(m);
    return -ENOENT;
  }

  MEMZERO(&oldmb, sizeof(oldmb));
  MEMZERO(&newmb, sizeof(newmb));
  newmb.ms = newms;

  if (sin->sin_family == AF_INET)
  {
    struct mbuf *newm = m;

    result = SKIP_DISCARD;

    /* Ensure IP header contiguous, hopefully mostly a no-op */
    if ((newm = interface_pullup(m, sizeof(struct ip))))
    {
      m = newm;

      if ((newm = interface_pullup(m, ipsp_getheadersize(mtod(m, void *)))))
      {
        m = newm;

        if (mbuf2memblk(m, &oldmb, oldms) == 0)
        {
          struct interface_fbparam fb;

          fb.ifp = ifp;
          fb.address = *sin;
          fb.rt = rt;

          result = skip_process(SKIP_OUTPUT, &fb, NULL, (void **)&newm, &oldmb,
                                &newmb);
        }
      }
    }

    switch (result)
    {
      default: case SKIP_DISCARD:
      {
        m_freem(m);
        if (newm != m)  /* Buffer already allocate before failure? */
          m_freem(newm);

        result = 0;
        break;
      }

      case SKIP_PROCESSED:
      {
        if (newm != m)
        {
          struct memseg *ms;

          m_freem(m);

          /* Adjust length fields of newly created packet */
          newm->m_pkthdr.len = newmb.len;

          m = newm;
          ms = newmb.ms;

          while (m)
          {
            m->m_data = ms->ptr + newmb.offset;
            m->m_len = (ms->len < newmb.len) ? ms->len : newmb.len;
  
            newmb.len -= m->m_len;
            newmb.offset = 0;
            ms++;
            m = m->m_next;
          }
        }

        result = i->output(ifp, newm, address, rt);
        break;
      }

      case SKIP_QUEUED:
      {
        if (newm != m)  /* Buffer already allocate before failure? */
          m_freem(newm);
        result = 0;
        break;
      }
    }  
  }
  else
    result = i->output(ifp, m, address, rt);

  if (oldmb.dynamic)
    KFREE(oldmb.dynamic, oldmb.dynamiclen);
  if (newmb.dynamic)
    KFREE(newmb.dynamic, newmb.dynamiclen);

  return result;
}

int interface_outfeedback(struct mbuf *m)
{
  struct mbuf *pkt;
  struct interface_fbparam *fb = mtod(m, struct interface_fbparam *);
  struct ifnet *ifp = fb->ifp;
  struct sockaddr_in address = fb->address;
  struct rtentry *rt = fb->rt;

  MFREE(m, pkt);

  return interface_output(ifp, pkt, (struct sockaddr *)&address, rt);
}

int interface_infeedback(struct mbuf *m)
{
  interface_doinput(m);
#ifdef AVOID_SOFTINTS
  schednetisr(NETISR_IP);
#endif

  return 0;
}

/* Parameter mp is assumed to point to old pkt, size is assumed to be > 0 */
int interface_getbuf(struct memblk *m, void **mp, int size)
{
  int result = -ENOMEM;
  struct mbuf *t, *new;

  m->len = size;
  MGETHDR(t, M_DONTWAIT, MT_HEADER);	/* Allocate packet header */

  if ((new = t))
  {
    struct memseg *ms = m->ms;

    M_COPY_PKTHDR(t, ((struct mbuf *)*mp));	/* Copy flags from old pkt */

    if (MHLEN < size)		/* Need cluster? */
      MCLGET(t, M_DONTWAIT);

    if ((MHLEN >= size) || (t->m_flags & M_EXT))	/* Got enough? */
    {
      ms->len = t->m_ext.ext_size;

      /*
       * Note: Scheme to ensure last mbuf contains enough data for ESP.
       * Assumes clusters are of equal size, last two are equally used.
       */
      if (ms->len >= size)			/* Fits in last mbuf */
        ms->len = size;
      else if (ms->len > (size - ms->len))	/* Second last? */
        ms->len = size / 2;

      size -= ms->len;
      ms->ptr = mtod(t, u_char *);
      ms++;

      while (size > 0)		/* Get as many clusters as needed */
      {
        MGET(t, M_DONTWAIT, MT_DATA);

        if (t)
        {
          t->m_next = new->m_next;	/* Insert behind packet header */
          new->m_next = t;

          if (MLEN < size)	/* Need cluster? */
            MCLGET(t, M_DONTWAIT);

          if ((MLEN >= size) || (t->m_flags & M_EXT))	/* Got enough? */
          {
            ms->len = t->m_ext.ext_size;

            /* Note: Scheme for last two mbufs, see comment above */
            if (ms->len >= size)			/* Fits in last mbuf */
              ms->len = size;
            else if (ms->len > (size - ms->len))	/* Second last? */
              ms->len = size / 2;

            size -= ms->len;
            ms->ptr = mtod(t, u_char *);
            ms++;
          }
          else
            break;	/* Could not get cluster for mbuf */
        }
        else
          break;	/* Could not get mbuf */
      }
    }

    if (size <= 0)	/* Everything ok? */
    {
      m->offset = 0;
      m->freems = ms;

      new->m_flags &= ~M_EOR;	/* Move End Of Record to last mbuf */
      t->m_flags |= M_EOR;

      *mp = new;	/* Store newly allocated mbuf */

      result = 0;
    }
    else
      m_freem(new);
  }

  return result;
}

int interface_attach(void *interface, u_char *IPaddr)
{
  struct ifnet *ifn;
  struct interface *new;
  int c;

  if (interface_search(IPaddr))
    return -EEXIST;

  for (c = 0; (ifn = interface_find(IPaddr)); c++)
  {
    if ((new = KALLOC(sizeof(*new))) == NULL)
      return -ENOMEM;

    MEMZERO(new, sizeof(*new));
    MEMCPY(new->IPaddr, IPaddr, 4);
    new->ifp = (struct ifnet *)ifn;
    new->output = ifn->if_output;

    ifn->if_output = interface_output;
    ifn->if_mtu -= ipsp_maxheadergrowth();

    new->next = head;
    new->prev = NULL;
    if (head)
      head->prev = new;
    head = new;
  }

  return c ? 0 : -ENOENT;
}

static int interface_remove(struct interface *i)
{
  struct ifnet *ifn;

  ifn = (struct ifnet *)i->ifp;

  ifn->if_output = i->output;
  ifn->if_mtu += ipsp_maxheadergrowth();

  if (i->next)
    i->next->prev = i->prev;
  if (i->prev)
    i->prev->next = i->next;
  else
    head = i->next;

  KFREE(i, sizeof(*i));
  return 0;
}

int interface_detach(void *interface, u_char *IPaddr)
{
  struct interface *i;
  int c;

  for (c = 0; (i = interface_search(IPaddr)); c++)
    interface_remove(i);

  return c ? 0 : -ENOENT;
}

int interface_init()
{
  int result = ENFILE;
  struct protosw *pr;

  if ((pr = pffindproto(AF_INET, IPPROTO_RAW, SOCK_RAW)))
  {
    oldinput = pr->pr_input;
    pr->pr_input = interface_input;
    result = 0;
  }

  head = NULL;

  return result;
}

int interface_exit()
{
  struct interface *walk, *next;
  struct protosw *pr;

  for (walk = head; walk; walk = next)
  {
    next = walk->next;
    interface_remove(walk);
  }

  if ((pr = pffindproto(AF_INET, IPPROTO_RAW, SOCK_RAW)))
  {
    if (pr->pr_input == interface_input)
      pr->pr_input = oldinput;
  }

  head = NULL;

  return 0;
}
