/*
 * Copyright (C) 1995 M. Hauber, Ch. Schneider, G. Caronni
 * See COPYING for more details
 */
#include "config.h"
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/conf.h>
#include <sys/mount.h>
#include <sys/exec.h>
#include <sys/lkm.h>
#include <sys/file.h>
#include <sys/errno.h>
#include <netinet/in.h>

#include "skip_defs.h"
#include "dynamic.h"
#include "memblk.h"
#include "skipcache.h"
#include "random.h"
#include "ipsp.h"
#include "crypt.h"
#include "sign.h"
#include "interface.h"
#include "queue.h"
#include "com.h"

#ifdef __GNUC__
#ident "$Id: skipmod.c,v 1.7 1996/04/25 15:02:21 cschneid Exp $"
#else
static char rcsid[] = "$Id: skipmod.c,v 1.7 1996/04/25 15:02:21 cschneid Exp $";
#endif

extern int com_skipcall();

/* New system calls */
static struct sysent newent = 
{
  2,				/* # of args */
#if NetBSD >= 199511		/* NetBSD 1.1 or newer */
  sizeof(struct skipcallparam),	/* size of args */
#endif
  com_skipcall			/* function pointer*/
};

MOD_SYSCALL("skipmod", -1, &newent)

static int skipmod_init(void)
{
  crypt_init();
  sign_init();
  random_init();
  queue_init();
  skipcache_init(SKIPCACHE_MAXTTL, SKIPCACHE_MAXENTRIES);
  ipsp_init();
  interface_init();

  return 0;
}

static int skipmod_exit(void)
{
  interface_exit();
  ipsp_exit();
  skipcache_exit();
  queue_exit();
  random_exit();
  sign_exit();
  crypt_exit();

  return 0;
}

/*
 * This function is called each time the module is loaded.   Technically,
 * we could have made this "lkm_nofunc" in the "DISPATCH" in "newsyscall()",
 * but it's a convenient place to kick a copyright out to the console.
 */
static int skipmod_handle(struct lkm_table *lkmtp, int cmd)
{
  extern int lkmexists(struct lkm_table *);
  int err = 0;            /* default = success */

  switch (cmd) 
  {
    case LKM_E_LOAD:
    {
      /*
       * Don't load twice! (lkmexists() is exported by kern_lkm.c)
       */
      if (lkmexists(lkmtp))
      {
        err = EEXIST;
        break;
      }

      if ((err = interface_init()))
        break;  /* Could not initialize interface - voodoo failed :-\ */
      skipmod_init();

      /* if we make it to here, print copyright on console*/
      printf( "Skipmod loaded\n");
      break;    /* Success*/
    }

    case LKM_E_UNLOAD:
    {
      skipmod_exit();
      break;    /* Success*/
    }

    default:  /* we only understand load/unload*/
    {
      err = EINVAL;
      break;
    }
  }

  return err;
}


/*
 * External entry point; should generally match name of .o file.  The
 * arguments are always the same for all loaded modules.  The "load",
 * "unload", and "stat" functions in "DISPATCH" will be called under
 * their respective circumstances unless their value is "nosys".  If
 * called, they are called with the same arguments (cmd is included to
 * allow the use of a single function, ver is included for version
 * matching between modules and the kernel loader for the modules).
 *
 * Since we expect to link in the kernel and add external symbols to
 * the kernel symbol name space in a future version, generally all
 * functions used in the implementation of a particular module should
 * be static unless they are expected to be seen in other modules or
 * to resolve unresolved symbols alread existing in the kernel (the
 * second case is not likely to ever occur).
 *
 * The entry point should return 0 unless it is refusing load (in which
 * case it should return an errno from errno.h).
 */
int skipmod(struct lkm_table *lkmtp, int cmd, int ver)
{
  extern int lkmdispatch(struct lkm_table *, int);

#if NetBSD < 199511	/* Release before NetBSD 1.1 */
  DISPATCH(lkmtp, cmd, ver, skipmod_handle, skipmod_handle, nosys)
#else			/* NetBSD 1.1 or newer */
  DISPATCH(lkmtp, cmd, ver, skipmod_handle, skipmod_handle, lkm_nofunc)
#endif
}
