/**
 * Ask the skip kernel module for a list of its Primary Cache entries
 *
 * @author Marcel Waldvogel
 */

import java.io.IOException;
import java.util.Hashtable;
import java.util.Enumeration;

public class SkipCacheEntryList extends SkipIoctl {
  /**
   * This is some static #define in some #include file.
   * SkipInterface.c checks whether both are identical.
   */
  public static final int HASHTBLSIZE = 73;
  /**
   * This dynamic size limit is passed to the kernel module at
   * initialization time to tell it, how many entries it should keep
   * in the primary cache. Here, it is being used to specify how much
   * space should be reserved for return values.
   *  
   * This is the maximum retrievable number of entries, should be >=
   * the limit the kernel module got.
   */
  public static final int CACHEENTRIES = 64;
  /**
   * The number of entries that could fit into the cache at once
   */
  int maxEntries;
  /**
   * The number of entries that are currently in the cache
   */
  int currentEntries;
  /**
   * The number of entries that were returned with the last call (this
   * currently is MIN(currentEntries, CACHEENTRIES))
   */
  int returnedEntries;
  /**
   * The maximum length and the count of non-null hash queues
   */
  int maximumDepth, nonZeros;
  /**
   * The depth of the hash queues
   */
  int hashDepth[] = new int[HASHTBLSIZE];
  /**
   * The entries
   */
  String entries[] = new String[CACHEENTRIES];
  
  /**
   * Fill the structure with the current stats
   */
public SkipCacheEntryList() throws IOException {
  if (comCacheEntryList() < 0)
    throw new IOException("Cannot read Skip cache statistics");
  /*
    // Debugging
    System.err.println(new Integer(maxEntries).toString());
    System.err.println(new Integer(curEntries).toString());
    for (int i = 0; i < HASHTBLSIZE; i++)
      System.err.print(new Integer(hashDepth[i]).toString());
      System.err.println("");
      for (int i = 0; i < curEntries && i < CACHEENTRIES; i++)
        System.err.println(entries[i]);
  */

  // Some statistics
  for (int i = 0; i < HASHTBLSIZE; i++) {
    if (hashDepth[i] != 0)
      nonZeros++;
    if (hashDepth[i] > maximumDepth)
      maximumDepth = hashDepth[i];
  }
}

  /**
    Fills the instance variable with the current stats
   */
private native int comCacheEntryList();

  /**
   * Returns a hashtable containing name/value pairs for all statistics
   */
public Hashtable values() {
  Hashtable h = new Hashtable();
  h.put("Entries Cached", new Long(currentEntries));
  h.put("Entries Max", new Long(maxEntries));
  h.put("Entries Returned", new Long(returnedEntries));
  h.put("Hash Queue Length Max", new Long(maximumDepth));
  h.put("Hash Queue Length Average", new Double((double)currentEntries
                                                / (double)nonZeros));
  return h;
}

  /**
   * Returns the available keys
   */
public synchronized Hashtable cacheEntries() {
  Hashtable h = new Hashtable();
  for (int i = 0; i < returnedEntries; i++) {
    h.put(entries[i], entries[i]);
  }
  return h;
}

  /**
   * Returns the cache depths
   */
public int[] hashDepths() {
  int hd[] = new int[HASHTBLSIZE];
  System.arraycopy(hashDepth, 0, hd, 0, HASHTBLSIZE);
  return hd;
}

}
