/*
 * Copyright (C) 1995,96,97 M. Hauber, Ch. Schneider, G. Caronni, R. Muchsel
 * See COPYING for more details
 */

/* #define DEBUG_SKIPD */
/* #define DEBUG_SKIPD_KIJN */
/* #define DEBUG_GETKEY */	/* forces the kernel to send getkey request */
                                /* since Kijns are not valid */

#include "config.h"

#include <stdio.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>

#include "skip_defs.h"
#include "dynamic.h"
#include "skipcache.h"
#include "memblk.h"
#include "ipsp.h"
#include "com.h"
#include "parse.h"
#include "crypt.h"
#include "../lib/signmd5.h"
#include "random.h"
#include "req.h"
#include "cert.h"

#ifdef __GNUC__
#ident "$Id: skipd.c,v 1.9 1996/06/07 12:00:43 skip Exp $"
#else
static char rcsid[] = "$Id: skipd.c,v 1.9 1996/06/07 12:00:43 skip Exp $";
#endif


static int getreqflag;
static int saveconfigflag;
static int reinitflag;
static int showcacheflag;
static int certinitflag;

static FILE *log;

static struct skipcache *def;	/* default entry */

void signal_handler(int sig)
{
#ifndef __USE_BSD_SIGNAL
  signal(sig, signal_handler);
#endif

  switch (sig) {
    case SIGURG:
      getreqflag++;
      break;
    case SIGUSR1:
      saveconfigflag = 1;
      break;
    case SIGUSR2:
      showcacheflag = 1;
      break;
    case SIGHUP:
      reinitflag = 1;
      break;
  }
}

static void usage(char *name)
{
  fprintf(stderr, "Usage: %s [-fhv] [-c configfile]\n", name);
  fprintf(stderr, " -c <file>  alternative configuration file (default is " CONFIGFILE ")\n");
  fprintf(stderr, " -f         flush primary (kernel) cache\n");
  fprintf(stderr, " -v         verbose, print a message for each lookup on stdout\n");
  fprintf(stderr, " -k <path>  path to certificate directory (default is " CERTDIR ")\n");
  fprintf(stderr, " -d         become a daemon and disconnect from terminal\n");
  fprintf(stderr, " -h         this help text\n");
  exit(1);
}

static int skipd_flush(void)
{
  int result = -1;
  struct skipreq_hdr hdr;

#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: skipd_flush()\n");
#endif

  hdr.req = SKIPREQ_FLUSH;

  if (com_request((u_char *) & hdr, sizeof(hdr)) == sizeof(hdr))
    result = 0;

  return result;
}

static int skipd_register(void)
{
  int result = -1;
  struct skipreq_register r;

#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: skipd_register()\n");
#endif

  r.hdr.req = SKIPREQ_REGISTER;
  r.pid = getpid();

  if (com_request((u_char *) & r, sizeof(r)) == sizeof(r))
    result = 0;

  return result;
}

static int skipd_discover(struct skipreq_hdr *hdr, struct skipcache *c)
{
#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: skipd_discover()\n");
#endif

  /* copy id from entry if not default */
  if (c != def) 
    MEMCPY(&hdr->id, &c->id, sizeof(skip_id_t));

  cert_discover((u_long *) hdr->srcip, (u_long *) hdr->dstip, hdr->id.type[0],
		hdr->id.type[1], hdr->id.id[0], hdr->id.id[1],
		c->lookuphosts.len / IPADDRSIZE,
		(u_long *) (c->data + c->lookuphosts.offset), hdr->req);
  return 0;
}

static struct skipcache *skipd_clonedefault(struct skipreq_hdr *hdr)
{
  struct skipcache *new = NULL;

#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: skipd_clonedefault()\n");
#endif

  if ((new = KALLOC(sizeof(*def) + def->datalen))) {
    MEMCPY(new, def, sizeof(*def) + def->datalen);
    MEMCPY(&new->id, &hdr->id, sizeof(new->id));
    new->data = (u_char *) (new + 1);
    if (skipcache_update(new) < 0) {
      fprintf(log, "skipd: could not update cache.\n");
      KFREE(new, sizeof(*new) + new->datalen);
      new = NULL;
    }
  }

  return new;
}

static void skipd_calckijn(struct skipcache *c, u_int32 n)
{
  MD5_CTX context;
  u_char x;
  u_int32 netn = SKIP_HTONL(n);
  u_char tmp[2 * sizeof(context.digest)];
  int l;

#ifdef DEBUG_SKIPD_KIJN
  fprintf(log, "skipd: skipd_calckijn:\n");
  fprintf(log, "shkey-calc(%li): ", c->sharedkey.len);
  for (l = 0; l < c->sharedkey.len; l++)
    fprintf(log, "%02X", *(c->data + c->sharedkey.offset + l));
  fprintf(log, "\n");
#endif

  MD5Init(&context);
  MD5Update(&context, c->data + c->sharedkey.offset, c->sharedkey.len);
  MD5Update(&context, (u_char *) & netn, sizeof(netn));
  x = 0x01;
  MD5Update(&context, &x, sizeof(x));
  MD5Final(&context);
  MEMCPY(tmp, context.digest, sizeof(context.digest));

  MD5Init(&context);
  MD5Update(&context, c->data + c->sharedkey.offset, c->sharedkey.len);
  MD5Update(&context, (u_char *) & netn, sizeof(netn));
  x = 0x00;
  MD5Update(&context, &x, sizeof(x));
  MD5Final(&context);
  MEMCPY(tmp + sizeof(context.digest), context.digest, sizeof(context.digest));

  l = c->Kijn[n % 3].key.len;
  dynamic_copybuf2dyn(c->data, &c->Kijn[n % 3].key, tmp + sizeof(tmp) - l, l);
  c->Kijn[n % 3].n = n;
}

static int skipd_checkkijn(struct skipcache *c)
{
#ifdef DEBUG_SKIPD_KIJN
  fprintf(log, "skipd: skipd_checkkijn()\n");
#endif

  if (c->Kijn[(skip_n - 1) % 3].n != (skip_n - 1))
    skipd_calckijn(c, (skip_n - 1));
  if (c->Kijn[skip_n % 3].n != skip_n)
    skipd_calckijn(c, skip_n);
  if (c->Kijn[(skip_n + 1) % 3].n != (skip_n + 1))
    skipd_calckijn(c, (skip_n + 1));

  return 0;
}

static int skipd_updateentry(struct skipreq_hdr *old_hdr,
			     struct skipcache *c)
{
  char t[16 * 1024];
  struct skipreq_updateentry *r = (struct skipreq_updateentry *) t;

#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: skipd_updateentry()\n");
#endif

  /* is it default entry? */
  if (c == def) {
    if ((c = skipd_clonedefault(old_hdr)) == NULL)
      return -1;

#ifdef DEBUG_SKIPD
    fprintf(log, "skipd: skipd_updateentry cloning default, mapping\n");
#endif

    if (skipcache_addmapping(c, old_hdr->srcip, old_hdr->dstip) < 0) {
      skipcache_remove(&c->id);
      return -1;
    }
  }

  if (c->flags & SKIP_VALIDKEY)
    skipd_checkkijn(c);

  /* entry is in second level cache with valid id + mapping */
  MEMZERO(r, sizeof(*r));
  r->hdr.req = SKIPREQ_UPDATEENTRY;
  MEMCPY(&r->hdr.id, &c->id, sizeof(r->hdr.id));
  MEMCPY(&r->oldhdr, old_hdr, sizeof(r->oldhdr));
  MEMCPY(r->hdr.srcip, old_hdr->srcip, IPADDRSIZE);
  MEMCPY(r->hdr.dstip, old_hdr->dstip, IPADDRSIZE);

  skipcache_compressentry(c, (u_char *) &r->entry, sizeof(t));
  r->entry.flags &= ~SKIP_DONTAGE;

#ifdef DEBUG_GETKEY
  r->entry.Kijn[0].n = r->entry.Kijn[1].n = r->entry.Kijn[2].n = 0;
#endif

#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: updateentry skip_n=%i hdr.id%c==oldhdr.id srcip=%s dstip=%s\n",
    skip_n, MEMCMP(&(r->hdr.id), &(r->oldhdr.id), sizeof(skip_id_t)) ? '!' : '=',
	  inet_ntoa(*((struct in_addr *) &(r->hdr.srcip))),
	  inet_ntoa(*((struct in_addr *) &(r->hdr.dstip))));
  {
	  int walk;

	  fprintf(log, "updateentry REQUEST oldhdr.id0 %u-", r->oldhdr.id.type[0]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r->oldhdr.id.id[0][walk]);
	  fprintf(log, "\n                    oldhdr.id1 %u-", r->oldhdr.id.type[1]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r->oldhdr.id.id[1][walk]);
	  fprintf(log, "\n                       hdr id0 %u-", r->hdr.id.type[0]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r->hdr.id.id[0][walk]);
	  fprintf(log, "\n                       hdr id1 %u-", r->hdr.id.type[1]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r->hdr.id.id[1][walk]);
	  fprintf(log, "\n");
  }
#endif

  if (com_request((u_char *) r, sizeof(*r) + r->entry.datalen) < 0) {
    fprintf(log, "skipd: entry update failed.\n");
    return -1;
  }

  return 0;
}

static int skipd_updatekijn(struct skipreq_hdr *old_hdr,
			    struct skipcache *c, int keyttl)
{
  struct skipreq_updateKijn r;

#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: skipd_updatekijn()\n");
#endif

  /* is it default entry? */
  if (c == def) {
    if ((c = skipd_clonedefault(old_hdr)) == NULL)
      return -1;

    if (skipcache_addmapping(c, old_hdr->srcip, old_hdr->dstip) < 0) {
      skipcache_remove(&c->id);
      return -1;
    }
  }

  /* ok, now we have an entry in second level cache 4 sure */
  MEMZERO(&r, sizeof(r));
  r.hdr.req = SKIPREQ_UPDATEKIJN;
  MEMCPY(&r.hdr.id, &c->id, sizeof(r.hdr.id));
  MEMCPY(&r.oldhdr, old_hdr, sizeof(*old_hdr));

  if (c->flags & SKIP_VALIDKEY) {
    skipd_checkkijn(c);
    r.n = skip_n;
    if (dynamic_copydyn2buf(c->data, &c->Kijn[(skip_n - 1) % 3].key,
			r.Kijn[(skip_n - 1) % 3], sizeof(r.Kijn[0])) < 0)
      return -1;
    if (dynamic_copydyn2buf(c->data, &c->Kijn[skip_n % 3].key,
			    r.Kijn[skip_n % 3], sizeof(r.Kijn[1])) < 0)
      return -1;
    if (dynamic_copydyn2buf(c->data, &c->Kijn[(skip_n + 1) % 3].key,
			r.Kijn[(skip_n + 1) % 3], sizeof(r.Kijn[2])) < 0)
      return -1;
  }
  else
    r.n = 0;

  r.keyttl = keyttl;

#ifdef DEBUG_SKIPD
  fprintf(log, "skipd: updatekijn skip_n=%i hdr.id%c==oldhdr.id\n",
    skip_n, MEMCMP(&(r.hdr.id), &(r.oldhdr.id), sizeof(skip_id_t)) ? '!' : '=');
  {
	  int walk;

	  fprintf(log, "updatekijn  REQUEST oldhdr.id0 %u-", r.oldhdr.id.type[0]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r.oldhdr.id.id[0][walk]);
	  fprintf(log, "\n                    oldhdr.id1 %u-", r.oldhdr.id.type[1]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r.oldhdr.id.id[1][walk]);
	  fprintf(log, "\n                       hdr id0 %u-", r.hdr.id.type[0]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r.hdr.id.id[0][walk]);
	  fprintf(log, "\n                       hdr id1 %u-", r.hdr.id.type[1]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", r.hdr.id.id[1][walk]);
	  fprintf(log, "\n");
 }
#endif
  if (com_request((u_char *) &r, sizeof(r)) < 0) {
    fprintf(log, "skipd: Kijn update failed.\n");
    return -1;
  }

  return 0;
}

/* called from certificate module */
static void certificate_recv(dh_data * reply)
{
  struct skipcache *l, *l1, *l2;
  skip_id_t id;
  struct skipreq_hdr oldhdr;
  int keyttl;

  id.type[0] = reply->ons;
  id.type[1] = reply->rns;
  MEMCPY(&id.id[0], reply->omk, SKIP_ID_MAXLEN);
  MEMCPY(&id.id[1], reply->rmk, SKIP_ID_MAXLEN);

  MEMZERO(&oldhdr, sizeof(oldhdr));
  oldhdr.req = reply->ireq;
  oldhdr.id.type[0] = reply->old_ons;
  oldhdr.id.type[1] = reply->old_rns;
  MEMCPY(&oldhdr.id.id[0], reply->old_omk, SKIP_ID_MAXLEN);
  MEMCPY(&oldhdr.id.id[1], reply->old_rmk, SKIP_ID_MAXLEN);
  MEMCPY(oldhdr.srcip, reply->oip, IPADDRSIZE);
  MEMCPY(oldhdr.dstip, reply->rip, IPADDRSIZE);

  l1 = skipcache_lookup(&oldhdr.id);
  l2 = skipcache_lookup(&id);

#ifdef DEBUG_SKIPD
  fprintf(log, "certificate_recv: o.              r.\n");
  fprintf(log, "            nsid  %2d              %2d\n", id.type[0], id.type[1]);
  /* fprintf(log, "            mkid  %s       %s\n", reply->omk, reply->rmk); */
  fprintf(log, "        old nsid  %2d              %2d\n", oldhdr.id.type[0], oldhdr.id.type[1]);
  /* fprintf(log, "        old mkid  %s       %s\n", reply->old_omk, reply->old_rmk); */
  fprintf(log, "              ip  %-16s%s\n", inet_ntoa(*((struct in_addr *) &(oldhdr.srcip))),
	  inet_ntoa(*((struct in_addr *) &(oldhdr.dstip))));
  fprintf(log, "              l1=%p l2=%p\n", l1, l2);
#endif

  /* master key id change? */
  if (MEMCMP(&oldhdr.id, &id, sizeof(id))) {
#if 1
    /* XXX: this causes system panics! check later... */
    fprintf(log, "skipd: master key id changes not supported at the moment.\n");
    return;
#else
    if (l1 == l2) {
      if (l1) { /* l1 == l2 != NULL */
        l = l1;
        if (skipcache_addmapping(l, (u_char *) &reply->oip, 
                                    (u_char *) &reply->rip) < 0)
          return;
      }
      else { /* both NULL */
        /* since ids differ, the new id must be valid */
        struct skipreq_hdr tmp;

        MEMCPY(&tmp.id, &id, sizeof(tmp.id));
        l = skipd_clonedefault(&tmp);

        if (skipcache_addmapping(l, (u_char *) &reply->oip, 
                                    (u_char *) &reply->rip) < 0)
          return;
      }
    }
    else /* l1 != l2 */
      if (l2) { /* l1 != l2, l2 != NULL */
        l = l2;
        if (skipcache_addmapping(l, (u_char *) &reply->oip, 
                                    (u_char *) &reply->rip) < 0)
          return;

        /* remove old entry, if there is one */
        if (l1 && (skipcache_remove(&oldhdr.id) < 0)) {
	  fprintf(log, "skipd: could not remove entry from second level cache.\n");
	  return;
        }
      }
      else { /* l1 != NULL, l2 == NULL */
        /* we change the entry id, so we do not lose any mappings */
        if (skipcache_change(l1, &id) < 0) {
	  fprintf(log, "skipd: could not change id in second level cache.\n");
	  return;
        }
        l = l1;
        if (skipcache_addmapping(l, (u_char *) &reply->oip, 
                                    (u_char *) &reply->rip) < 0)
          return;
      }
#endif
  }
  else {			/* l1 == l2 */
    if (l2 == NULL) {
      /* new entry in 2. level cache */
      l = skipd_clonedefault(&oldhdr);
      if (skipcache_addmapping(l, (u_char *) & reply->oip, (u_char *) & reply->rip) < 0)
	return;
    }
    else
      l = l2;
  }

  if (l == NULL)
    return;

  /* now we have an entry for the given master key id */

  /* update shared secret */
  if (l->flags & SKIP_MANUALKEY)
    return;

  if (reply->secret_len > 0) {
#if 0
    if (reply->secret_len > l->sharedkey.len) {
      fprintf(log, "skipd: shared secret too long (%d, %ld bytes).\n",
	      reply->secret_len, l->sharedkey.len);
      return;
    }
#endif

    /*dynamic_copybuf2dyn(l->data, &l->sharedkey, reply->secret, reply->secret_len); */

    /* fixed len of 256 bits as of now! */
    if (reply->secret_len < 256 / 8) {
      fprintf(log, "skipd: shared secret too short (%d bytes).\n",
	      reply->secret_len);
      return;
    }

    dynamic_copybuf2dyn(l->data, &l->sharedkey,
		   reply->secret + reply->secret_len - 256 / 8, 256 / 8);
#ifdef DEBUG_SKIPD
    {
      int i;

      fprintf(log, "Shared secret: ");
      for (i = 0; i < 256 / 8; i++)
	fprintf(log, "%02X", *(reply->secret + reply->secret_len - 256 / 8 + i));
      fprintf(log, "\n");
    }
#endif

    l->flags |= SKIP_VALIDKEY;
    l->flags &= ~SKIP_NOKEY;
  }
  else {
    l->flags |= SKIP_NOKEY;
    l->flags &= ~SKIP_VALIDKEY;
  }

  keyttl = 0;			/* XXX: should be in dh_data */

  /* send reply to kernel */
  switch (oldhdr.req) {
    case SKIPREQ_GETENTRY:
      {
#ifdef DEBUG_SKIPD
	fprintf(log, "skipd: old_hdr.req=SKIPREQ_GETENTRY\n");
#endif
	l->keyttl = keyttl;
	skipd_updateentry(&oldhdr, l);
	break;
      }
    case SKIPREQ_GETENSKIPKEY:
    case SKIPREQ_GETDESKIPKEY:
      {
#ifdef DEBUG_SKIPD
	fprintf(log, "skipd: old_hdr.req=SKIPREQ_GET(en/de)SKIPKEY\n");
#endif
	skipd_updatekijn(&oldhdr, l, keyttl);
	break;
      }
    default:
      {
	fprintf(log, "skipd: cannot answer request %d.\n", oldhdr.req);
      }
  }

  skipcache_release(l1);
  skipcache_release(l2);
}

int main(int argc, char **argv)
{
  u_char buf[256];
  int len;
  struct skipcache *c;
  int ch;
  extern char *optarg;
  extern int optind;
  int errflg = 0, flush = 0, verbose = 0, daemon = 0;
  char certdir[2048];
  char *cnffile = CONFIGFILE, *cert_path = CERTDIR;
  time_t t;
  char ct[60];

  log = stderr;

  while ((ch = getopt(argc, argv, "c:fh?vk:d")) != EOF) {
    switch (ch) {
      case 'c':
	cnffile = optarg;
	break;

      case 'h':
      case '?':
      default:
	errflg++;
	break;

      case 'f':
	flush = 1;
	break;

      case 'v':
	verbose++;
	break;

      case 'k':
	cert_path = optarg;
	break;

      case 'd':
	daemon = 1;
	break;
    }
  }

  if (errflg || (optind < argc))
    usage(argv[0]);

  /* fork daemon process */
  if (daemon) {
    if (fork() != 0)		/* not child, error */
      exit(0);

    /* become a daemon */
    setsid();			/* disconnect from the terminal */
    chdir("/");			/* just in case someone wants to unmount the current dir */

    if ((log = fopen(LOGFILE, "a")) == NULL) {
      fprintf(stderr, "skipd: failed to append to log file %s\n", LOGFILE);
      exit(1);
    }
    setvbuf(log, NULL, _IOLBF, 0);
  }

  signal(SIGURG,  signal_handler);
  signal(SIGHUP,  signal_handler);
  signal(SIGUSR1, signal_handler);
  signal(SIGUSR2, signal_handler);

  crypt_init();
  random_init();
  skipcache_init(0, 0);		/* No maxttl, no maxentries */

  if (com_init() < 0) {
    fprintf(stderr, "%s: SKIP module not found\n", argv[0]);
    return 1;
  }

  /* Only flush primary (kernel) cache and exit */
  if (flush) {
    if (skipd_flush() < 0) {
      fprintf(stderr, "skipd: failed to flush cache\n");
      return 1;
    }
    else
      return 0;
  }

  strcpy(certdir, cert_path);
  if (certdir[strlen(certdir) - 1] != '/')
    strcat(certdir, "/");

  if (cert_init(certificate_recv, certdir) < 0) {
    fprintf(log, "skipd: failed to initialize certificate unit.\n");
    return 1;
  }
  certinitflag = 0;

  if (skipd_register() == -1) {
    cert_exit();
    fprintf(log, "skipd: failed to register daemon\n");
    return 1;
  }

  if (verbose) {
    t = time(NULL);
    sprintf(ct, ctime(&t));
    ct[strlen(ct)-6] = 0;
    fprintf(log, "%s skipd: initializing.\n", ct);
  }

  for (reinitflag = 1; reinitflag;) {
    reinitflag = 0;

    /* Flush both primary (kernel) and secondary (user) cache */

    skipcache_flush();

    if (certinitflag && (cert_init(certificate_recv, certdir) < 0)) {
      fprintf(log, "skipd: failed to initialize certificate unit.\n");
      break;
    }
    certinitflag = 0;

    parse_init();

    if ((c = parse_parse(cnffile)) == NULL) {
      fprintf(log, "skipd: no entry in config file, exiting...\n");
      exit(1);
    }

    skipd_flush();

    while (c->next) {		/* Fill skipd cache, except last which is default entry */
      struct skipcache *next = c->next;

      c->flags |= SKIP_DONTAGE;
      skipcache_update(c);

      c = next;
    }

    /* set gobal default entry pointer */
    c->flags |= SKIP_DONTAGE;
    def = c;


    /* Main loop, handling requests from kernel */
    while (!reinitflag) {
      u_char *walk;
      struct skipreq_hdr *hdr = (struct skipreq_hdr *) buf;
      int reqlen;

      while (!(getreqflag || saveconfigflag || reinitflag || showcacheflag)) {
	cert_iowait();		/* Wait for signals and cert traffic */
      }

      if (showcacheflag) {
        showcacheflag = 0;
      }

      getreqflag = 0;
      hdr->req = SKIPREQ_GETREQUESTS;
      len = com_request(buf, sizeof(buf));

      for (walk = buf; len > 0; walk += reqlen, len -= reqlen) {
	struct skipcache *l;

	hdr = (struct skipreq_hdr *) walk;

	if (verbose) {
	  int walk;

	  t = time(NULL);
          sprintf(ct, ctime(&t));
          ct[strlen(ct)-6] = 0;
	  fprintf(log, "%s request: primitive %d\n", ct, hdr->req);
	  fprintf(log, "                    src id %u-", hdr->id.type[0]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", hdr->id.id[0][walk]);
	  fprintf(log, "\n                    dst id %u-", hdr->id.type[0]);
	  for (walk = 0; walk < SKIP_ID_MAXLEN; walk++)
	    fprintf(log, "%02x", hdr->id.id[1][walk]);
	  fprintf(log, "\n                    src ip %u.%u.%u.%u, dst ip %u.%u.%u.%u\n",
	      hdr->srcip[0], hdr->srcip[1], hdr->srcip[2], hdr->srcip[3],
	     hdr->dstip[0], hdr->dstip[1], hdr->dstip[2], hdr->dstip[3]);
	}

	l = skipcache_lookup(&hdr->id);

	/* if no entry was found, we use the default entry */
	if (l == NULL) {
	  if (verbose)
	    fprintf(log, "                    using default entry\n");
	  l = def;
	}

	switch (hdr->req) {
	  case SKIPREQ_GETENTRY:
	    {
#ifdef DEBUG_SKIPD
	      fprintf(log, "skipd: SKIPREQ_GETENTRY\n");
#endif
	      reqlen = sizeof(struct skipreq_getentry);

	      /* shared secret available (manual or dh)? */
	      if (l->flags & SKIP_VALIDKEY) {
		if (verbose)
		  fprintf(log, "                    shared secret available\n");
		skipd_updateentry(hdr, l);
	      }
	      /* or is no key resolvable */
	      else if (l->flags & SKIP_NOKEY) {
		if (verbose)
		  fprintf(log, "                    no key resolvable\n");
		skipd_updateentry(hdr, l);
	      }
	      /* or is it a chip card and we can calc at least enskip_Kp* */
	      else if (l->flags & SKIP_CHIPCARD) {
		if (verbose)
		  fprintf(log, "                    chip card\n");
		/* XXX: calc enskip_A_Kp, enskip_E_Kp, enskip_Kp_Kijn */
		skipd_updateentry(hdr, l);
	      }
	      /* we cannot... start discovery */
	      else {
		if (verbose)
		  fprintf(log, "                    starting discovery\n");
		skipd_discover(hdr, l);
	      }
	      break;
	    }

	  case SKIPREQ_GETENSKIPKEY:
	    {
#ifdef DEBUG_SKIPD
	      fprintf(log, "skipd: SKIPREQ_GETENSKIPKEY\n");
#endif
	      reqlen = sizeof(struct skipreq_getenskipkey);

	      /* shared secret available (manual or dh)? */
	      if (l->flags & SKIP_VALIDKEY) {
		if (verbose)
		  fprintf(log, "                    shared secret available\n");
		skipd_updatekijn(hdr, l, l->keyttl);
	      }
	      /* or is it a chip card and we can calc at least enskip_Kp* */
	      else if (l->flags & SKIP_CHIPCARD) {
		/*
		 * XXX: calc enskip_A_Kp, enskip_E_Kp, enskip_Kp_Kijn
		 * with a kind of discovery
		 */
		/* skipd_updateenskipkp(hdr, l); */
	      }
	      /* we cannot... start discovery */
	      else {
		if (verbose)
		  fprintf(log, "                    starting discovery\n");
		skipd_discover(hdr, l);
	      }
	      break;
	    }

	  case SKIPREQ_GETDESKIPKEY:
	    {
#ifdef DEBUG_SKIPD
	      fprintf(log, "skipd: SKIPREQ_GETDESKIPKEY\n");
#endif
	      /* XXX: this part is not complete... check later */
	      reqlen = sizeof(struct skipreq_getdeskipkey);

	      /* shared secret available (manual or dh)? */
	      if (l->flags & SKIP_VALIDKEY) {
		if (verbose)
		  fprintf(log, "                    shared secret available\n");
		skipd_updatekijn(hdr, l, l->keyttl);
	      }
	      /* or is it a chip card and we can calc at least deskip_Kp */
	      else if (l->flags & SKIP_CHIPCARD) {
		/* XXX: calc deskip_A_Kp, deskip_E_Kp with a kind of discovery */
		/* skipd_updatedeskipkp(hdr, l); */
	      }
	      /* we cannot... start discovery */
	      else {
		if (verbose)
		  fprintf(log, "                    starting discovery\n");
		skipd_discover(hdr, l);
	      }
	      break;
	    }

	  default:
	    {
	      fprintf(log, "skipd: unknown request %d.\n", hdr->req);
	      reqlen = len;
	    }
	}

	if (l != def)
	  skipcache_release(l);
      }
    }

    /* Free default entry */
    parse_free(def);
    def = NULL;
    cert_exit();
    certinitflag = 1;

    if (verbose) {
      t = time(NULL);
      sprintf(ct, ctime(&t));
      ct[strlen(ct)-6] = 0;
      fprintf(log, "%s skipd: reinitializing.\n", ct);
    }
  }

  return 0;
}
