#include "config.h"

#include <sys/stream.h>
#include <inet/common.h>
#include <inet/ip.h>
#include <inet/ip_if.h>
#include <inet/ip_ire.h>
#include <sys/dlpi.h>

#include "skip_defs.h"
#include "memblk.h"
#include "dynamic.h"
#include "skipcache.h"
#include "skip.h"
#include "ipsp.h"
#include "interface.h"

#ifdef __GNUC__
#ident "$Id: interface.c,v 1.9 1996/04/25 14:56:44 cschneid Exp $"
#else
static char rcsid[] = "$Id: interface.c,v 1.9 1996/04/25 14:56:44 cschneid Exp $";
#endif

struct interface
{
  ill_t *ill;			/* interface identifier */
  ipif_t *ipif;
  queue_t *q;
};

static ill_t *interface_find(queue_t *q)
{
  ill_t *ill = NULL;

  if (q)
  {
#if 0
    for (ill = ill_g_head; ill; ill = ill->ill_next)
    {
      printf(">> ill_rq=%x rd=%x wr=%x rd->next=%x wr->next=%x, hdrlen = %d\n", ill->ill_rq, RD(q), WR(q), RD(q)->q_next, WR(q)->q_next, ill->ill_hdr_length);
    }
#endif

    for (ill = ill_g_head; ill; ill = ill->ill_next)
    {
#if 0
      printf(">> ill_rq=%x rd=%x wr=%x rd->next=%x wr->next=%x\n", ill->ill_rq, RD(q), WR(q), RD(q)->q_next, WR(q)->q_next);
#endif
      if (ill->ill_rq == RD(q)->q_next)
	break;  /* Found interface with matching q */
    }
  }

  return ill;
}

/*
 * Prepends <offset> bytes and ensures <len> bytes are contiguous afterwards
 * A trailer of <trllen> bytes is appended
 * Type of given mblk chain is ignored, newly allocated head is M_DATA
 */
static mblk_t *interface_pullup(mblk_t *m, int len)
{
  mblk_t *new = NULL, *t;

  /* Enough data contiguous? */
  if (m && ((m->b_wptr - m->b_rptr) < len))
  {
    /* No, prepare new mblk */
    if ((new = allocb(len, BPRI_LO)) != NULL)
    {
      while (m && (len > 0))
      {
        /* Calculate amount of data to copy from next mblk */
        int l = (len <= m->b_wptr - m->b_rptr) ? len : (m->b_wptr - m->b_rptr);

        MEMCPY(new->b_wptr, m->b_rptr, l);
        new->b_wptr += l;
        m->b_rptr += l;

        /* No data left in mblk -> free it and skip to next */
        if (m->b_rptr == m->b_wptr)
        {
          t = m->b_cont;
          freeb(m);
          m = t;
        }
      
        len -= l;
      }

      new->b_cont = m;
    }
  }
  else
    new = m;

  return new;
}

static int mblk2memblk(mblk_t *m, struct memblk *mb, struct memseg *ms)
{
  int count = 1;

  /* Construct memblk description */
  mb->dynamic = NULL;
  mb->dynamiclen = 0;
  mb->ms = ms;
  mb->offset = m->b_rptr - m->b_datap->db_base;

  /* First mblk is special case (offset & header pulled up) */
  ms->ptr = m->b_datap->db_base;
  ms->len = m->b_wptr - ms->ptr;
  mb->len = ms->len - mb->offset;
  ms++;

  if ((m = m->b_cont))  /* More than one mblk? */
  {
    while (m)  /* Process complex message */
    {
      int ms_len;

      if ((ms_len = m->b_wptr - m->b_rptr))  /* Ignore empty mblks */
      {
	if (count == (STATIC_MEMSEGS - 2))	/* Need dynamic :-\ */
	{
	  mblk_t *t;
	  int len;

	  for (len = STATIC_MEMSEGS * sizeof(*ms), t = m; t; t = t->b_cont)
	    len += sizeof(struct memseg);

	  mb->dynamiclen = len;
	  if ((mb->dynamic = KALLOC(len)) == NULL)
	    return -1;

	  MEMCPY(mb->dynamic, mb->ms, count * sizeof(struct memseg));
	  mb->ms = mb->dynamic;
	  ms = mb->dynamic + count;
	}

        ms->ptr = m->b_rptr;
	ms->len = ms_len;
        mb->len += ms_len;  /* Length of whole message */
        ms++;
	count++;
      }

      m = m->b_cont;
    }
  }

  mb->freems = ms;	/* First free memseg, ESP uses this internally */

#if 0
{
int hdrlen;
printf(">>skip: mblk2memblk: offset %d, len %d:", mb->offset, mb->len);
for (ms = mb->ms, hdrlen = -mb->offset; hdrlen < mb->len; hdrlen += ms->len, ms++)
  printf(" %d(%x)", ms->len, ms->ptr);
printf("\n");
}
#endif

  return 0;
}

#if 1
static void dump(u_char *data, int len)
{
  while (len--)
    printf(" %02x", *data++);
}
#endif

int interface_input(queue_t *q, mblk_t *mp)
{
  struct interface *i = (struct interface *)q->q_ptr;
  mblk_t *nodata = NULL;

#if 0
printf(">>interface_input() q->q_ptr=%x\n", q->q_ptr);
#endif

  if (i == NULL)  /* Attached? */
  {
#if 0
    printf(">>interface_input(): q->q_ptr is NULL (q=%x)!\n", q);
#endif
    putnext(q, mp);
    return 0;
  }

#if 0
    if (!mp) {
      printf(">>interface_input: got packet (no mp?!)\n");
    } else if (!mp->b_datap) {
      printf(">>interface_input: got packet (no b_datap, len %d)\n",
								msgdsize(mp));
    } else {
      printf(">>interface_input: got packet (type %d, len %d)\n", 
					mp->b_datap->db_type, msgdsize(mp));
    }
#endif

#if 0
  /* XXX: quick hack */
  if (mp && mp->b_datap && ((mp->b_datap->db_type == M_PROTO) || 
                               (mp->b_datap->db_type == M_PCPROTO)))
  {
    dl_unitdata_ind_t *x = (dl_unitdata_ind_t *)(mp->b_rptr);

    if (x->dl_primitive == DL_UNITDATA_IND)
    {
      mblk_t *tmp = mp;

      mp = mp->b_cont;
      freeb(tmp);
      mp->b_datap->db_type = M_DATA;
    }
  }
#else
  if ((mp->b_datap->db_type == M_PROTO) || (mp->b_datap->db_type == M_PCPROTO))
  {
    dl_unitdata_ind_t *dl = (dl_unitdata_ind_t *)(mp->b_rptr);

    if (dl->dl_primitive == DL_UNITDATA_IND)
    {
      nodata = mp;
      mp = mp->b_cont;
      nodata->b_cont = NULL;
    }
    else
    {
      nodata = mp;
      mp = NULL;
    }
  }
  else if (mp->b_datap->db_type != M_DATA)
  {
    nodata = mp; 
    mp = NULL;
  }
#endif

#if 0
  printf(">>interface_input: (post fake) got packet (type %d, len %d)\n", 
				    mp->b_datap->db_type, msgdsize(mp));
#endif

  if (mp)	/* Any data */
  {
    int result = SKIP_DISCARD;
    mblk_t *m, *newm;
    struct memblk oldmb, newmb;
    struct memseg oldms[STATIC_MEMSEGS], newms[2];

    /* XXX: IP header is assumed to be contiguous up to ip_p already */
    if ((newm = m = interface_pullup(mp, ipsp_getheadersize(mp->b_rptr))))
    {
      MEMZERO(&newmb, sizeof(newmb));
      newmb.ms = newms;
      if (mblk2memblk(m, &oldmb, oldms) == 0)
      {
	result = skip_process(SKIP_INPUT, i, nodata, (void **)&newm, &oldmb,
			      &newmb);
      }

      /* Note: new will never have dynamic memsegs on Solaris */
      if (oldmb.dynamic)
	KFREE(oldmb.dynamic, oldmb.dynamiclen);
    }
    else
      m = mp;  /* Discard original mblk */

#if 0
{
    mblk_t *t;
    printf(">>interface_input: start:");
    for (t = m; t; t = t->b_cont)
      dump(t->b_rptr, t->b_wptr - t->b_rptr);
    printf("\n");
}
#endif
    switch(result)
    {
      default:
      case SKIP_DISCARD:
      {
        if (nodata)
          freemsg(nodata);
        freemsg(m);
	if (newm != m)  /* Buffer already allocated before failure? */
	  freemsg(newm);
	return 0;
      }

      case SKIP_PROCESSED:
      {
#if 0
{
    mblk_t *t;
    printf(">>interface_input: end  :");
    for (t = m; t; t = t->b_cont)
      dump(t->b_rptr, t->b_wptr - t->b_rptr);
    printf("\n");
}
#endif
        if (newm != m)  /* New buffer allocated? */
        {
          freemsg(m);
	  /* Adjust length of newly created packet */
	  newm->b_rptr = BLKSTART(&newmb);
	  newm->b_wptr = newm->b_rptr + newmb.len;
#if 0
	  printf(">>interface_input: new pkt(%d):", 
		 newm->b_wptr - newm->b_rptr);
#if 0
	  dump(newm->b_rptr, newm->b_wptr - newm->b_rptr);
#endif
	  printf("\n");
#endif
#if 0
	  freemsg(newm);
	  return 0;
#endif
	}

        if (nodata)
        {
          nodata->b_cont = newm;
          newm = nodata;
        }

        return putnext(q, newm);
      }

      case SKIP_QUEUED:
      {
	if (newm != m)  /* Buffer already allocated before failure? */
	  freemsg(newm);
	return 0;
      }
    }
  }
  else
    mp = nodata;

  return putnext(q, mp);
}

int interface_output(queue_t *q, mblk_t *mp)
{
  struct interface *i = (struct interface *)q->q_ptr;
  mblk_t *nodata, *nodata_tail;
  int hdrlen;

#if 0
printf(">>interface_output() q->q_ptr=%x\n", q->q_ptr);
#endif

  if (i == NULL)  /* Attached? */
  {
#if 0
    printf(">>interface_output(): q->q_ptr is NULL! (q=%x)\n", q);
#endif
    putnext(q, mp);
    return 0;
  }

  /*
   * Separate M_* from M_DATA mblks
   */
  for (nodata = mp, nodata_tail = NULL; mp; nodata_tail = mp, mp = mp->b_cont)
  {
    if (mp->b_datap->db_type == M_DATA)
      break;  /* NOTE: nodata_tail is NULL if only M_DATA mblks there */
  }

  /*
   * Have to remove lower level header if
   * a) data left  b) hdrlen > 0  c) ll header not included in nodata
   */
  hdrlen = i->ill->ill_hdr_length;

  if (mp && hdrlen && !nodata_tail)
  {
    if ((mp->b_wptr - mp->b_rptr) == hdrlen) /* Separate lower level hdr */
    {
      nodata = nodata_tail = mp;
      mp = mp->b_cont;
      nodata_tail->b_cont = NULL;
    }
    else
    {
      if ((nodata = nodata_tail = allocb(hdrlen, BPRI_LO)))
      {
	MEMCPY(nodata->b_rptr, mp->b_rptr, hdrlen);
	nodata->b_wptr += hdrlen;
	mp->b_rptr += hdrlen;
      }
      else  /* Could not get mblk for lower level header */
      {
	freemsg(mp);
	return 0;
      }
    }
  }
  else if (nodata_tail)
    nodata_tail->b_cont = NULL;

  /*
   * Finally mp should point to an IP packet :-}
   */
  if (mp)  /* Any M_DATA? */
  {
    int result = SKIP_DISCARD;
    mblk_t *m, *newm;
    struct memblk oldmb, newmb;
    struct memseg oldms[STATIC_MEMSEGS], newms[2];

#if 1
    if ((*mp->b_rptr & 0xf0) != 0x40)
    {
      printf(">>interface_output: GOT BOGUS IP PACKET:");
      dump(mp->b_rptr, 32);
      return 0;
    }
#endif
#if 0
{
    mblk_t *t;
    printf(">>interface_output: start:");
    for (t = mp; t; t = t->b_cont)
      dump(t->b_rptr, t->b_wptr - t->b_rptr);
    printf("\n");
}
#endif
    /* XXX: IP header is assumed to be contiguous up to ip_p already */
    if ((newm = m = interface_pullup(mp, ipsp_getheadersize(mp->b_rptr))))
    {
      MEMZERO(&newmb, sizeof(newmb));
      newmb.ms = newms;

      if (mblk2memblk(m, &oldmb, oldms) == 0)
      {
	result = skip_process(SKIP_OUTPUT, i, nodata, (void **)&newm, &oldmb,
			      &newmb);
      }

      /* Note: new will never have dynamic memsegs on Solaris */
      if (oldmb.dynamic)
	KFREE(oldmb.dynamic, oldmb.dynamiclen);
    }
    else
      m = mp;  /* Discard original mblk */
  
    switch (result)
    {
      default:
      case SKIP_DISCARD:
      {
        freemsg(nodata);
        freemsg(m);
	if (newm != m)  /* Buffer already allocated before failure? */
	  freemsg(newm);
	return 0;
      }

      case SKIP_PROCESSED:
      {
	/* Prepend nodata header again */
#if 0
{
    mblk_t *t;
    printf(">>interface_output: end  :");
    for (t = mp; t; t = t->b_cont)
      dump(t->b_rptr, t->b_wptr - t->b_rptr);
    printf("\n");
}
#endif
        if (newm != m)  /* New buffer allocated? */
        {
          freemsg(m);
	  /* Adjust length of newly created packet */
	  newm->b_rptr = newm->b_datap->db_base + newmb.offset;
	  newm->b_wptr = newm->b_rptr + newmb.len;
	}

        if (nodata_tail)
        {
	  nodata_tail->b_cont = newm;
          newm = nodata;
        }
	return putnext(q, newm);
      }

      case SKIP_QUEUED:
      {
	if (newm != m)  /* Buffer already allocated before failure? */
	  freemsg(newm);
	return 0;
      }
    }
  }

  return putnext(q, nodata);
}

int interface_infeedback(void *feedbackif, void *m)
{
  struct interface *i = feedbackif;

  return interface_input(RD(i->q), (mblk_t *)m);
}

int interface_outfeedback(void *feedbackif, void *m)
{
  struct interface *i = feedbackif;

  return interface_output(WR(i->q), (mblk_t *)m);
}

/* Returns failure if called without queue, IPaddr is ignored */
int interface_attach(void *qptr, u_char *IPaddr)
{
  queue_t *q = (queue_t *)qptr;
  ill_t *ill;
  struct interface *new = NULL;
  int mtu_dec = ipsp_maxheadergrowth();
  ire_t *ire;

  if ((ill = interface_find(q)) && (new = KALLOC(sizeof(*new))))
  {
    MEMZERO(new, sizeof(*new));
    new->ill = ill;
    new->q = q;
    new->ipif = ill->ill_ipif_pending;
    ill->ill_max_frag -= mtu_dec;

    new->ipif->ipif_mtu -= mtu_dec;
    if ((ire = ipif_to_ire(new->ipif)))
      ire->ire_max_frag -= mtu_dec;
    q->q_ptr = new;
    OTHERQ(q)->q_ptr = new;
#if 0
    printf(">>interface_attach: q = %x, oq = %x\n", (int)q, (int)OTHERQ(q));
#endif
  }

  return new ? 0 : -ENOMEM;
}

static int interface_remove(struct interface *i)
{
  int mtu_inc = ipsp_maxheadergrowth();
  ire_t *ire;

  i->ill->ill_max_frag += mtu_inc;
  i->ipif->ipif_mtu += mtu_inc;

  if ((ire = ipif_to_ire(i->ipif)))
    ire->ire_max_frag += mtu_inc;

  KFREE(i, sizeof(*i));
  return 0;
}

/* Returns failure if called without queue, IPaddr is ignored */
int interface_detach(void *qptr, u_char *IPaddr)
{
  queue_t *q = (queue_t *)qptr;
  struct interface *i = q ? (struct interface *)q->q_ptr : NULL;

  return i ? interface_remove(i) : -ENOENT;
}

/*
 * NOTE: The following rules have to be followed:
 * sizeof(struct ip) + ipsp_maxheadergrowth() bytes contiguous at beginning
 * MAXBLOCKLEN + 1 contiguous bytes at end
 */
int interface_getbuf(struct memblk *m, void **mp, int size)
{
  int result = -ENOMEM;
  mblk_t *new;

  if ((new = allocb(size, BPRI_LO)))  /* Get one large buffer */
  {
    m->len = m->ms->len = size;
    m->offset = 0;
    m->ms->ptr = new->b_rptr;

    *mp = new;  /* Store newly allocated mblk */

    result = 0;
  }

  return result;
}

int interface_init()
{
  return 0;
}

int interface_exit()
{
  return 0;
}
