#include "config.h"

#include <sys/conf.h>
#include <sys/stream.h>
#include <sys/errno.h>
#include <sys/cred.h>
#include <sys/modctl.h>
#include <sys/devops.h>
#include <sys/ddi.h>
#include <sys/sunddi.h>
#include <sys/stat.h>
#include <sys/cmn_err.h>
#include <sys/dlpi.h>
#include <inet/common.h>
#include <inet/ip.h>

#include "skip_defs.h"
#include "memblk.h"
#include "dynamic.h"
#include "skipcache.h"
#include "random.h"
#include "queue.h"
#include "ipsp.h"
#include "crypt.h"
#include "sign.h"
#include "com.h"
#include "interface.h"
#include "req.h"

#ifdef __GNUC__
#ident "$Id: skipmod.c,v 1.9 1996/04/25 14:56:44 cschneid Exp $"
#else
static char rcsid[] = "$Id: skipmod.c,v 1.9 1996/04/25 14:56:44 cschneid Exp $";
#endif

#define MINSDUSZ 1
#define MAXSDUSZ INFPSZ

#undef USE_SERVICE_ROUTINE

char _depends_on[] = "drv/ip";

static struct module_info skip_minfo = 
{
  0x534b, "skip", MINSDUSZ, MAXSDUSZ, 0, 0
};

/************************************************************************
 * SKIP STREAMS device information (/dev/skip)
 */
static int skipopen(queue_t *q, dev_t *devp, int flag, int sflag, cred_t *crp);
static int skipclose(queue_t *q, int flag, cred_t *crp);
static int skipwput(queue_t *q, mblk_t *mp);

static struct qinit skip_rinit =
{
  NULL, NULL, skipopen, skipclose, NULL, &skip_minfo, NULL
};

static struct qinit skip_winit =
{
  skipwput, NULL, NULL, NULL, NULL, &skip_minfo, NULL
};

struct streamtab skipinfo =
{
  &skip_rinit, &skip_winit, NULL, NULL
};

extern int nulldev();
extern int nodev();

static int skip_info(dev_info_t *dip, ddi_info_cmd_t infocmd, void *arg,
			void **result);
static int skip_attach(dev_info_t * devi,  ddi_attach_cmd_t cmd);
static int skip_identify(dev_info_t * devi);
static int skip_detach(dev_info_t * devi,  ddi_detach_cmd_t cmd);

static struct cb_ops skip_ops = 
{
  nulldev,		/* cb_open */
  nulldev,		/* cb_close */
  nodev,		/* cb_strategy */
  nodev,		/* cb_print */
  nodev,		/* cb_dump */
  nodev,		/* cb_read */
  nodev,		/* cb_write */
  nodev,		/* cb_ioctl */
  nodev,		/* cb_devmap */
  nodev,		/* cb_mmap */
  nodev,		/* cb_segmap */
  nochpoll,		/* cb_chpoll */
  ddi_prop_op,		/* cb_prop_op */
  &skipinfo,		/* cb_stream */
  0			/* cb_flag */
};

static struct dev_ops dev_ops = 
{
  DEVO_REV,		/* devo_rev */
  0,			/* devo_refcnt */
  skip_info,		/* devo_getinfo */
  skip_identify,	/* devo_identify */
  nulldev,		/* devo_probe */
  skip_attach,		/* devo_attach */
  skip_detach,		/* devo_detach */
  nodev,		/* devo_reset */
  &skip_ops,		/* devo_cb_ops */
  NULL			/* devo_bus_ops */
};

static struct modldrv modldrv =
{
  &mod_driverops, "SKIP Streams device", &dev_ops
};

/************************************************************************
 * SKIP STREAMS module information
 */
static int skipstrmodopen(queue_t *q, dev_t *devp, int flag, int sflag,
			  cred_t *crp);
static int skipstrmodclose(queue_t *q, int flag, cred_t *crp);
static int skipstrmodrput(queue_t *q, mblk_t *mp);
static int skipstrmodwput(queue_t *q, mblk_t *mp);
#ifdef USE_SERVICE_ROUTINE
static int skipstrmodrsrv(queue_t *q);
#endif

static struct qinit skipstrmod_rinit =
{
#ifdef USE_SERVICE_ROUTINE
  skipstrmodrput, skipstrmodrsrv, skipstrmodopen, skipstrmodclose, NULL, &skip_minfo, NULL
#else
  skipstrmodrput, NULL, skipstrmodopen, skipstrmodclose, NULL, &skip_minfo, NULL
#endif
};

static struct qinit skipstrmod_winit =
{
  skipstrmodwput, NULL, NULL, NULL, NULL, &skip_minfo, NULL
};

struct streamtab skipstrmodinfo =
{
  &skipstrmod_rinit, &skipstrmod_winit, NULL, NULL
};

static struct fmodsw skip_fmodsw =
{
  "skip", &skipstrmodinfo, D_MP, NULL
};

static struct modlstrmod modlstrmod =
{
  &mod_strmodops, "SKIP Streams module", &skip_fmodsw
};

/************************************************************************
 * SKIP STREAMS externally visible information for _init() and _info ()
 */
static struct modlinkage modlinkage =
{
  MODREV_1,
  { (void *)&modlstrmod, (void *)&modldrv, NULL }
};

/************************************************************************
 * SKIP STREAMS device functions
 */
static dev_info_t *skip_dev_info;

static int skip_attach(dev_info_t *devi, ddi_attach_cmd_t cmd)
{
  int result = DDI_SUCCESS;

  skip_dev_info = devi;

  if (ddi_create_minor_node(devi, "skip", S_IFCHR, 0, NULL, CLONE_DEV) ==
      DDI_FAILURE) 
  {
    ddi_remove_minor_node(devi, NULL);
    result =  -1;
  }

  return result;
}

static int skip_detach(dev_info_t *devi, ddi_detach_cmd_t cmd)
{
  return DDI_SUCCESS;
}


static int skip_info(dev_info_t *dip, ddi_info_cmd_t infocmd, void *arg,
		     void **res)
{
  int result = DDI_FAILURE;

  switch (infocmd) 
  {
    case DDI_INFO_DEVT2DEVINFO:
      if (skip_dev_info != NULL) 
      {
	*res = (void *)skip_dev_info;
	result = DDI_SUCCESS;
      }
      break;

    case DDI_INFO_DEVT2INSTANCE:
      *res = (void *)0;
      result = DDI_SUCCESS;
      break;
  }

  return result;
}

static int skip_identify(dev_info_t *devi)
{
  int result = DDI_NOT_IDENTIFIED;

  if (strcmp((char *)ddi_get_name(devi), "skip") == 0)
    result = DDI_IDENTIFIED;

  return result;
}

static int skipopen(queue_t *q, dev_t *devp, int flag, int sflag, cred_t *crp)
{
  return 0;
}

static int skipclose(queue_t *q, int flag, cred_t *crp)
{
  return 0;
}

static void copymblk2data(mblk_t *m, u_char *data, int len)
{
  while (m)
  {
    int l = m->b_wptr - m->b_rptr;

    if (l > len)
      l = len;

    if (l > 0)
    {
      MEMCPY(data, m->b_rptr, l);
      data += l;
      if ((len -= l) <= 0)
	break;
    }
    m = m->b_cont;
  }
}

static void copydata2mblk(mblk_t *m, u_char *data, int len)
{
  while (m)
  {
    int l = m->b_wptr - m->b_rptr;

    if (l > len)
      l = len;

    if (l > 0)
    {
      MEMCPY(m->b_rptr, data, l);
      data += l;
      if ((len -= l) <= 0)
	break;
    }
    m = m->b_cont;
  }
}

static void confirmioctl(mblk_t *mp)
{
  struct iocblk *iocp = (struct iocblk *)mp->b_rptr;

  iocp->ioc_error = 0;
  iocp->ioc_rval = 0;
  mp->b_datap->db_type = M_IOCACK;
}

/* XXX: check ioctl data sizes, e.g. REMOVE */
static int skipwput_ioctl(queue_t *q, mblk_t *mp)
{
  struct iocblk *iocp = (struct iocblk *)mp->b_rptr;
  mblk_t *m = mp->b_cont;

  mp->b_datap->db_type = M_IOCNAK;

  if (m && (iocp->ioc_count != TRANSPARENT))
  {
    u_char *data;
    int len = iocp->ioc_count, newlen;

    if ((data = KALLOC(len)))
    {
      switch (iocp->ioc_cmd)
      {
	case SKIPIOCREQUEST:
	{
	  copymblk2data(m, data, len);
	  if ((newlen = req_handle(data, len)) >= 0)
	  {
	    copydata2mblk(m, data, newlen);
	    iocp->ioc_count = newlen;
	    confirmioctl(mp);
	  }
	}
      }

      KFREE(data, len);
    }
  }

  qreply(q, mp);

  return 0;
}

static int skipwput(queue_t *q, mblk_t *mp)
{
  int result = 0;

  switch (mp->b_datap->db_type)
  {
    case M_IOCTL:
      result = skipwput_ioctl(q, mp);
      break;

    default:
      freemsg(mp);
      break;
  }

  return result;
}

/************************************************************************
 * SKIP STREAMS module functions
 */
static int skipstrmodopen(queue_t *q, dev_t *devp, int flag, int sflag,
			  cred_t *crp)
{
  qprocson(q);
#if 0
  printf(">>skipstrmodopen of q %d: dev %d, flag %d, sflag %d, nq %d, onq %d, lq %d, olq %d\n", 
	 (int)q, *devp, flag, sflag,
	 (int)(q->q_next), (int)(WR(q)->q_next),
	 (int)(q->q_link), (int)(WR(q)->q_link));
#endif
  return (sflag == MODOPEN) ? 0 : EINVAL;
}

static int skipstrmodclose(queue_t *q, int flag, cred_t *crp)
{
  qprocsoff(q);

  /* lazy... :-/ */
  interface_detach(q, NULL);
  interface_detach(OTHERQ(q), NULL);

#if 0
  printf(">>skipstrmodclose\n");
#endif
  return 0;
}

#ifdef USE_SERVICE_ROUTINE
static int skipstrmodrsrvput(queue_t *q, mblk_t *mp)
#else
static int skipstrmodrput(queue_t *q, mblk_t *mp)
#endif
{
  if ((mp->b_datap->db_type == M_PROTO) || (mp->b_datap->db_type == M_PCPROTO))
  {
    dl_bind_ack_t *b = (dl_bind_ack_t *)mp->b_rptr;

    switch (b->dl_primitive)
    {
      case DL_BIND_ACK:
      {
        if ((q->q_ptr == NULL) && (b->dl_sap == IP_ARP_PROTO_TYPE) &&
            (interface_attach(q, NULL) < 0))
        {
          dl_error_ack_t *e = (dl_error_ack_t *)mp->b_rptr;
#if 1
          printf(">> could not attach -> dl_error_ack\n");
#endif
          mp->b_datap->db_type = M_PCPROTO;
	  mp->b_wptr = (mp->b_rptr + sizeof(dl_error_ack_t));
	  e->dl_primitive = DL_ERROR_ACK;
	  e->dl_error_primitive = DL_BIND_REQ;
	  e->dl_unix_errno = ENOMEM;
	  e->dl_errno = DL_SYSERR;
	  putnext(q, mp);
	  return -1;
	}
#if 0
        else
          printf(">>skipstrmodrput: ATTACHED\n");
#endif

#if 0
        printf(">> ack received q=%x sap=%x\n", q, b->dl_sap);
#endif
        break;
      }
    }
  }

  return interface_input(q, mp);
}

#ifdef USE_SERVICE_ROUTINE
static int skipstrmodrsrv(queue_t *q)
{
  int result = 0;
  mblk_t *mp;

#if 1
  printf(">>skipstrmodrsrv: start\n");
#endif
  while ((mp = getq(q)))
  {
#if 1
    printf(">>skipstrmodrsrv: got msg (q %x, mp %x)\n", (int)q, (int)mp);
#endif
    if ((result = skipstrmodrsrvput(q, mp)))
      break;
  }

  return result;
}

static int skipstrmodrput(queue_t *q, mblk_t *mp)
{
  if ((mp->b_datap->db_type == M_DATA) ||
      (mp->b_datap->db_type == M_PROTO) || (mp->b_datap->db_type == M_PCPROTO))
  {
#if 0
    printf(">>skipstrmodrput: putting msg to queue (q %x, mp %x)\n", (int)q, (int)mp);
#endif
    return putq(q, mp);
  }
  else
    return putnext(q, mp);
}
#endif

static int skipstrmodwput(queue_t *q, mblk_t *mp)
{
#if 0
  printf(">>skipstrmodwput: out pkt\n");
#endif
  return interface_output(q, mp);
}

/************************************************************************
 * SKIP kernel module initialization
 */
int _init(void)
{
  int result;

  if ((result = mod_install(&modlinkage)) == 0)
  {
    crypt_init();
    sign_init();
    random_init();
    queue_init();
    skipcache_init(SKIPCACHE_MAXTTL, SKIPCACHE_MAXENTRIES);
    ipsp_init();
    interface_init();
  }

  return result;
}

int _fini(void)
{
  int result;

  if ((result = mod_remove(&modlinkage)) == 0)
  {
    interface_exit();
    ipsp_exit();
    skipcache_exit();
    queue_exit();
    random_exit();
    sign_exit();
    crypt_exit();
  }

  return result;
}

int _info(struct modinfo *modinfop)
{
  return mod_info(&modlinkage, modinfop);
}

