/************************************************************************/
/*									*/
/* Sentry - Port Scan Detector						*/
/*									*/
/* Author: Craig H. Rowland <crowland@psionic.com> 			*/
/*			    <crowland@bipolar.net>			*/
/* Created: 10-12-97							*/
/* Modified: 05-25-98							*/
/*									*/
/* Send all changes/modifications/bugfixes to the above addresses.	*/
/*									*/
/*									*/
/* This software is Copyright(c) 1997-98 Craig H. Rowland		*/
/*									*/
/* Disclaimer:								*/
/*									*/
/* All software distributed by Craig H. Rowland ("the author") and	*/
/* Psionic Systems is distributed AS IS and carries NO WARRANTY or	*/
/* GUARANTEE OF ANY KIND. End users of the software acknowledge that	*/
/* they will not hold the author, Psionic Systems, and any employer of	*/
/* the author liable for failure or non-function of a software		*/
/* product. YOU ARE USING THIS PRODUCT AT YOUR OWN RISK			*/
/*									*/
/* Licensing restrictions apply. See the license that came with this	*/
/* file for more information or visit http://www.psionic.com for more	*/
/* information.								*/
/*									*/
/*$Id: sentry.c,v 1.16 1998/05/28 13:19:11 crowland Exp crowland $*/
/************************************************************************/


#include "sentry.h"
#include "sentry_config.h"


/* Global variable for state engine */
char scanDetectHost[MAXSTATE][IPMAXBUF];
int scanDetectCount=0;

/* here we go */
void main(int argc, char *argv[])
{
/* check args */
   if(argc != 2)
   {
      usage();
      abacusExit(ERROR);
   }

   if((geteuid()) && (getuid()) != 0)
   {
      printf("You need to be root to run this.\n");
      abacusExit(ERROR);
   }

   if(strcmp(argv[1],"-tcp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryModeTCP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }

/* Stealth TCP detection */
#ifdef SUPPORT_STEALTH
   else if(strcmp(argv[1],"-stcp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryStealthModeTCP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }

   /* Advanced Stealth TCP detection */
   else if(strcmp(argv[1],"-atcp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryAdvancedStealthModeTCP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }

   /* "stealth" UDP detection */
   else if(strcmp(argv[1],"-sudp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryStealthModeUDP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }

   /* Advanced "stealth" UDP detection */
   else if(strcmp(argv[1],"-audp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryAdvancedStealthModeUDP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }
#endif
   
   /* else do a UDP port bind */
   else if(strcmp(argv[1],"-udp") == 0)
   {
      abacusStart();
      if(abacusCheckConfig() == FALSE)
      {
	 abacusLog("adminalert: ERROR: Configuration files are missing/corrupted. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(abacusDaemonSeed() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into daemon mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
      if(sentryModeUDP() == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not go into sentry mode. Shutting down.\n");
	 abacusExit(ERROR);
      }
   }
   else
      usage();

   abacusExit(TRUE);
}


#ifdef SUPPORT_STEALTH
/****************************************************************/
/* Stealth scan detection Mode One				*/
/* 								*/
/* This mode will read in a list of ports to monitor and will 	*/
/* then open a raw socket to look for packets matching the port.*/
/* 								*/
/****************************************************************/
int sentryStealthModeTCP(void)
{
   struct sockaddr_in client, server;
   int portCount=0, portCount2=0, ports[MAXSOCKS], ports2[MAXSOCKS], result=TRUE;
   int count=0,scanDetectTrigger=TRUE, gotBound=FALSE;
   int testSockfd, openSockfd, incomingPort;
   int syn=0, fin=0, ack=0, urg=0, psh=0, rst=0;
   char logbuffer[MAXBUF], *temp, target[IPMAXBUF], configToken[MAXBUF];
   char resolvedHost[DNSMAXBUF];
   struct hostent *hostPtr;
   struct ipPacket 
   {
      struct ip ipPkt;
      struct tcphdr tcpPkt;
   } packet;
   
   
   if((abacusConfigTokenRetrieve("TCP_PORTS", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read TCP_PORTS option from config file");
      abacusExit(ERROR);
   }

   /* break out the ports */
   if((temp=(char *)strtok(configToken, ",")) != NULL)
   {
      ports[0]=atoi(temp);
      for(count=1; count < MAXSOCKS; count++)
      {
	 if((temp=(char *)strtok(NULL,",")) != NULL)
	    ports[count]=atoi(temp);
	 else
	    break;
      }
      portCount=count;
   }
   else
   {
      abacusLog("adminalert: ERROR: No TCP ports supplied in config file. Aborting");
      abacusExit(ERROR);
   }

   /* ok, now check if they have a network daemon on the socket already, if they do */
   /* then skip that port because it will cause false alarms */
   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Going into stealth listen mode on TCP port: %d\n",ports[count]);
      abacusLog(logbuffer);
      if((openSockfd=abacusOpenTCPSocket()) == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not open TCP socket. Aborting.\n");
	 abacusExit(ERROR);
      }

      if(abacusBindSocket(openSockfd, client, server, ports[count]) == ERROR)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Socket %d is in use and will not be monitored. Attempting to continue\n", ports[count]);
	 abacusLog(logbuffer);
      }
      else /* well we at least bound to one socket so we'll continue */
      {
	 gotBound=TRUE;
         ports2[portCount2++] = ports[count];
      }
      close(openSockfd);
   }

   /* if we didn't bind to anything then abort */
   if(gotBound == FALSE)
   {
      abacusLog("adminalert: ERROR: All supplied TCP sockets are in use and will not be listened to. Shutting down.\n");
      abacusExit(ERROR);
   }

   /* Open our raw socket for network IO */
   if((openSockfd=abacusOpenRAWTCPSocket()) == ERROR)   
   {
      abacusLog("adminalert: ERROR: could not open RAW TCP socket. Aborting.\n");
      abacusExit(ERROR);
   }

   abacusLog("adminalert: Sentry is now active and listening.\n");

   /* main detection loop */   
   for( ; ; )
   {
      	read(openSockfd, (struct ipPacket *) &packet, sizeof(packet));
      	incomingPort=ntohs(packet.tcpPkt.th_dport);
    
        /* check for an ACK to weed out established connections in case the user */
        /* is monitoring high ephemeral port numbers */  
       	if (!(packet.tcpPkt.th_flags & TH_ACK))
        {
        	/* this iterates the list of ports looking for a match */
           	for(count = 0; count < portCount; count++)
                {
                   if(incomingPort == ports2[count])
                   {
               		/* Ok here is where I "Smart-Verify" the socket. If the port was previously */
                        /* unbound, but now appears to have someone there, then we will skip responding */
                        /* to this inbound packet. This essentially is a "stateful" inspection of the */
                        /* the connection */
                        if((testSockfd=abacusOpenTCPSocket()) == ERROR)
                           abacusLog("adminalert: ERROR: could not open TCP socket to smart-verify. Blocking host.\n");
               		else
                        {
                 	if(abacusBindSocket(testSockfd, client, server, incomingPort) == ERROR)
                           {
#ifdef DEBUG
                        	snprintf(logbuffer, MAXBUF, "debug: sentryAdvancedStealthModeTCP: Smart-Verify Port In Use: %d", incomingPort);
                        	abacusLog(logbuffer);
#endif
                                close(testSockfd);
                                break;
                           }
                        else
                           close(testSockfd);
                        }
                        
                        /* copy the clients address into our buffer for nuking */
                        strncpy(target, inet_ntoa(packet.ipPkt.ip_src), IPMAXBUF);
                        target[IPMAXBUF-1]='\0';
                        /* check if we should ignore this IP */
                        result = abacusNeverBlock(target);

                        if(result == ERROR)
                        {
                        	snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
                                abacusLog(logbuffer);
                                result = FALSE;
                        }

                  	if(result == FALSE)
                        {
                                /* check if they've visited before */
                                scanDetectTrigger=checkStateEngine(target);
                        	if(scanDetectTrigger == TRUE)
                                {
                                	hostPtr = gethostbyaddr((char *)&packet.ipPkt.ip_src,                                                                                          	sizeof(packet.ipPkt.ip_src),AF_INET);
                                        if(hostPtr != NULL)
                                           snprintf(resolvedHost, DNSMAXBUF, "%s", hostPtr->h_name);
                                        else
                                           snprintf(resolvedHost, DNSMAXBUF, "%s", target);

                                	if(packet.tcpPkt.th_flags & TH_FIN)
                                        {
                                           snprintf(logbuffer, MAXBUF, "attackalert: FIN stealth scan from host: %s/%s to TCP port: %d", resolvedHost, target ,ports2[count]);
                                           abacusLog(logbuffer);
                                        }
                                	else if(packet.tcpPkt.th_flags & TH_SYN)
                                        {
                                           snprintf(logbuffer, MAXBUF, "attackalert: SYN/Normal scan from host: %s/%s to TCP port: %d", resolvedHost, target ,ports2[count]);
                                           abacusLog(logbuffer);
                                        }
                                	else
                                        {
                                           snprintf(logbuffer, MAXBUF, "attackalert: Unknown packet type from host: %s/%s to TCP port: %d", resolvedHost, target ,ports2[count]);
                                           abacusLog(logbuffer);
                                           if(packet.tcpPkt.th_flags & TH_SYN)
                                              syn=1;
                                           if(packet.tcpPkt.th_flags & TH_FIN)
                                              fin=1;
                                           if(packet.tcpPkt.th_flags & TH_ACK)
                                              ack=1;
                                           if(packet.tcpPkt.th_flags & TH_URG)
                                              urg=1;
                                           if(packet.tcpPkt.th_flags & TH_PUSH)
                                              psh=1;
                                           if(packet.tcpPkt.th_flags & TH_RST)
                                              rst=1;
                                           snprintf(logbuffer, MAXBUF, "attackalert: Packet Flags: SYN: %d FIN: %d ACK: %d PSH: %d URG: %d RST: %d", syn, fin, ack, psh, urg, rst);
                                	   abacusLog(logbuffer);
                                           syn=0; fin=0; ack=0; urg=0; psh=0; rst=0;
                                        }

                                	/* check if this target is already blocked */
                                        if(abacusIsBlocked(target) == FALSE)
                                        {
                                        	/* toast the prick */
                                                if(disposeTCP(target) != TRUE)
                                                {
                                        		snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s/%s !!"                                                            	,resolvedHost, target);
                                                        abacusLog(logbuffer);
                                                }
                                                else
                                                   	abacusWriteBlocked(target, resolvedHost, ports2[count]);
                                        } /* end IsBlocked check */
                                        else
                                        {
                                        	snprintf(logbuffer, MAXBUF, "attackalert: Host: %s/%s is already blocked Ignoring", resolvedHost, target);
                                                abacusLog(logbuffer);
                                        }
                                } /* end if(scanDetectTrigger) */
                        } /* end if(never block) check */
                        break; /* get out of for(count) loop above */
                   } /* end if(incoming port) ==  protected port */
                } /* end for( check for protected port loop ) loop */
         } /* end if(TH_ACK) check */
   } /* end for( ; ; ) loop */

} /* end sentryStealthModeTCP */


/****************************************************************/
/* Advanced Stealth scan detection Mode One			*/
/* 								*/
/* This mode will see what ports are listening below 1024	*/
/* and will then monitor all the rest. This is very sensitive 	*/
/* and will react on any packet hitting any monitored port,	*/
/* regardless of TCP flags set 					*/
/* 								*/
/****************************************************************/
int sentryAdvancedStealthModeTCP(void)
{
   struct sockaddr_in client, server;
   int result=TRUE, scanDetectTrigger=TRUE, hotPort=TRUE;
   int testSockfd, openSockfd, incomingPort, smartVerify=0;
   unsigned int advancedPorts=1024;
   unsigned int count=0, inUsePorts[MAXSOCKS], portCount=0;
   int syn=0, fin=0, ack=0, urg=0, psh=0, rst=0;
   char logbuffer[MAXBUF], target[IPMAXBUF], configToken[MAXBUF];
   char resolvedHost[DNSMAXBUF], *temp;
   struct hostent *hostPtr;
   struct ipPacket 
   {
      struct ip ipPkt;
      struct tcphdr tcpPkt;
   } packet;


   if((abacusConfigTokenRetrieve("ADVANCED_PORTS_TCP", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read ADVANCED_PORTS_TCP option from config file. Assuming 1024.");
      advancedPorts=1024;
   }
   else
      advancedPorts=atoi(configToken);

   snprintf(logbuffer, MAXBUF, "adminalert: Advanced mode will monitor first %d ports", advancedPorts);
   abacusLog(logbuffer);  

   /* try to bind to all ports below 1024, any that are taken we exclude later */
   for(count = 0 ; count < advancedPorts; count++)
    {
    	if((openSockfd=abacusOpenTCPSocket()) == ERROR)
        {
               	abacusLog("adminalert: ERROR: could not open TCP socket. Aborting.\n");
                abacusExit(ERROR);
        }
      	if(abacusBindSocket(openSockfd, client, server, count) == ERROR)
           inUsePorts[portCount++] = count;

        close(openSockfd);
    }

   if((abacusConfigTokenRetrieve("ADVANCED_EXCLUDE_TCP", configToken)) != FALSE)
   {
    /* break out the ports */
    if((temp=(char *)strtok(configToken, ",")) != NULL)
     {
      inUsePorts[portCount++]=atoi(temp);
      snprintf(logbuffer, MAXBUF, "adminalert: Advanced mode will manually exclude port: %d ", 
         	inUsePorts[portCount-1]);
      abacusLog(logbuffer);  
      for(count=0; count < MAXSOCKS; count++)
      {
	 if((temp=(char *)strtok(NULL,",")) != NULL)
          {
           inUsePorts[portCount++]=atoi(temp);
           snprintf(logbuffer, MAXBUF, "adminalert: Advanced mode will manually exclude port: %d ", 
                	inUsePorts[portCount-1]);
           abacusLog(logbuffer);  
          }
	 else
	    break;
      }
     }
   }
   else
      abacusLog("adminalert: Advanced mode will manually exclude no ports");
      
      
   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Advanced Stealth scan detection mode activated. Ignored TCP port: %d\n",inUsePorts[count]);
      abacusLog(logbuffer);
   }
     

   /* open raw socket for reading */
   if((openSockfd=abacusOpenRAWTCPSocket()) == ERROR)   
   {
      abacusLog("adminalert: ERROR: could not open RAW TCP socket. Aborting.\n");
      abacusExit(ERROR);
   }

   abacusLog("adminalert: Sentry is now active and listening.\n");

   /* main detection loop */    
   for( ; ; )
   {
        smartVerify=0;
      	read(openSockfd, (struct ipPacket *) &packet, sizeof(packet));
       	incomingPort=ntohs(packet.tcpPkt.th_dport);

   	/* don't monitor packets with ACK set (established) */
        /* This could be a hole in some cases */
   	if(!(packet.tcpPkt.th_flags & TH_ACK))
        {
   	        /* check if we should ignore this connection to this port */
                for(count = 0; count < portCount; count++)
                { 
                  if((incomingPort == inUsePorts[count]) || (incomingPort > advancedPorts))
                     {
                     	hotPort = FALSE;
                        break;
                     }
                  else
                     hotPort = TRUE;
                }
              
                if(hotPort)
                   {
                        /* Ok here is where I "Smart-Verify" the socket. If the port was previously */
                        /* unbound, but now appears to have someone there, then we will skip responding */
                        /* to this inbound packet. This essentially is a "stateful" inspection of the */
                        /* the connection */
                        if((testSockfd=abacusOpenTCPSocket()) == ERROR)
                	 abacusLog("adminalert: ERROR: could not open TCP socket to smart-verify. Blocking host.\n");
                        else
                  	 if(abacusBindSocket(testSockfd, client, server, incomingPort) == ERROR)
                         {
#ifdef DEBUG
                           snprintf(logbuffer, MAXBUF, "debug: sentryAdvancedStealthModeTCP: Smart-Verify Port In Use: %d", incomingPort);
                           abacusLog(logbuffer);
#endif
                           close(testSockfd);
                           smartVerify=1;
                         }
                         else
                           close(testSockfd);

                	if(smartVerify != 1)
                        {                         
                        /* copy the clients address into our buffer for nuking */
                        strncpy(target, inet_ntoa(packet.ipPkt.ip_src), IPMAXBUF);
                        target[IPMAXBUF-1]='\0';
                        /* check if we should ignore this IP */
                        result = abacusNeverBlock(target);

                        if(result == ERROR)
                        {
                        	snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
                                abacusLog(logbuffer);
                                result = FALSE;
                        }

                  	if(result == FALSE)
                        {
                        	/* report the connection */
                                /* check if they've visited before */
                                scanDetectTrigger=checkStateEngine(target);

                        	if(scanDetectTrigger == TRUE)
                                {
                                	hostPtr = gethostbyaddr((char *)&packet.ipPkt.ip_src,                                                                                          	sizeof(packet.ipPkt.ip_src),AF_INET);
                                        if(hostPtr != NULL)
                                           snprintf(resolvedHost, DNSMAXBUF, "%s", hostPtr->h_name);
                                        else
                                           snprintf(resolvedHost, DNSMAXBUF, "%s", target);

                                	if(packet.tcpPkt.th_flags & TH_FIN)
                                        {
                                           snprintf(logbuffer, MAXBUF, "attackalert: FIN stealth scan from host: %s/%s to TCP port: %d", resolvedHost, target , incomingPort);
                                           abacusLog(logbuffer);
                                        }
                                	else if(packet.tcpPkt.th_flags & TH_SYN)
                                        {
                                           snprintf(logbuffer, MAXBUF, "attackalert: SYN/Normal scan from host: %s/%s to TCP port: %d", resolvedHost, target , incomingPort);
                                           abacusLog(logbuffer);
                                        }
                                	else
                                        {
                                           snprintf(logbuffer, MAXBUF, "attackalert: Unknown packet type from host: %s/%s to TCP port: %d", resolvedHost, target , incomingPort);
                                           abacusLog(logbuffer);
                                           if(packet.tcpPkt.th_flags & TH_SYN)
                                              syn=1;
                                           if(packet.tcpPkt.th_flags & TH_FIN)
                                              fin=1;
                                           if(packet.tcpPkt.th_flags & TH_ACK)
                                              ack=1;
                                           if(packet.tcpPkt.th_flags & TH_URG)
                                              urg=1;
                                           if(packet.tcpPkt.th_flags & TH_PUSH)
                                              psh=1;
                                           if(packet.tcpPkt.th_flags & TH_RST)
                                              rst=1;
                                           snprintf(logbuffer, MAXBUF, "attackalert: Packet Flags: SYN: %d FIN: %d ACK: %d PSH: %d URG: %d RST: %d", syn, fin, ack, psh, urg, rst);
                                	   abacusLog(logbuffer);
                                           syn=0; fin=0; ack=0; urg=0; psh=0; rst=0;
                                        }

                                	/* check if this target is already blocked */
                                        if(abacusIsBlocked(target) == FALSE)
                                        {
                                        	/* toast the prick */
                                                if(disposeTCP(target) != TRUE)
                                                {
                                        		snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s/%s!!", resolvedHost, target);
                                                        abacusLog(logbuffer);
                                                }
                                                else
                                                   	abacusWriteBlocked(target, resolvedHost, incomingPort);
                                        } /* end IsBlocked check */
                                        else
                                        {
                                        	snprintf(logbuffer, MAXBUF, "attackalert: Host: %s/%s is already blocked Ignoring", resolvedHost, target);
                                                abacusLog(logbuffer);
                                        }
                                } /* end if(scanDetectTrigger) */
                        } /* end if(never block) check */
                      } /* end if(smartVerify) */  
                   } /* end if(hotPort) */
         } /* end if(TH_ACK) */
   } /* end for( ; ; ) loop */
}
/* end sentryAdvancedStealthModeTCP */



/****************************************************************/
/* UDP "stealth" scan detection 				*/
/* 								*/
/* This mode will read in a list of ports to monitor and will 	*/
/* then open a raw socket to look for packets matching the port.*/
/* 								*/
/****************************************************************/
int sentryStealthModeUDP(void)
{
   struct sockaddr_in client, server;
   int portCount=0, portCount2=0, ports[MAXSOCKS], ports2[MAXSOCKS], result=TRUE;
   int count=0,scanDetectTrigger=TRUE, gotBound=FALSE;
   int testSockfd, openSockfd, incomingPort;
   char logbuffer[MAXBUF], *temp, target[IPMAXBUF], configToken[MAXBUF];
   char resolvedHost[DNSMAXBUF];
   struct hostent *hostPtr;
   struct ipPacket 
   {
      struct ip ipPkt;
      struct udphdr udpPkt;
   } packet;
   
   
   if((abacusConfigTokenRetrieve("UDP_PORTS", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read UDP_PORTS option from config file");
      abacusExit(ERROR);
   }

   /* break out the ports */
   if((temp=(char *)strtok(configToken, ",")) != NULL)
   {
      ports[0]=atoi(temp);
      for(count=1; count < MAXSOCKS; count++)
      {
	 if((temp=(char *)strtok(NULL,",")) != NULL)
	    ports[count]=atoi(temp);
	 else
	    break;
      }
      portCount=count;
   }
   else
   {
      abacusLog("adminalert: ERROR: No UDP ports supplied in config file. Aborting");
      abacusExit(ERROR);
   }

   /* ok, now check if they have a network daemon on the socket already, if they do */
   /* then skip that port because it will cause false alarms */
   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Going into stealth listen mode on UDP port: %d\n",ports[count]);
      abacusLog(logbuffer);
      if((openSockfd=abacusOpenUDPSocket()) == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not open UDP socket. Aborting.\n");
	 abacusExit(ERROR);
      }

      if(abacusBindSocket(openSockfd, client, server, ports[count]) == ERROR)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: Socket %d is in use and will not be monitored. Attempting to continue\n", ports[count]);
	 abacusLog(logbuffer);
      }
      else /* well we at least bound to one socket so we'll continue */
      {
	 gotBound=TRUE;
         ports2[portCount2++] = ports[count];
      }
      close(openSockfd);
   }

   /* if we didn't bind to anything then abort */
   if(gotBound == FALSE)
   {
      abacusLog("adminalert: ERROR: All supplied UDP sockets are in use and will not be listened to. Shutting down.\n");
      abacusExit(ERROR);
   }

   /* Open our raw socket for network IO */
   if((openSockfd=abacusOpenRAWUDPSocket()) == ERROR)   
   {
      abacusLog("adminalert: ERROR: could not open RAW UDP socket. Aborting.\n");
      abacusExit(ERROR);
   }

   abacusLog("adminalert: Sentry is now active and listening.\n");

   /* main detection loop */   
   for( ; ; )
   {
      	read(openSockfd, (struct ipPacket *) &packet, sizeof(packet));
      	incomingPort=ntohs(packet.udpPkt.uh_dport);
    
       	/* this iterates the list of ports looking for a match */
       	for(count = 0; count < portCount; count++)
        {
           if(incomingPort == ports2[count])
           {
               /* Ok here is where I "Smart-Verify" the socket. If the port was previously */
               /* unbound, but now appears to have someone there, then we will skip responding */
               /* to this inbound packet. This essentially is a "stateful" inspection of the */
               /* the connection */
               if((testSockfd=abacusOpenUDPSocket()) == ERROR)
     		  abacusLog("adminalert: ERROR: could not open UDP socket to smart-verify. Blocking host.\n");
               else
               {
          	 if(abacusBindSocket(testSockfd, client, server, incomingPort) == ERROR)
                 {
#ifdef DEBUG
                    snprintf(logbuffer, MAXBUF, "debug: sentryAdvancedStealthModeUDP: Smart-Verify Port In Use: %d", incomingPort);
                    abacusLog(logbuffer);
#endif
                    close(testSockfd);
                    break;
                 }
                 else
                   close(testSockfd);
               }
                   
                /* copy the clients address into our buffer for nuking */
                strncpy(target, inet_ntoa(packet.ipPkt.ip_src), IPMAXBUF);
                target[IPMAXBUF-1]='\0';
                /* check if we should ignore this IP */
                result = abacusNeverBlock(target);

                if(result == ERROR)
                {
               	  snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
                  abacusLog(logbuffer);
                  result = FALSE;
                }

          	if(result == FALSE)
                {
                  /* check if they've visited before */
                  scanDetectTrigger=checkStateEngine(target);
                  if(scanDetectTrigger == TRUE)
                  {
                  	hostPtr = gethostbyaddr((char *)&packet.ipPkt.ip_src,                                                                  	       	sizeof(packet.ipPkt.ip_src),AF_INET);
                        if(hostPtr != NULL)
                           snprintf(resolvedHost, DNSMAXBUF, "%s", hostPtr->h_name);
                        else
                           snprintf(resolvedHost, DNSMAXBUF, "%s", target);

                        snprintf(logbuffer, MAXBUF, "attackalert: UDP scan from host: %s/%s to UDP port: %d", resolvedHost, target ,ports2[count]);
                        abacusLog(logbuffer);

                       	/* check if this target is already blocked */
                        if(abacusIsBlocked(target) == FALSE)
                        {
                           /* toast the prick */
                           if(disposeUDP(target) != TRUE)
                           {
                              snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s/%s!!",resolvedHost, target);
                              abacusLog(logbuffer);
                           }
                           else
                       	      abacusWriteBlocked(target, resolvedHost, ports2[count]);
                        } /* end IsBlocked check */
                        else
                        {
                           snprintf(logbuffer, MAXBUF, "attackalert: Host: %s/%s is already blocked Ignoring", resolvedHost, target);
                	   abacusLog(logbuffer);
                        }
                   } /* end if(scanDetectTrigger) */
                } /* end if(never block) check */
                break; /* get out of for(count) loop above */
            } /* end if(incoming port) ==  protected port */
         } /* end for( check for protected port loop ) loop */
   } /* end for( ; ; ) loop */

} /* end sentryStealthModeUDP */


/****************************************************************/
/* Advanced Stealth scan detection mode for UDP 		*/
/* 								*/
/* This mode will see what ports are listening below 1024	*/
/* and will then monitor all the rest. This is very sensitive 	*/
/* and will react on any packet hitting any monitored port.	*/
/* This is a very dangerous option and is for advanced users 	*/
/* 								*/
/****************************************************************/
int sentryAdvancedStealthModeUDP(void)
{
   struct sockaddr_in client, server;
   int result=TRUE, scanDetectTrigger=TRUE, hotPort=TRUE;
   int testSockfd, openSockfd, incomingPort, smartVerify=0;
   unsigned int advancedPorts=1024;
   unsigned int count=0, inUsePorts[MAXSOCKS], portCount=0;
   char logbuffer[MAXBUF], target[IPMAXBUF], configToken[MAXBUF];
   char resolvedHost[DNSMAXBUF], *temp;
   struct hostent *hostPtr;
   struct ipPacket 
   {
      struct ip ipPkt;
      struct udphdr udpPkt;
   } packet;


   if((abacusConfigTokenRetrieve("ADVANCED_PORTS_UDP", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read ADVANCED_PORTS_UDP option from config file. Assuming 1024.");
      advancedPorts=1024;
   }
   else
      advancedPorts=atoi(configToken);

   snprintf(logbuffer, MAXBUF, "adminalert: Advanced mode will monitor first %d ports", advancedPorts);
   abacusLog(logbuffer);  

   /* try to bind to all ports below 1024, any that are taken we exclude later */
   for(count = 0 ; count < advancedPorts; count++)
    {
    	if((openSockfd=abacusOpenUDPSocket()) == ERROR)
        {
               	abacusLog("adminalert: ERROR: could not open UDP socket. Aborting.\n");
                abacusExit(ERROR);
        }
      	if(abacusBindSocket(openSockfd, client, server, count) == ERROR)
           inUsePorts[portCount++] = count;

        close(openSockfd);
    }
      
   if((abacusConfigTokenRetrieve("ADVANCED_EXCLUDE_UDP", configToken)) != FALSE)
   {
    /* break out the ports */
    if((temp=(char *)strtok(configToken, ",")) != NULL)
     {
      inUsePorts[portCount++]=atoi(temp);
      snprintf(logbuffer, MAXBUF, "adminalert: Advanced mode will manually exclude port: %d ", 
         	inUsePorts[portCount-1]);
      abacusLog(logbuffer);  
      for(count=0; count < MAXSOCKS; count++)
      {
	 if((temp=(char *)strtok(NULL,",")) != NULL)
          {
           inUsePorts[portCount++]=atoi(temp);
           snprintf(logbuffer, MAXBUF, "adminalert: Advanced mode will manually exclude port: %d ", 
                	inUsePorts[portCount-1]);
           abacusLog(logbuffer);  
          }
	 else
	    break;
      }
     }
   }
   else
      abacusLog("adminalert: Advanced mode will manually exclude no ports");


   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Advanced Stealth scan detection mode activated. Ignored UDP port: %d\n",inUsePorts[count]);
      abacusLog(logbuffer);
   }
     
   /* open raw socket for reading */
   if((openSockfd=abacusOpenRAWUDPSocket()) == ERROR)   
   {
      abacusLog("adminalert: ERROR: could not open RAW UDP socket. Aborting.\n");
      abacusExit(ERROR);
   }

   abacusLog("adminalert: Sentry is now active and listening.\n");

   /* main detection loop */    
   for( ; ; )
   {
        smartVerify=0;
      	read(openSockfd, (struct ipPacket *) &packet, sizeof(packet));
       	incomingPort=ntohs(packet.udpPkt.uh_dport);

   	        /* check if we should ignore this connection to this port */
                for(count = 0; count < portCount; count++)
                { 
                  if((incomingPort == inUsePorts[count]) || (incomingPort > advancedPorts))
                     {
                     	hotPort = FALSE;
                        break;
                     }
                  else
                     hotPort = TRUE;
                }
              
                if(hotPort)
                   {
                        /* Ok here is where I "Smart-Verify" the socket. If the port was previously */
                        /* unbound, but now appears to have someone there, then we will skip responding */
                        /* to this inbound packet. This essentially is a "stateful" inspection of the */
                        /* the connection */
                        if((testSockfd=abacusOpenUDPSocket()) == ERROR)
                	 abacusLog("adminalert: ERROR: could not open UDP socket to smart-verify. Blocking host.\n");
                        else
                  	 if(abacusBindSocket(testSockfd, client, server, incomingPort) == ERROR)
                         {
#ifdef DEBUG
                           snprintf(logbuffer, MAXBUF, "debug: sentryAdvancedStealthModeUDP: Smart-Verify Port In Use: %d", incomingPort);
                           abacusLog(logbuffer);
#endif
                           close(testSockfd);
                           smartVerify=1;
                         }
                         else
                           close(testSockfd);
                           
                        if(smartVerify != 1)
                        {
                        /* copy the clients address into our buffer for nuking */
                        strncpy(target, inet_ntoa(packet.ipPkt.ip_src), IPMAXBUF);
                        target[IPMAXBUF-1]='\0';
                        /* check if we should ignore this IP */
                        result = abacusNeverBlock(target);

                        if(result == ERROR)
                        {
                        	snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
                                abacusLog(logbuffer);
                                result = FALSE;
                        }

                  	if(result == FALSE)
                        {
                        	/* report the connection */
                                /* check if they've visited before */
                                scanDetectTrigger=checkStateEngine(target);

                        	if(scanDetectTrigger == TRUE)
                                {
                                	hostPtr = gethostbyaddr((char *)&packet.ipPkt.ip_src,                                                                                          	sizeof(packet.ipPkt.ip_src),AF_INET);
                                        if(hostPtr != NULL)
                                           snprintf(resolvedHost, DNSMAXBUF, "%s", hostPtr->h_name);
                                        else
                                           snprintf(resolvedHost, DNSMAXBUF, "%s", target);

        	                        snprintf(logbuffer, MAXBUF, "attackalert: UDP scan from host: %s/%s to UDP port: %d",resolvedHost, target , incomingPort);
        	                        abacusLog(logbuffer);

                                	/* check if this target is already blocked */
                                        if(abacusIsBlocked(target) == FALSE)
                                        {
                                        	/* toast the prick */
                                                if(disposeUDP(target) != TRUE)
                                                {
                                        		snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s/%s!!", resolvedHost, target);
                                                        abacusLog(logbuffer);
                                                }
                                                else
                                                   	abacusWriteBlocked(target, resolvedHost, incomingPort);
                                        } /* end IsBlocked check */
                                        else
                                        {
                                        	snprintf(logbuffer, MAXBUF, "attackalert: Host: %s/%s is already blocked Ignoring", resolvedHost, target);
                                                abacusLog(logbuffer);
                                        }
                                } /* end if(scanDetectTrigger) */
                        } /* end if(never block) check */
                     } /* end if (smartVerify) */
                   } /* end if(hotPort) */
   } /* end for( ; ; ) loop */
}
/* end sentryAdvancedStealthModeUDP */

#endif




/****************************************************************/
/* Classic detection Mode 					*/
/* 								*/
/* This mode will bind to a list of TCP sockets and wait for  	*/
/* connections to happen. Although the least prone to false 	*/
/* alarms, it also won't detect stealth scans			*/
/* 								*/
/****************************************************************/
int sentryModeTCP(void)
{

   struct sockaddr_in client, server;
   int length, portCount=0, ports[MAXSOCKS], openSockfd[MAXSOCKS], incomingSockfd, result=TRUE;
   int count=0,scanDetectTrigger=TRUE, showBanner=FALSE;
   int selectResult=0, gotBound=FALSE;
   char logbuffer[MAXBUF], *temp, target[IPMAXBUF], bannerBuffer[MAXBUF], configToken[MAXBUF];
   char resolvedHost[DNSMAXBUF];
   fd_set selectFds;
   struct hostent *hostPtr;

   if((abacusConfigTokenRetrieve("TCP_PORTS", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read TCP_PORTS option from config file");
      abacusExit(ERROR);
   }

   /* break out the ports */
   if((temp=(char *)strtok(configToken, ",")) != NULL)
   {
      ports[0]=atoi(temp);
      for(count=1; count < MAXSOCKS; count++)
      {
	 if((temp=(char *)strtok(NULL,",")) != NULL)
	    ports[count]=atoi(temp);
	 else
	    break;
      }
      portCount=count;
   }
   else
   {
      abacusLog("adminalert: ERROR: No TCP ports supplied in config file. Aborting");
      abacusExit(ERROR);
   }

   /* read in the banner if one is given */
   if((abacusConfigTokenRetrieve("PORT_BANNER", configToken)) == TRUE)
   {
      showBanner=TRUE;
      strncpy(bannerBuffer, configToken, MAXBUF);
      bannerBuffer[MAXBUF-1]='\0';
   }


   /* setup select call */
   FD_ZERO(&selectFds);

   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Going into listen mode on TCP port: %d\n",ports[count]);
      abacusLog(logbuffer);
      if((openSockfd[count]=abacusOpenTCPSocket()) == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not open TCP socket. Aborting.\n");
	 abacusExit(ERROR);
      }

      if(abacusBindSocket(openSockfd[count], client, server, ports[count]) == ERROR)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: could not bind TCP socket: %d. Attempting to continue\n",             	  ports[count]);
	 abacusLog(logbuffer);
      }
      else /* well we at least bound to one socket so we'll continue */
	 gotBound=TRUE;
   }

   /* if we didn't bind to anything then abort */
   if(gotBound == FALSE)
   {
      abacusLog("adminalert: ERROR: could not bind ANY TCP sockets. Shutting down.\n");
      abacusExit(ERROR);
   }

   /* length for accept() call */
   length = sizeof(client);

   abacusLog("adminalert: Sentry is now active and listening.\n");

   /* main loop for multiplexing/resetting */
   for( ; ; )
   {
      	 /* set up select call */
	 for(count = 0; count < portCount; count++)
            FD_SET(openSockfd[count], &selectFds);

   	 /* setup the select multiplexing (blocking mode)*/
	 selectResult=select(MAXSOCKS, &selectFds, NULL, NULL, (struct timeval *)NULL);

   	 /* something blew up */
	 if(selectResult < 0)
	 {
	    abacusLog("adminalert: ERROR: select call failed. Shutting down.\n");
	    abacusExit(ERROR);
	 }
      else if(selectResult == 0)
      {
#ifdef DEBUG
	 abacusLog("Select timeout");
#endif
      }

   	 /* select is reporting a waiting socket. Poll them all to find out which */
	 else if(selectResult > 0)
	 {
	    for(count = 0; count < portCount; count++)
	    {
	       if(FD_ISSET(openSockfd[count], &selectFds))
	       {
		  incomingSockfd = accept(openSockfd[count], (struct sockaddr *)&client, &length);
		  if(incomingSockfd < 0)
		  {
		     snprintf(logbuffer, MAXBUF, "attackalert: Possible stealth scan from unknown host to TCP port: %d                               (accept failed)",ports[count]);
		     abacusLog(logbuffer);
		     break;
		  }

        	  /* copy the clients address into our buffer for nuking */
		  strncpy(target, inet_ntoa(client.sin_addr.s_addr), IPMAXBUF);
		  target[IPMAXBUF-1]='\0';
                  /* check if we should ignore this IP */
		  result = abacusNeverBlock(target);

		  if(result == ERROR)
		  {
		     snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
		     abacusLog(logbuffer);
		     result = FALSE;
		  }

		  if(result == FALSE)
		  {
                     /* report the connection */
                     /* check if they've visited before */
  		     scanDetectTrigger=checkStateEngine(target);

                     if(scanDetectTrigger == TRUE)
		     {
                        /* show the banner if one was selected */
			if(showBanner == TRUE)
                           write(incomingSockfd, bannerBuffer, strlen(bannerBuffer));
                        /* we don't need the bonehead anymore */
			close(incomingSockfd);
                        hostPtr = gethostbyaddr((char *)&client.sin_addr.s_addr, sizeof(client.sin_addr.s_addr),                                   AF_INET);
                        if(hostPtr != NULL)
                           snprintf(resolvedHost, DNSMAXBUF, "%s", hostPtr->h_name);
                        else
                           snprintf(resolvedHost, DNSMAXBUF, "%s", target);

               	   	snprintf(logbuffer, MAXBUF, "attackalert: Connect from host: %s/%s to TCP port: %d",
                                 resolvedHost, target ,ports[count]);
                        abacusLog(logbuffer);

                        /* check if this target is already blocked */
                   	if(abacusIsBlocked(target) == FALSE)
			{
                           /* toast the prick */
			   if(disposeTCP(target) != TRUE)
			   {
			      snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s !!"
                                       ,target);
			      abacusLog(logbuffer);
			   }
			   else
			      abacusWriteBlocked(target, resolvedHost, ports[count]);
			}
			else
			{
			   snprintf(logbuffer, MAXBUF, "attackalert: Host: %s is already blocked. Ignoring",
			            target);
			   abacusLog(logbuffer);
			}
		     }
		  }
                  /* never block this person*/
		  else
		     close(incomingSockfd);

                  /* safety */
		  close(incomingSockfd);
	       }/* end if(FD_ISSET) */
	    } /* end for() */
	 } /* end else (selectResult > 0) */
   } /* end main for(; ; ) loop */

/* not reached */
close(incomingSockfd);
}





/****************************************************************/
/* Classic detection Mode 					*/
/* 								*/
/* This mode will bind to a list of UDP sockets and wait for  	*/
/* connections to happen. Stealth scanning really doesn't apply */
/* here.							*/
/* 								*/
/****************************************************************/
int sentryModeUDP(void)
{
   struct sockaddr_in client, server;
   int length, ports[MAXSOCKS], openSockfd[MAXSOCKS], result=TRUE;
   int count=0, portCount=0, selectResult=0, scanDetectTrigger=0, gotBound=FALSE, showBanner=FALSE;
   char logbuffer[MAXBUF], *temp, target[IPMAXBUF], bannerBuffer[MAXBUF],configToken[MAXBUF], buffer[MAXBUF];
   char resolvedHost[DNSMAXBUF];
   fd_set selectFds;
   struct hostent *hostPtr;

   if((abacusConfigTokenRetrieve("UDP_PORTS", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read UDP_PORTS option from config file");
      abacusExit(ERROR);
   }

   /* break out the ports */
   if((temp=(char *)strtok(configToken, ",")) != NULL)
   {
      ports[0]=atoi(temp);
	 for(count=1; count < MAXSOCKS; count++)
         {
         	if((temp=(char *)strtok(NULL,",")) != NULL)
                   ports[count]=atoi(temp);
                else
                   break;
         }
      portCount=count;
   }
   else
   {
      abacusLog("adminalert: ERROR: No UDP ports supplied in config file. Aborting");
      abacusExit(ERROR);
   }

   /* read in the banner if one is given */
   if((abacusConfigTokenRetrieve("PORT_BANNER", configToken)) == TRUE)
   {
      showBanner=TRUE;
      strncpy(bannerBuffer, configToken, MAXBUF);
      bannerBuffer[MAXBUF-1]='\0';
   }

   /* setup select call */
   FD_ZERO(&selectFds);

   for(count=0; count < portCount; count++ )
   {
      snprintf(logbuffer, MAXBUF, "adminalert: Going into listen mode on UDP port: %d\n",ports[count]);
	       abacusLog(logbuffer);
      if((openSockfd[count]=abacusOpenUDPSocket()) == ERROR)
      {
	 abacusLog("adminalert: ERROR: could not open UDP socket. Aborting\n");
	 abacusExit(ERROR);
      }
      if(abacusBindSocket(openSockfd[count], client, server, ports[count]) == ERROR)
      {
	 snprintf(logbuffer, MAXBUF, "adminalert: ERROR: could not bind UDP socket: %d. Attempting to continue\n",
                  ports[count]);         
         abacusLog(logbuffer);
      }
      else /* well we at least bound to one socket so we'll continue */
	 gotBound=TRUE;
   }

/* if we didn't bind to anything then abort */
   if(gotBound == FALSE)
   {
      abacusLog("adminalert: ERROR: could not bind ANY UDP sockets. Shutting down.\n");
	 abacusExit(ERROR);
   }


/* length for accept() call */
length = sizeof(client);

abacusLog("adminalert: Sentry is now active and listening.\n");

/* main loop for multiplexing/resetting */
for( ; ; )
{
   	/* set up select call */
	 for(count = 0; count < portCount; count++)
            FD_SET(openSockfd[count], &selectFds);

   	/* setup the select multiplexing (blocking mode) */
	selectResult=select(MAXSOCKS, &selectFds, NULL, NULL, (struct timeval *)NULL);

      	if(selectResult < 0)
        {
           abacusLog("adminalert: ERROR: select call failed. Shutting down.\n");
	   abacusExit(ERROR);
        }
        else if(selectResult == 0)
        {
#ifdef DEBUG
	 abacusLog("Select timeout");
#endif
        }

   	/* select is reporting a waiting socket. Poll them all to find out which */
   	else if(selectResult > 0)
        {
	 for(count = 0; count < portCount; count++)
	 {
	    if(FD_ISSET(openSockfd[count], &selectFds))
	    {
            /* here just read in one byte from the UDP socket, that's all we need to */
            /* know that this person is a jerk */
	    if(recvfrom(openSockfd[count], buffer, 1, 0, (struct sockaddr *)&client, &length) < 0)
	    {
		snprintf(logbuffer, MAXBUF, "adminalert: ERROR: could not accept incoming socket for UDP port: %d\n"
                   	 , ports[count]);
		abacusLog(logbuffer);
		break;
	    }

            /* copy the clients address into our buffer for nuking */
	    strncpy(target,inet_ntoa(client.sin_addr.s_addr), IPMAXBUF);
	    target[IPMAXBUF-1]='\0';
#ifdef DEBUG
            snprintf(logbuffer, MAXBUF, "debug: sentryModeUDP: accepted UDP connection from: %s\n", target);
		     abacusLog(logbuffer);
#endif
            /* check if we should ignore this IP */
  	    result = abacusNeverBlock(target);
	    if(result == ERROR)
	    {
		snprintf(logbuffer, MAXBUF, "attackalert: ERROR: cannot open ignore file. Blocking host anyway.\n");
		         abacusLog(logbuffer);
  	        result = FALSE;
	    }
	    if(result == FALSE)
	    {
               /* check if they've visited before */
               scanDetectTrigger=checkStateEngine(target);
               if(scanDetectTrigger == TRUE)
               {
                  /* show the banner if one was selected */
                  if(showBanner == TRUE)
                     sendto(openSockfd[count], bannerBuffer, strlen(bannerBuffer),0,
			    (struct sockaddr *)&client, length);

                  hostPtr = gethostbyaddr((char *)&client.sin_addr.s_addr, sizeof(client.sin_addr.s_addr),                             AF_INET);
                  if(hostPtr != NULL)
                     snprintf(resolvedHost, DNSMAXBUF, "%s", hostPtr->h_name);
                  else
                     snprintf(resolvedHost, DNSMAXBUF, "%s", target);

      	   	  snprintf(logbuffer, MAXBUF, "attackalert: Connect from host: %s/%s to TCP port: %d",
                           resolvedHost, target ,ports[count]);
                  abacusLog(logbuffer);
                  /* check if this target is already blocked */
                  if(abacusIsBlocked(target)== FALSE)
                  {
                  /* toast the prick */
                     if(disposeUDP(target) != TRUE)
                     {
                  	snprintf(logbuffer, MAXBUF, "attackalert: ERROR: Could not block host %s !!"
			         ,target);
                        abacusLog(logbuffer);
                     }
                     else
                        abacusWriteBlocked(target, resolvedHost, ports[count]);
                  }
                  else
		  {
                     snprintf(logbuffer, MAXBUF, "attackalert: Host: %s is already blocked. Ignoring",
			      target);
                     abacusLog(logbuffer);
                  }
               }
            }
         }/* end if(FD_ISSET) */
        } /* end for() */
   } /* end else (selectResult > 0) */
} /* end main for(; ; ) loop */

} /* end UDP sentry */




/* kill the TCP connection depending on config option */
int disposeTCP(char *target)
{
   int status=TRUE;

/* Should we ignore TCP from active response? */
   if(abacusDoBlockTCP() == TRUE)
   {
/* run external command first, hosts.deny second, dead route last */
   	if(abacusKillRunCmd(target) != TRUE)
           status=FALSE;
        else if(abacusKillHostsDeny(target) != TRUE)
           status=FALSE;
        else if(abacusKillRoute(target) != TRUE)
           status=FALSE;
   }
   else
      abacusLog("attackalert: Ignoring TCP response per configuration file setting.");
           
   return(status);
}


/* kill the UDP connection depending on config option */
int disposeUDP(char *target)
{
   int status=TRUE;


/* Should we ignore UDP from active response? */
   if(abacusDoBlockUDP() == TRUE)
   {
/* run external command first, hosts.deny second, dead route last */
	 if(abacusKillRunCmd(target) != TRUE)
            status=FALSE;
	 else if(abacusKillHostsDeny(target) != TRUE)
            status=FALSE;
	 else if(abacusKillRoute(target) != TRUE)
            status=FALSE;
   }
   else
      abacusLog("attackalert: Ignoring UDP response per configuration file setting.");

   return(status);
}


/* duh */
void usage(void)
{
   printf("Sentry - Port Scan Detector.\n");
   printf("Copyright 1997,98 Craig H. Rowland <crowland@psionic.com>\n");
   printf("Licensing restrictions apply. Please see documentation\n");
   printf("Version: %s\n\n", VERSION);
#ifdef SUPPORT_STEALTH
            printf("usage: sentry [-tcp -udp -stcp -atcp -sudp -audp]\n\n");
#else
            printf("Stealth scan detection not supported on this platform\n");
            printf("usage: sentry [-tcp -udp]\n\n");
#endif
      printf("*** PLEASE READ THE DOCS BEFORE USING *** \n\n");
}



/* our cheesy state engine to monitor who has connected here before */
int checkStateEngine(char *target)
{
   int count=0, scanDetectTrigger=TRUE;
      int gotOne=0, configTriggerCount=0;
      char logbuffer[MAXBUF], configToken[MAXBUF];


/* get SCAN_TRIGGER option from config */
   if((abacusConfigTokenRetrieve("SCAN_TRIGGER", configToken)) == FALSE)
   {
      abacusLog("adminalert: ERROR: Could not read SCAN_TRIGGER option from config file. Disabling SCAN DETECTION");
	 configTriggerCount=0;
   }
   else
   {
#ifdef DEBUG
      snprintf(logbuffer, MAXBUF, "debug: sentryModeTCP: retrieved SCAN_TRIGGER option: %s \n", configToken);
	 abacusLog(logbuffer);
#endif
      configTriggerCount=atoi(configToken);
   }


/* This is the rather dumb scan state engine. It maintains     */
/* an array of past hosts who triggered a connection on a port */
/* when a new host arrives it is compared against the array */
/* if it is found in the array it increments a state counter by */
/* one and checks the remainder of the array. It does this until */
/* the end is reached or the trigger value has been exceeded */
/* This would probably be better as a linked list, but for the number */
/* of hosts we are tracking this is just as good. I'll probably change */
/* this in the future */

   gotOne=1; /* our flag counter if we get a match */
      scanDetectTrigger = TRUE; /* set to TRUE until set otherwise */

   if(configTriggerCount > 0 ) /* only do this if they want the option */
   { /* crawl through the array looking for a previous host connect */
      for(count = 0; count < MAXSTATE; count++)
      { /* if the array has the IP address then increment the gotOne counter and */
/* check the trigger value. If it is exceeded break out of the loop and */
/* set the detecttrigger to TRUE */
	    if(strstr(scanDetectHost[count], target) != (char)NULL)
	 { /* compare the number of matches to the configured trigger value */
/* if we've exceeded we can stop this noise */
	       if(++gotOne >= configTriggerCount )
	    {
	       scanDetectTrigger = TRUE; /* got one */
#ifdef DEBUG
	       snprintf(logbuffer, MAXBUF, "debug: checkStateEngine: host: %s has exceeded trigger value: %d\n", 		                    	scanDetectHost[count], configTriggerCount);
		abacusLog(logbuffer);
#endif
	       break;
	    }
	 }
/* Ok not a match, set the trigger to false and continue on */
	    else
 	       scanDetectTrigger = FALSE;
      }

/* now add the fresh meat into the state engine */
/* if our array is still less than MAXSTATE large add it to the end */
      if(scanDetectCount < MAXSTATE)
      {
	 strncpy(scanDetectHost[scanDetectCount], target, IPMAXBUF);
	    scanDetectHost[scanDetectCount][IPMAXBUF-1]='\0';
	    scanDetectCount++;
      }
      else /* otherwise tack it to the beginning and start overwriting older ones */
      {
	 scanDetectCount=0;
	    strncpy(scanDetectHost[scanDetectCount], target, IPMAXBUF);
	    scanDetectHost[scanDetectCount][IPMAXBUF-1]='\0';
      }

#ifdef DEBUG
      for(count = 0; count < MAXSTATE; count++)
      {
	 snprintf(logbuffer, MAXBUF, "debug: checkStateEngine: state engine host: %s -> position: %d Detected: %d\n",	    scanDetectHost[count], count, scanDetectTrigger);
	    abacusLog(logbuffer);
      }
#endif
/* end catch to set state if configTriggerCount == 0 */
      if(gotOne >= configTriggerCount )
	 scanDetectTrigger = TRUE;
   } /* end if(scanDetectTrigger > 0)  */


   if(configTriggerCount > MAXSTATE)
      {
	 snprintf(logbuffer, MAXBUF, "securityalert: WARNING: Trigger value %d is larger than state engine capacity of %d. Adjust the value lower or recompile with a larger state engine value.\n", configTriggerCount, MAXSTATE);
	 abacusLog(logbuffer);
         abacusLog("securityalert: Blocking host anyway because of invalid trigger value");
         scanDetectTrigger = TRUE;
      }
   return(scanDetectTrigger);
}





