/* -*-Mode: C;-*-
 * $Id: edsio.c 1.10 Tue, 16 Mar 1999 12:43:19 -0800 jmacd $
 * ser.c:
 *
 * Copyright (C) 1998, Josh MacDonald.
 * All Rights Reserved.
 *
 * Author: Josh MacDonald <jmacd@CS.Berkeley.EDU>
 */

#include "edsio.h"
#include <stdio.h>

/* Type-based selectors for unknown types
 */

typedef struct {
  const char* name;
  gboolean (*unserialize_func) ();
  gboolean (*serialize_func) ();
  guint    (*count_func) ();
  void     (*print_func) ();
  guint32    val;
} SerEntry;

static GArray   *ser_array;
static gboolean  ser_array_sorted;

static void serializeio_source_reset_allocation (SerialSource* source);

void
serializeio_initialize_type (const char* name,
			     guint32    val,
			     gboolean (*unserialize_func) (),
			     gboolean (*serialize_func) (),
			     guint    (*count_func) (),
			     void      (*print_func) ())
{
  SerEntry it;

  it.name = name;
  it.val = val;
  it.unserialize_func = unserialize_func;
  it.serialize_func = serialize_func;
  it.count_func = count_func;
  it.print_func = print_func;

  if (ser_array == NULL)
    ser_array = g_array_new (FALSE, TRUE, sizeof (SerEntry));

  g_array_append_val (ser_array, it);

  ser_array_sorted = FALSE;
}

static int
ser_entry_compare (const void* va, const void* vb)
{
  SerEntry* a = (SerEntry*) va;
  SerEntry* b = (SerEntry*) vb;

  return a->val - b->val;
}

static SerEntry*
serializeio_find_entry (SerialType type)
{
  gint high_index, low_index = 0, index;
  gboolean par = TRUE;
  gint count = 0;

  if (ser_array)
    {
      high_index = ser_array->len;

      if (! ser_array_sorted)
	{
	  ser_array_sorted = TRUE;
	  qsort (ser_array->data, ser_array->len, sizeof (SerEntry), ser_entry_compare);
	}

      /* Binary search. */
      for (;;)
	{
	  gint this_val;

	  index = low_index + (high_index-low_index+par)/2;

	  par = !par;

	  this_val = g_array_index (ser_array, SerEntry, index).val;

	  if (this_val == type)
	    return & g_array_index (ser_array, SerEntry, index);

	  if (low_index >= high_index-1 && count++ == 1)
	    break;

	  if (this_val < type)
	    low_index = index;
	  else
	    high_index = index;
	}
    }

  edsio_generate_intint_event (EC_EdsioUnregisteredType,
			       type & SER_LIBRARY_OFFSET_MASK,
			       type >> SER_LIBRARY_OFFSET_BITS);
  return NULL;
}

gboolean
serializeio_unserialize_generic_internal (SerialSource *source,
					  SerialType   *object_type,
					  void        **object,
					  gboolean      set_allocation)
{
  SerialType type = (* source->source_type) (source, set_allocation);
  SerEntry* ent;
  gboolean res = FALSE;

  if (type < 0)
    return FALSE;

  ent = serializeio_find_entry (type);

  (*object_type) = type;

  if (ent)
    {
      res = ent->unserialize_func (source, object);

      if (set_allocation && res)
	{
	  if (! serializeio_source_object_received (source))
	    return FALSE;
	}
    }

  if (set_allocation)
    serializeio_source_reset_allocation (source);

  return res;
}

gboolean
serializeio_unserialize_generic (SerialSource *source,
				 SerialType   *object_type,
				 void        **object)
{
  return serializeio_unserialize_generic_internal (source, object_type, object, TRUE);
}

gboolean
serializeio_serialize_generic (SerialSink    *sink,
			       SerialType     object_type,
			       void          *object)
{
  return serializeio_serialize_generic_internal (sink, object_type, object, TRUE);
}

gboolean
serializeio_serialize_generic_internal (SerialSink    *sink,
					SerialType     object_type,
					void          *object,
					gboolean       set_allocation)
{
  SerEntry* ent;
  gboolean res = FALSE;

  if (! (* sink->sink_type) (sink, object_type, set_allocation ? serializeio_generic_count (object_type, object) : 0, set_allocation))
    return FALSE;

  ent = serializeio_find_entry (object_type);

  if (ent)
    res = ent->serialize_func (sink, object);

  return res;
}

const char*
serializeio_generic_type_to_string (SerialType type)
{
  SerEntry* ent;
  const char* res = "*Unknown*";

  ent = serializeio_find_entry (type);

  if (ent)
    res = ent->name;

  return res;
}

guint
serializeio_generic_count (SerialType     object_type,
			   void          *object)
{
  SerEntry* ent;
  gboolean res = FALSE;

  ent = serializeio_find_entry (object_type);

  if (ent)
    res = ent->count_func (object);

  return res;
}

void
serializeio_generic_print (SerialType type, void* object, guint indent_spaces)
{
  SerEntry* ent;

  ent = serializeio_find_entry (type);

  if (ent)
    ent->print_func (object, indent_spaces);
  else
    {
      int i = 0;

      for (; i < indent_spaces; i += 1)
	g_print (" ");

      g_print ("*Type Not Registered*\n");
    }
}

gboolean
serializeio_unserialize_generic_acceptable (SerialSource *source,
					    guint32       accept,
					    SerialType   *object_type,
					    void        **object)
{
  gboolean s;

  s = serializeio_unserialize_generic (source, object_type, object);

  if (s)
    {
      if (accept != -1)
	{
	  if ((*object_type & SER_LIBRARY_OFFSET_MASK) != (accept & SER_LIBRARY_OFFSET_MASK))
	    {
	      edsio_generate_intint_event (EC_EdsioUnexpectedLibraryType,
					   accept & SER_LIBRARY_OFFSET_MASK,
					   *object_type & SER_LIBRARY_OFFSET_MASK);

	      return FALSE;
	    }

	  if (! ((*object_type & ~SER_LIBRARY_OFFSET_MASK) |
		 (accept       & ~SER_LIBRARY_OFFSET_MASK)))
	    {
	      edsio_generate_void_event (EC_EdsioUnexpectedType);

	      return FALSE;
	    }
	}
    }

  return s;
}

/* Segment source/sink impls
 */
typedef struct _HandleSerialSource HandleSerialSource;
typedef struct _HandleSerialSink HandleSerialSink;

struct _HandleSerialSource {
  SerialSource source;

  FileHandle* fh;

  const HandleFuncTable* table;
};

struct _HandleSerialSink {
  SerialSink sink;

  FileHandle* fh;

  const HandleFuncTable* table;
};

static SerialType handle_source_type           (SerialSource* source, gboolean set_allocation);
static gboolean   handle_source_close          (SerialSource* source);
static gboolean   handle_source_read           (SerialSource* source, guint8 *ptr, guint32 len);
static void       handle_source_free           (SerialSource* source);

static gboolean     handle_sink_type             (SerialSink* sink, SerialType type, guint len, gboolean set_allocation);
static gboolean     handle_sink_close            (SerialSink* sink);
static gboolean     handle_sink_write            (SerialSink* sink, const guint8 *ptr, guint32 len);
static void         handle_sink_free             (SerialSink* sink);

SerialSource*
serializeio_handle_source (FileHandle* fh, const HandleFuncTable* table)
{
  HandleSerialSource* it = g_new0 (HandleSerialSource, 1);

  serializeio_source_init (&it->source,
			   handle_source_type,
			   handle_source_close,
			   handle_source_read,
			   handle_source_free,
			   NULL,
			   NULL);

  it->fh = fh;
  it->table = table;

  return &it->source;
}

SerialSink*
serializeio_handle_sink (FileHandle* fh, const HandleFuncTable* table)
{
  HandleSerialSink* it = g_new0 (HandleSerialSink, 1);

  serializeio_sink_init (&it->sink,
			 handle_sink_type,
			 handle_sink_close,
			 handle_sink_write,
			 handle_sink_free,
			 NULL);

  it->fh = fh;
  it->table = table;

  return &it->sink;
}

SerialType
handle_source_type (SerialSource* source, gboolean set_allocation)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;
  guint32 x;

  if (! ssource->table->table_handle_getui (ssource->fh, &x))
    return ST_Error;

  if (set_allocation)
    {
      if (! ssource->table->table_handle_getui (ssource->fh, &source->alloc_total))
	return ST_Error;
    }

  return x;
}

gboolean
handle_source_close (SerialSource* source)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;

  return ssource->table->table_handle_close (ssource->fh, 0);
}

gboolean
handle_source_read (SerialSource* source, guint8 *ptr, guint32 len)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;

  return ssource->table->table_handle_read (ssource->fh, ptr, len) == len;
}

void
handle_source_free (SerialSource* source)
{
  g_free (source);
}

gboolean
handle_sink_type (SerialSink* sink, SerialType type, guint len, gboolean set_allocation)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  if (! ssink->table->table_handle_putui (ssink->fh, type))
    return FALSE;

  if (set_allocation && ! ssink->table->table_handle_putui (ssink->fh, len))
    return FALSE;

  return TRUE;
}

gboolean
handle_sink_close (SerialSink* sink)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  return ssink->table->table_handle_close (ssink->fh, 0);
}

gboolean
handle_sink_write (SerialSink* sink, const guint8 *ptr, guint32 len)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  return ssink->table->table_handle_write (ssink->fh, ptr, len);
}

void
handle_sink_free (SerialSink* sink)
{
  g_free (sink);
}

/* BASE64 Encoding
 */

static const unsigned char base64_table[64] = {
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
  'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a',
  'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o',
  'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2',
  '3', '4', '5', '6', '7', '8', '9', '+', '/'
};

static gint16 base64_inverse_table[128];

static void
init_inverse_table (void)
{
  static int i = 0;
  static int j = 0;

  for (; j < 128; j += 1)
    base64_inverse_table[j] = -1;

  for (; i < 64; i += 1)
    base64_inverse_table[base64_table[i]] = i;

  base64_inverse_table['='] = 0;
}

/* BASE64 Source
 */

typedef struct _ByteArraySource ByteArraySource;

struct _ByteArraySource {
  SerialSource source;

  const guint8* in_data;
  guint         in_len;

  const guint8* out_data;
  guint         out_len;

  GByteArray* out_base64;

  gint  source_type;
  guint read_pos;
};

#define SOURCE_TYPE_BASE64 1
#define SOURCE_TYPE_SIMPLE 2

static SerialType   byte_array_source_type         (SerialSource* source, gboolean set_allocation);
static gboolean     byte_array_source_close        (SerialSource* source);
static gboolean     byte_array_source_read         (SerialSource* source, guint8 *ptr, guint32 len);
static void         byte_array_source_free         (SerialSource* source);

static gboolean
serializeio_base64_decode_region_into_internal (const guint8* data, guint len, guint8* out, guint *out_len, gboolean verify)
{
  guint32 pos = 0;
  gboolean found_end = FALSE;
  gint found_end_at = 0;
  guint8 buf[16];

  if ((*out_len) < (len*3/4))
    {
      edsio_generate_void_event (EC_EdsioOutputBufferShort);
      return FALSE;
    }

  (*out_len) = 0;

  while (pos < len)
    {
      gint i, x;
      gint word = 0, junk = 0;

      if (len - pos < 4)
	{
	  edsio_generate_void_event (EC_EdsioInvalidBase64Encoding);
	  return FALSE;
	}

      for (i = 0; i < 4; i += 1)
	{
	  x = data[pos++];

	  if (x > 127 || base64_inverse_table[x] < 0)
	    {
	      edsio_generate_void_event (EC_EdsioInvalidBase64Encoding);
	      return FALSE;
	    }

	  if (x == '=')
	    {
	      if (! found_end)
		found_end_at = i;

	      found_end = TRUE;
	    }
	  else
	    {
	      if (found_end)
		{
		  edsio_generate_void_event (EC_EdsioInvalidBase64Encoding);
		  return FALSE;
		}

	      word |= base64_inverse_table[x] << (6*(3-i));
	    }
	}

      if (found_end)
	{
	  if (found_end_at < 2)
	    {
	      edsio_generate_void_event (EC_EdsioInvalidBase64Encoding);
	      return FALSE;
	    }

	  if (found_end_at == 2)
	    junk = 2;
	  else if (found_end_at == 3)
	    junk = 1;
	}
      else
	junk = 0;

      out[(*out_len)++] = (word >> 16) & 0xff;

      if (junk < 2)
	out[(*out_len)++] = (word >>  8) & 0xff;

      if (junk < 1)
	out[(*out_len)++] = (word >>  0) & 0xff;
    }

  if (verify)
    {
      EdsioMD5Ctx ctx;

      if ((*out_len) < 24)
	{
	  edsio_generate_void_event (EC_EdsioMissingChecksum);
	  return FALSE;
	}

      edsio_md5_init   (& ctx);
      edsio_md5_update (& ctx, out, (*out_len) - 16);
      edsio_md5_final  (buf, & ctx);

      if (memcmp (buf, out + (*out_len) - 16, 16) != 0)
	{
	  edsio_generate_void_event (EC_EdsioInvalidChecksum);
	  return FALSE;
	}

      (*out_len) -= 16;
    }

  return TRUE;
}

gboolean
serializeio_base64_decode_region_into (const guint8* data, guint len, guint8* out, guint *out_len)
{
  return serializeio_base64_decode_region_into_internal (data, len, out, out_len, FALSE);
}

static GByteArray*
serializeio_base64_decode_region_internal (const guint8* data, guint data_len, gboolean verify)
{
  GByteArray* it = g_byte_array_new ();
  guint real_len;

  g_byte_array_set_size (it, data_len*3/4);

  real_len = it->len;

  if (! serializeio_base64_decode_region_into_internal (data, data_len, it->data, &real_len, verify))
    {
      g_byte_array_free (it, TRUE);
      return NULL;
    }

  g_byte_array_set_size (it, real_len);

  return it;
}

GByteArray*
serializeio_base64_decode_region (const guint8* data, guint data_len)
{
  return serializeio_base64_decode_region_internal (data, data_len, FALSE);
}

static SerialSource*
byte_array_source (const guint8* data, guint len, gint type)
{
  ByteArraySource* it = g_new0 (ByteArraySource, 1);

  init_inverse_table ();

  serializeio_source_init (& it->source,
			   byte_array_source_type,
			   byte_array_source_close,
			   byte_array_source_read,
			   byte_array_source_free,
			   NULL,
			   NULL);

  it->in_data = data;
  it->in_len = len;

  it->out_data = NULL;

  it->source_type = type;

  if (type == SOURCE_TYPE_BASE64)
    {
      if ((it->out_base64 = serializeio_base64_decode_region_internal (data, len, TRUE)))
	{
	  it->out_data = it->out_base64->data;
	  it->out_len  = it->out_base64->len;
	}
    }
  else
    {
      it->out_data = data;
      it->out_len = len;
    }

  return &it->source;
}

static SerialType
byte_array_source_type (SerialSource* source, gboolean set_allocation)
{
  ByteArraySource* ssource = (ByteArraySource*) source;
  guint32 x;

  if (! ssource->out_data)
    {
      /* This is a delayed error, see the function above. */
      return ST_Error;
    }

  if (! byte_array_source_read (source, (guint8*) &x, 4))
    return ST_Error;

  x = g_ntohl (x);

  if (set_allocation)
    {
      if (! byte_array_source_read (source, (guint8*) &source->alloc_total, 4))
	return ST_Error;

      source->alloc_total = g_ntohl (source->alloc_total);
    }

  return x;
}

static gboolean
byte_array_source_close (SerialSource* source)
{
  return TRUE;
}

static gboolean
byte_array_source_read (SerialSource* source, guint8 *buf, guint32 len)
{
  ByteArraySource* ssource = (ByteArraySource*) source;

  if (! ssource->out_data)
    {
      /* again, delayed, see above */
      return FALSE;
    }

  if (len + ssource->read_pos > ssource->out_len)
    {
      edsio_generate_source_event (EC_EdsioSourceEof, source);
      return FALSE;
    }

  memcpy (buf, ssource->out_data + ssource->read_pos, len);

  ssource->read_pos += len;

  return TRUE;
}

static void
byte_array_source_free (SerialSource* source)
{
  ByteArraySource* ssource = (ByteArraySource*) source;

  if (ssource->out_base64)
    g_byte_array_free (ssource->out_base64, TRUE);

  g_free (ssource);
}

SerialSource*
serializeio_base64_source (const guint8* data, guint len)
{
  return byte_array_source (data, len, SOURCE_TYPE_BASE64);
}

SerialSource*
serializeio_simple_source (const guint8* data, guint len)
{
  return byte_array_source (data, len, SOURCE_TYPE_SIMPLE);
}

/* BASE64 Sink
 */

typedef struct _ByteArraySink   ByteArraySink;

struct _ByteArraySink {
  SerialSink sink;

  GByteArray* out;
  GByteArray* out_base64;
  gint        sink_type;
  gboolean    free_result;
};

#define SINK_TYPE_SIMPLE 1
#define SINK_TYPE_BASE64 2

static gboolean     byte_array_sink_type           (SerialSink* sink, SerialType type, guint len, gboolean set_allocation);
static gboolean     byte_array_sink_close          (SerialSink* sink);
static gboolean     byte_array_sink_write          (SerialSink* sink, const guint8 *ptr, guint32 len);
static void         byte_array_sink_free           (SerialSink* sink);

GByteArray*
serializeio_base64_encode_region (const guint8* data, guint len)
{
  GByteArray* out = g_byte_array_new ();
  guint real_len;

  g_byte_array_set_size (out, (len+2)*4/3);

  real_len = out->len;

  if (! serializeio_base64_encode_region_into (data, len, out->data, &real_len))
    {
      g_byte_array_free (out, TRUE);
      return NULL;
    }

  g_byte_array_set_size (out, real_len);

  return out;
}

gboolean
serializeio_base64_encode_region_into (const guint8* data, guint len, guint8* out, guint *out_len)
{
  gint i;
  guint32 word = 0, count = 0;

  if ((*out_len) < (len + 2) * 4/3)
    {
      edsio_generate_void_event (EC_EdsioInvalidBase64Encoding);
      return FALSE;
    }

  *out_len = 0;

  for (i = 0; i < len; i += 1)
    {
      word |= data[i] << (8*(2-(count++)));

      if (count == 3)
	{
	  out[(*out_len)++] = base64_table[(word>>18) & 0x3f];
	  out[(*out_len)++] = base64_table[(word>>12) & 0x3f];
	  out[(*out_len)++] = base64_table[(word>> 6) & 0x3f];
	  out[(*out_len)++] = base64_table[(word    ) & 0x3f];

	  count = 0;
	  word = 0;
	}
    }

  if (count > 0)
    {
      out[(*out_len)++] = base64_table[(word>>18) & 0x3f];
      out[(*out_len)++] = base64_table[(word>>12) & 0x3f];
      out[(*out_len)++] = (count > 1) ? base64_table[(word>>6) & 0x3f] : '=';
      out[(*out_len)++] = '=';
    }

  return TRUE;
}

static SerialSink*
byte_array_sink (gint type)
{
  ByteArraySink* it = g_new0 (ByteArraySink, 1);

  init_inverse_table ();

  it->out = g_byte_array_new ();

  serializeio_sink_init (&it->sink,
			 byte_array_sink_type,
			 byte_array_sink_close,
			 byte_array_sink_write,
			 byte_array_sink_free,
			 NULL);

  it->sink_type = type;
  it->free_result = TRUE;

  return &it->sink;
}

static GByteArray*
byte_array_sink_array (SerialSink* sink, gboolean free_result)
{
  ByteArraySink* ssink = (ByteArraySink*) sink;

  ssink->free_result = free_result;

  if (ssink->sink_type == SINK_TYPE_BASE64)
    {
      if (! ssink->out_base64)
	ssink->out_base64 = serializeio_base64_encode_region (ssink->out->data, ssink->out->len);

      g_byte_array_free (ssink->out, TRUE);
      ssink->out = NULL;

      return ssink->out_base64;
    }
  else
    return ssink->out;
}

static gboolean
byte_array_sink_type (SerialSink* sink, SerialType type, guint len, gboolean set_allocation)
{
  guint32 hlen = g_htonl (len), htype = g_htonl (type);

  if (! byte_array_sink_write (sink, (guint8*) & htype, 4))
    return FALSE;

  if (set_allocation && ! byte_array_sink_write (sink, (guint8*) & hlen, 4))
    return FALSE;

  return TRUE;
}

static gboolean
byte_array_sink_close (SerialSink* sink)
{
  ByteArraySink* ssink = (ByteArraySink*) sink;

  EdsioMD5Ctx ctx;
  guint8 buf[16];

  edsio_md5_init   (& ctx);
  edsio_md5_update (& ctx, ssink->out->data, ssink->out->len);
  edsio_md5_final  (buf, & ctx);

  return byte_array_sink_write (sink, buf, 16);
}

static gboolean
byte_array_sink_write (SerialSink* sink, const guint8 *ptr, guint32 len)
{
  ByteArraySink* ssink = (ByteArraySink*) sink;

  g_byte_array_append (ssink->out, ptr, len);

  return TRUE;
}

static void
byte_array_sink_free (SerialSink* sink)
{
  ByteArraySink* ssink = (ByteArraySink*) sink;

  if (ssink->out && ssink->free_result)
    g_byte_array_free (ssink->out, TRUE);

  if (ssink->out_base64 && ssink->free_result)
    g_byte_array_free (ssink->out_base64, TRUE);

  g_free (sink);
}

SerialSink*
serializeio_simple_sink (void)
{
  return byte_array_sink (SINK_TYPE_SIMPLE);
}

GByteArray*
serializeio_simple_sink_array (SerialSink* sink, gboolean free_result)
{
  return byte_array_sink_array (sink, free_result);
}

SerialSink*
serializeio_base64_sink (void)
{
  return byte_array_sink (SINK_TYPE_BASE64);
}

GByteArray*
serializeio_base64_sink_array (SerialSink* sink, gboolean free_result)
{
  return byte_array_sink_array (sink, free_result);
}

/* Default Sink methods
 */

static gboolean
sink_next_uint16 (SerialSink* sink, guint16 num)
{
  num = g_htons (num);

  return sink->sink_write (sink, (guint8*) &num, sizeof (num));
}

static gboolean
sink_next_uint32 (SerialSink* sink, guint32 num)
{
  num = g_htonl (num);

  return sink->sink_write (sink, (guint8*) &num, sizeof (num));
}

static gboolean
sink_next_uint (SerialSink* sink, guint32 num)
{
  /* This is mostly because I dislike endian, and less to save space
   * on small ints.  However, the uint32 and uint16 functions are used
   * when the number is expected to be large, in which case this
   * format can expand the number. */

  guint8 sink_buf[16];  /* this is enough room for a 12-byte int */
  guint  sink_count = 0;

  do
    {
      guint left = num & 0x7f;
      guint outnum;

      num >>= 7;

      outnum = left | (num ? 0x80 : 0);

      sink_buf[sink_count++] = outnum;
    }
  while (num);

  return sink->sink_write (sink, sink_buf, sink_count);
}

static gboolean
sink_next_uint8 (SerialSink* sink, guint8 val)
{
  return sink->sink_write (sink, &val, 1);
}

static gboolean
sink_next_bool (SerialSink* sink, gboolean val)
{
  guint8 sink_buf[1];
  sink_buf[0] = val;
  return sink->sink_write (sink, sink_buf, 1);
}

static gboolean
sink_next_string (SerialSink* sink, const char   *ptr)
{
  return sink->next_bytes (sink, ptr, strlen (ptr));
}

static gboolean
sink_next_bytes (SerialSink* sink, const guint8   *ptr, guint32 len)
{
  return sink->next_uint (sink, len) &&
         sink->sink_write (sink, ptr, len);
}

static gboolean
sink_next_bytes_known (SerialSink* sink, const guint8   *ptr, guint32 len)
{
  return sink->sink_write (sink, ptr, len);
}

void
serializeio_sink_init (SerialSink* it,
		       gboolean (* sink_type) (SerialSink* sink,
					       SerialType type,
					       guint mem_size,
					       gboolean set_allocation),
		       gboolean (* sink_close) (SerialSink* sink),
		       gboolean (* sink_write) (SerialSink* sink,
						const guint8 *ptr,
						guint32 len),
		       void     (* sink_free) (SerialSink* sink),
		       gboolean (* sink_quantum) (SerialSink* sink))
{
  it->next_bytes_known = sink_next_bytes_known;
  it->next_bytes = sink_next_bytes;
  it->next_uint = sink_next_uint;
  it->next_uint32 = sink_next_uint32;
  it->next_uint16 = sink_next_uint16;
  it->next_uint8 = sink_next_uint8;
  it->next_bool = sink_next_bool;
  it->next_string = sink_next_string;

  it->sink_type = sink_type;
  it->sink_close = sink_close;
  it->sink_write = sink_write;
  it->sink_free = sink_free;
  it->sink_quantum = sink_quantum;
}

/* Default Source methods
 */

static gboolean
source_next_uint32 (SerialSource* source, guint32 *ptr)
{
  guint32 x;

  if (! source->source_read (source, (guint8*) &x, sizeof (x)))
    return FALSE;

  (*ptr) = g_ntohl (x);

  return TRUE;
}

static gboolean
source_next_uint16 (SerialSource* source, guint16 *ptr)
{
  guint16 x;

  if (! source->source_read (source, (guint8*) &x, sizeof (x)))
    return FALSE;

  (*ptr) = g_ntohs (x);

  return TRUE;
}

static gboolean
source_next_uint (SerialSource* source, guint32 *ptr)
{
  /* This is mostly because I dislike endian, and less to save space
   * on small ints */
  guint8 c;
  guint8 arr[16];
  gint i = 0;
  gint donebit = 1;
  gint bits;

  while (source->next_uint8 (source, &c))
    {
      donebit = c & 0x80;
      bits = c & 0x7f;

      arr[i++] = bits;

      if (!donebit)
	break;
    }

  if (donebit)
    return FALSE;

  *ptr = 0;

  for (i -= 1; i >= 0; i -= 1)
    {
      *ptr <<= 7;
      *ptr |= arr[i];
    }

  return TRUE;
}

static gboolean
source_next_uint8 (SerialSource* source, guint8 *ptr)
{
  return source->source_read (source, ptr, 1);
}

static gboolean
source_next_bool (SerialSource* source, gboolean *ptr)
{
  guint8 sink_buf[1];

  if (! source->source_read (source, sink_buf, 1))
    return FALSE;

  if (sink_buf[0])
    *ptr = TRUE;
  else
    *ptr = FALSE;

  return TRUE;
}

static gboolean
source_next_string (SerialSource* source, const char **ptr)
{
  guint32 len;
  guint8* buf;

  if (! source->next_uint (source, &len))
    return FALSE;

  if (! (buf = serializeio_source_alloc (source, len+1)))
    return FALSE;

  buf[len] = 0;

  (*ptr) = buf;

  return source->source_read (source, buf, len);
}

static gboolean
source_next_bytes (SerialSource* source, const guint8 **ptr, guint32 *len_ptr)
{
  guint32 len;
  guint8* buf;

  if (! source->next_uint (source, &len))
    return FALSE;

  if (! (buf = serializeio_source_alloc (source, len)))
    return FALSE;

  (*len_ptr) = len;
  (*ptr) = buf;

  return source->source_read (source, buf, len);
}

static gboolean
source_next_bytes_known (SerialSource* source, guint8 *ptr, guint32 len)
{
  return source->source_read (source, ptr, len);
}

void*
serializeio_source_alloc (SerialSource* source, guint32 len)
{
  void* ret;

  if (! source->alloc_buf)
    {
      if (source->salloc_func)
	source->alloc_buf_orig = source->salloc_func (source, source->alloc_total + 8);
      else
	source->alloc_buf_orig = g_malloc0 (source->alloc_total + 8);

      source->alloc_buf = source->alloc_buf_orig;

      ALIGN_8 ((long)source->alloc_buf);
    }

  if (len+source->alloc_pos > source->alloc_total)
    {
      edsio_generate_source_event (EC_EdsioIncorrectAllocation, source);
      return NULL;
    }

  ret = ((guint8*)source->alloc_buf) + source->alloc_pos;

  source->alloc_pos += len;

  ALIGN_8 (source->alloc_pos);

  g_assert (((long)ret) % 8 == 0);
  g_assert (source->alloc_pos % 8 == 0);

  return ret;
}

gboolean
serializeio_source_object_received (SerialSource* source)
{
  source->alloc_pos = 0;
  source->alloc_total = 0;
  source->alloc_buf_orig = NULL;
  source->alloc_buf = NULL;

  return TRUE;
}

void
serializeio_source_reset_allocation (SerialSource* source)
{
  source->alloc_pos = 0;
  source->alloc_total = 0;
  source->alloc_buf = NULL;

  if (source->alloc_buf_orig)
    {
      if (source->sfree_func)
	source->sfree_func (source, source->alloc_buf_orig);
      else
	g_free (source->alloc_buf_orig);
    }
}

void
serializeio_source_init (SerialSource* it,
			 SerialType (* source_type) (SerialSource* source,
						     gboolean set_allocation),
			 gboolean   (* source_close) (SerialSource* source),
			 gboolean   (* source_read) (SerialSource* source,
						     guint8 *ptr,
						     guint32 len),
			 void       (* source_free) (SerialSource* source),
			 void*      (* salloc_func) (SerialSource* source,
						     guint32       len),
			 void       (* sfree_func) (SerialSource* source,
						    void*         ptr))
{
  it->next_bytes_known = source_next_bytes_known;
  it->next_bytes = source_next_bytes;
  it->next_uint = source_next_uint;
  it->next_uint32 = source_next_uint32;
  it->next_uint16 = source_next_uint16;
  it->next_uint8 = source_next_uint8;
  it->next_bool = source_next_bool;
  it->next_string = source_next_string;

  it->source_type = source_type;
  it->source_close = source_close;
  it->source_read = source_read;
  it->source_free = source_free;
  it->salloc_func = salloc_func;
  it->sfree_func = sfree_func;
}

/* Event delivery
 */

static GHashTable* all_event_defs = NULL;

void
eventdelivery_initialize_event_def (gint        code,
				    gint        level,
				    gint        flags,
				    const char* name,
				    const char* oneline,
				    const char * (* field_to_string) (GenericEvent* ev, gint field))
{
  GenericEventDef* def = g_new0 (GenericEventDef, 1);

  if (! all_event_defs)
    all_event_defs = g_hash_table_new (g_int_hash, g_int_equal);

  def->code = code;
  def->level = level;
  def->flags = flags;
  def->name = name;
  def->oneline = oneline;
  def->field_to_string = field_to_string;

  g_hash_table_insert (all_event_defs, & def->code, def);
}

void
eventdelivery_event_deliver (GenericEvent* e)
{
  g_assert (e);

  edsio_edsio_init ();

  if (all_event_defs)
    {
      GenericEventDef* def = g_hash_table_lookup (all_event_defs, & e->code);

      if (def)
	{
	  static GString* out = NULL;
	  const char* oneline = def->oneline;
	  char c;

	  /*if (def->level < EL_Error)
	    return;*/

	  if (! out)
	    out = g_string_new (NULL);

	  g_string_sprintf (out, "%s:%d: ", e->srcfile, e->srcline);

	  while ((c = *oneline++))
	    {
	      switch (c)
		{
		case '$':
		  {
		    const char* field;
		    char *end;
		    int f;

		    if ((*oneline++) != '{')
		      goto badevent;

		    f = strtol (oneline, &end, 10);

		    if (f < 0 || !end || end[0] != '}')
		      goto badevent;

		    oneline = end+1;

		    field = def->field_to_string (e, f);

		    if (field)
		      {
			g_string_append (out, field);

			g_free ((void*) field);
		      }
		    else
		      goto badevent;
		  }
		  break;
		default:
		  g_string_append_c (out, c);
		}
	    }

	  fprintf (stderr, "%s\n", out->str);

	  return;
	}
    }

  g_warning ("%s:%d: Unrecognized event delivered (code=%d)\n", e->srcfile, e->srcline, e->code);

  return;

 badevent:

  g_warning ("%s:%d: An malformed error could not print here (code=%d)\n", e->srcfile, e->srcline, e->code);

  return;
}

const char*
eventdelivery_int_to_string (int x)
{
  return g_strdup_printf ("%d", x);
}

const char*
eventdelivery_string_to_string  (const char* x)
{
  return g_strdup (x);
}

const char*
eventdelivery_source_to_string  (SerialSource* x)
{
  return g_strdup ("@@@SerialSource");
}

const char*
eventdelivery_sink_to_string  (SerialSink* x)
{
  return g_strdup ("@@@SerialSink");
}

/* Misc crap.
 */

gboolean
strtosl_checked (const char* str, long* l, const char* errmsg)
{
  char* end;

  (*l) = strtol (str, &end, 10);

  if (!end || end[0])
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioInvalidIntegerString, errmsg, str);

      (*l) = 0;
      return FALSE;
    }

  return TRUE;
}

gboolean
strtosi_checked (const char* str, int* i,  const char* errmsg)
{
  long l;

  if (! strtosl_checked (str, &l, errmsg))
    {
      (*i) = 0;
      return FALSE;
    }

  if (l > G_MAXINT || l < G_MININT)
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioIntegerOutOfRange, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  (*i) = l;

  return TRUE;
}

gboolean
strtoul_checked (const char* str, unsigned long* l, const char* errmsg)
{
  long sl;

  if (! strtosl_checked (str, &sl, errmsg))
    {
      (*l) = 0;
      return FALSE;
    }

  if (sl < 0)
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioInvalidIntegerSign, errmsg, str);

      (*l) = 0;
      return FALSE;
    }

  (*l) = (unsigned long) sl;
  return TRUE;
}

gboolean
strtoui_checked (const char* str, unsigned int* i, const char* errmsg)
{
  int si;

  if (! strtosi_checked (str, &si, errmsg))
    {
      (*i) = 0;
      return FALSE;
    }

  if (si < 0)
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioInvalidIntegerSign, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  (*i) = (unsigned int) si;
  return TRUE;
}

gint
edsio_md5_equal (gconstpointer   v,
		 gconstpointer   v2)
{
  return memcmp (v, v2, 16) == 0;
}

guint
edsio_md5_hash  (gconstpointer   v)
{
  guint8* md5 = (guint8*) v;
  guint x = 0;
  gint i, j;

  for (i = 0, j = 0; i < 16; i += 1, j += 1, j %= sizeof (guint))
    x ^= md5[i] << (8*j);

  return x;
}

void
edsio_md5_to_string (const guint8* md5, char buf[33])
{
  gint i;

  for (i = 0; i < 16; i += 1)
    sprintf (buf + 2*i, "%02x", md5[i]);
}
