/* -*-Mode: C;-*-
 * $Id: edsio.h 1.5 Tue, 16 Mar 1999 00:13:39 -0800 jmacd $
 *
 * Copyright (C) 1998, Josh MacDonald.
 * All Rights Reserved.
 *
 * Author: Josh MacDonald <jmacd@CS.Berkeley.EDU>
 */

#ifndef _EDSIO_H_
#define _EDSIO_H_

#ifdef __cplusplus
extern "C" {
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <glib.h>

typedef struct _SerialSource    SerialSource;
typedef struct _SerialSink      SerialSink;
typedef gint32                  SerialType;
typedef struct _HandleFuncTable HandleFuncTable;

#include "edsio_edsio.h"

#if !defined(HAVE_FILE_HANDLE_TYPEDEF)
typedef struct _FileHandle    FileHandle;
#define HAVE_FILE_HANDLE_TYPEDEF
#endif

#define ST_Error         -1
#define ST_IncorrectType -2
#define ST_NotFound      -3

#define SER_LIBRARY_OFFSET_BITS 8
#define SER_LIBRARY_OFFSET_MASK 0xff

#define ALIGN_8(v) if (((v) % 8) != 0) { (v) += 8; (v) &= ~7; }

/* This serves as a virtual table for I/O to the FileHandle */

struct _HandleFuncTable
{
  gssize            (* table_handle_length)       (FileHandle *fh);
  gssize            (* table_handle_pages)        (FileHandle *fh);
  gssize            (* table_handle_pagesize)     (FileHandle *fh);
  gssize            (* table_handle_map_page)     (FileHandle *fh, guint pgno, const guint8** mem);
  gboolean          (* table_handle_unmap_page)   (FileHandle *fh, guint pgno, const guint8** mem);
  const guint8*     (* table_handle_checksum_md5) (FileHandle *fh);
  gboolean          (* table_handle_close)        (FileHandle *fh, gint flags);
  gboolean          (* table_handle_write)        (FileHandle *fh, const guint8 *buf, gsize nbyte);
  gboolean          (* table_handle_copy)         (FileHandle *from, FileHandle *to, guint off, guint len);
  gboolean          (* table_handle_getui)        (FileHandle *fh, guint32* i);
  gboolean          (* table_handle_putui)        (FileHandle *fh, guint32 i);
  gssize            (* table_handle_read)         (FileHandle *fh, guint8 *buf, gsize nbyte);
};

struct _SerialSource {

  /* Internal variables: don't touch. */
  guint32  alloc_total;
  guint32  alloc_pos;
  void    *alloc_buf;
  void    *alloc_buf_orig;

  /* These are setup by init.
   */
  SerialType (* source_type)           (SerialSource* source, gboolean set_allocation);
  gboolean   (* source_close)          (SerialSource* source);
  gboolean   (* source_read)           (SerialSource* source, guint8 *ptr, guint32 len);
  void       (* source_free)           (SerialSource* source);

  /* These may be NULL
   */
  void*  (* salloc_func)           (SerialSource* source,
				    guint32       len);
  void   (* sfree_func)            (SerialSource* source,
				    void*         ptr);

  /* Public functions, defaulted, but may be over-ridden
   * before calls to unserialize.
   */
  gboolean   (* next_bytes_known)    (SerialSource* source, guint8        *ptr, guint32  len);
  gboolean   (* next_bytes)          (SerialSource* source, const guint8 **ptr, guint32 *len);
  gboolean   (* next_uint)           (SerialSource* source, guint32       *ptr);
  gboolean   (* next_uint32)         (SerialSource* source, guint32       *ptr);
  gboolean   (* next_uint16)         (SerialSource* source, guint16       *ptr);
  gboolean   (* next_uint8)          (SerialSource* source, guint8        *ptr);
  gboolean   (* next_bool)           (SerialSource* source, gboolean      *ptr);
  gboolean   (* next_string)         (SerialSource* source, const char   **ptr);
};

struct _SerialSink {

  /* These are setup by init.
   */
  gboolean     (* sink_type)          (SerialSink* sink, SerialType type, guint mem_size, gboolean set_allocation);
  gboolean     (* sink_close)         (SerialSink* sink);
  gboolean     (* sink_write)         (SerialSink* sink, const guint8 *ptr, guint32 len);
  void         (* sink_free)          (SerialSink* sink);

  /* This may be null, called after each object is serialized. */
  gboolean     (* sink_quantum)       (SerialSink* sink);

  /* Public functions, defaulted, but may be over-ridden
   * before calls to serialize.
   */
  gboolean   (* next_bytes_known)   (SerialSink* sink, const guint8 *ptr, guint32 len);
  gboolean   (* next_bytes)         (SerialSink* sink, const guint8 *ptr, guint32 len);
  gboolean   (* next_uint)          (SerialSink* sink, guint32       ptr);
  gboolean   (* next_uint32)        (SerialSink* sink, guint32       ptr);
  gboolean   (* next_uint16)        (SerialSink* sink, guint16       ptr);
  gboolean   (* next_uint8)         (SerialSink* sink, guint8        ptr);
  gboolean   (* next_bool)          (SerialSink* sink, gboolean      ptr);
  gboolean   (* next_string)        (SerialSink* sink, const char   *ptr);
};

void           serializeio_initialize_type                (const char* name,
							   guint32     val,
							   gboolean  (*unserialize_func) (),
							   gboolean  (*serialize_func) (),
							   guint     (*count_func) (),
							   void      (*print_func) ());

const char*    serializeio_generic_type_to_string         (SerialType type);
void           serializeio_generic_print                  (SerialType type, void* object, guint indent_spaces);

gboolean       serializeio_serialize_generic              (SerialSink    *sink,
							   SerialType     object_type,
							   void          *object);

gboolean       serializeio_serialize_generic_internal     (SerialSink    *sink,
							   SerialType     object_type,
							   void          *object,
							   gboolean       set_allocation);

guint          serializeio_generic_count                  (SerialType     object_type,
							   void          *object);

gboolean       serializeio_unserialize_generic            (SerialSource  *source,
							   SerialType    *object_type,
							   void         **object);

gboolean       serializeio_unserialize_generic_internal   (SerialSource  *source,
							   SerialType    *object_type,
							   void         **object,
							   gboolean       set_allocation);

gboolean       serializeio_unserialize_generic_acceptable (SerialSource*  source,
							   guint32        acceptable,
							   SerialType    *object_type,
							   void         **object);

void           serializeio_sink_init                      (SerialSink* sink,
							   gboolean (* sink_type) (SerialSink* sink,
										   SerialType type,
										   guint mem_size,
										   gboolean set_allocation),
							   gboolean (* sink_close) (SerialSink* sink),
							   gboolean (* sink_write) (SerialSink* sink,
										    const guint8 *ptr,
										    guint32 len),
							   void     (* sink_free) (SerialSink* sink),
							   gboolean (* sink_quantum) (SerialSink* sink));

void           serializeio_source_init                    (SerialSource* source,
							   SerialType (* source_type) (SerialSource* source,
										       gboolean set_allocation),
							   gboolean   (* source_close) (SerialSource* source),
							   gboolean   (* source_read) (SerialSource* source,
										       guint8 *ptr,
										       guint32 len),
							   void       (* source_free) (SerialSource* source),
							   void*      (* salloc_func) (SerialSource* source,
										       guint32       len),
							   void       (* sfree_func) (SerialSource* source,
										      void*         ptr));

/* These two functions are internal, don't use. */
gboolean       serializeio_source_object_received         (SerialSource* source);
void*          serializeio_source_alloc                   (SerialSource* source,
							   guint32       len);

SerialSource*  serializeio_handle_source                  (FileHandle *fh, const HandleFuncTable* table);
SerialSink*    serializeio_handle_sink                    (FileHandle *fh, const HandleFuncTable* table);

/* Note: these embed a checksum. */
SerialSource*  serializeio_base64_source                  (const guint8* data, guint len);
SerialSink*    serializeio_base64_sink                    (void);
GByteArray*    serializeio_base64_sink_array              (SerialSink* sink, gboolean free_result);
gboolean       serializeio_base64_sink_into               (SerialSink* sink, guint8* buf, guint len);

GByteArray*    serializeio_base64_encode_region           (const guint8* data, guint data_len);
GByteArray*    serializeio_base64_decode_region           (const guint8* data, guint data_len);
gboolean       serializeio_base64_encode_region_into      (const guint8* data, guint data_len, guint8* out, guint *out_len);
gboolean       serializeio_base64_decode_region_into      (const guint8* data, guint data_len, guint8* out, guint *out_len);

SerialSource*  serializeio_simple_source                  (const guint8* data, guint len);
SerialSink*    serializeio_simple_sink                    (void);
GByteArray*    serializeio_simple_sink_array              (SerialSink* sink, gboolean free_result);

/* Event delivery
 */

#define ERROR_LIBRARY_OFFSET_BITS 8
#define ERROR_LIBRARY_OFFSET_MASK 0xff

enum _EventLevel
{
  EL_Warning       = 1<<0,
  EL_Error         = 1<<1,
  EL_InternalError = 1<<2,
  EL_FatalError    = 1<<3
};

typedef enum _EventLevel EventLevel;

enum _EventFlags
{
  EF_None    = 1<<0,
  EF_OpenSSL = 1<<1
};

typedef enum _EventFlags EventFlags;

const char* eventdelivery_int_to_string     (int x);
const char* eventdelivery_string_to_string  (const char* x);
const char* eventdelivery_source_to_string  (SerialSource* x);
const char* eventdelivery_sink_to_string    (SerialSink* x);

/* Event delivery privates
 */

typedef struct _GenericEventDef GenericEventDef;
typedef struct _GenericEvent    GenericEvent;

struct _GenericEvent
{
  gint        code;
  const char* srcfile;
  guint       srcline;
};

void eventdelivery_event_deliver        (GenericEvent* e);

void eventdelivery_initialize_event_def (gint        code,
					 gint        level,
					 gint        flags,
					 const char* name,
					 const char* oneline,
					 const char * (* field_to_string) (GenericEvent* ev, gint field));

const char* eventdelivery_ssl_errors_to_string (void);

struct _GenericEventDef
{
  gint        code;
  gint        level;
  gint        flags;
  const char *name;
  const char *oneline;

  const char * (* field_to_string) (GenericEvent* ev, gint field);
};

/* MD5.H - header file for MD5C.C */

/* Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
   rights reserved.

   License to copy and use this software is granted provided that it
   is identified as the "RSA Data Security, Inc. MD5 Message-Digest
   Algorithm" in all material mentioning or referencing this software
   or this function.

   License is also granted to make and use derivative works provided
   that such works are identified as "derived from the RSA Data
   Security, Inc. MD5 Message-Digest Algorithm" in all material
   mentioning or referencing the derived work.

   RSA Data Security, Inc. makes no representations concerning either
   the merchantability of this software or the suitability of this
   software for any particular purpose. It is provided "as is"
   without express or implied warranty of any kind.

   These notices must be retained in any copies of any part of this
   documentation and/or software.
   */

/* MD5 context. */
typedef struct {
    guint32 state[4]; /* state (ABCD) */
    guint32 count[2]; /* number of bits, modulo 2^64 (lsb first) */
    guint8 buffer[64]; /* input buffer */
} EdsioMD5Ctx;

void edsio_md5_init   (EdsioMD5Ctx *);
void edsio_md5_update (EdsioMD5Ctx *, const guint8 *, guint);
void edsio_md5_final  (guint8*, EdsioMD5Ctx *);

gint  edsio_md5_equal (gconstpointer   v,
		       gconstpointer   v2);
guint edsio_md5_hash  (gconstpointer   v);

void  edsio_md5_to_string (const guint8* md5, char buf[33]);


/* NIST Secure Hash Algorithm */
/* heavily modified from Peter C. Gutmann's implementation */
/* then taken from from Uwe Hollerbach, */
/* and then modified a little by Josh MacDonald. */

/* This code is in the public domain */

typedef struct {
    guint32 digest[5];             /* message digest */
    guint32 count_lo, count_hi;    /* 64-bit bit count */
    guint32 data[16];              /* SHA data buffer */
    int local;                  /* unprocessed amount in data */
} EdsioSHACtx;

void edsio_sha_init   (EdsioSHACtx *);
void edsio_sha_update (EdsioSHACtx *, const guint8 *, guint);
void edsio_sha_final  (guint8 *, EdsioSHACtx *);

gint  edsio_sha_equal (gconstpointer   v,
		       gconstpointer   v2);
guint edsio_sha_hash  (gconstpointer   v);

/* Misc stuff.
 */

/* These raise an error if errmsg is non-null.  The errmsg should
 * be something like "Invalid port number".  See edsio.ser for the
 * format.
 */
gboolean strtosl_checked (const char* str, long* l, const char* errmsg);
gboolean strtosi_checked (const char* str, int* i,  const char* errmsg);

gboolean strtoul_checked (const char* str, unsigned long* l, const char* errmsg);
gboolean strtoui_checked (const char* str, unsigned int* i, const char* errmsg);

#ifdef __cplusplus
}
#endif

#endif /* _EDSIO_H_ */
