/* -*- Mode: C;-*-
 *
 * This file is part of XDelta - A binary delta generator.
 *
 * Copyright (C) 1997, 1998  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Josh MacDonald <jmacd@CS.Berkeley.EDU>
 *
 * $Id: xdelta.h 1.4.1.8.1.46 Wed, 10 Mar 1999 20:17:43 -0800 jmacd $
 */

#ifndef _XDELTA_H_
#define _XDELTA_H_

#include "xd_edsio.h"

typedef SerialRsyncIndex          XdeltaRsync;
typedef SerialRsyncIndexElt       XdeltaRsyncElt;
typedef SerialXdeltaChecksum      XdeltaChecksum;
typedef SerialXdeltaIndex         XdeltaIndex;
typedef SerialXdeltaSourceInfo    XdeltaSourceInfo;
typedef SerialXdeltaControl       XdeltaControl;
typedef SerialXdeltaInstruction   XdeltaInstruction;

typedef struct _XdeltaMD5Table    XdeltaMD5Table;

typedef struct _XdeltaGenerator   XdeltaGenerator;
typedef struct _XdeltaSource      XdeltaSource;

typedef FileHandle XdeltaStream;
typedef FileHandle XdeltaOutStream;

/* Note: FileHandle is an opaque type, you must define it
 * to use this library.  See how its done in xdmain.c.
 */

/* $Format: "#define XDELTA_VERSION \"$ReleaseVersion$\"" $ */
#define XDELTA_VERSION "1.0.6"

/* $Format: "#define XDELTA_MAJOR_VERSION $ReleaseMajorVersion$" $ */
#define XDELTA_MAJOR_VERSION 1

/* $Format: "#define XDELTA_MINOR_VERSION $ReleaseMinorVersion$" $ */
#define XDELTA_MINOR_VERSION 0

/* $Format: "#define XDELTA_MICRO_VERSION $ReleaseMicroVersion$" $ */
#define XDELTA_MICRO_VERSION 6

extern const guint xdelta_major_version;
extern const guint xdelta_minor_version;
extern const guint xdelta_micro_version;

#define QUERY_SIZE          4  /* segments are of length 1<<QUERY_SIZE */
#define QUERY_SIZE_POW      (1<<QUERY_SIZE)
#define QUERY_SIZE_MASK     (QUERY_SIZE_POW-1)
#define XDELTA_MAX_FILE_LEN (1<<(32-QUERY_SIZE-2))

struct _XdeltaMD5Table
{
  void              (* table_md5_init)            (gpointer);
  void              (* table_md5_update)          (gpointer, const guint8 *, guint);
  void              (* table_md5_final)           (guint8*, gpointer);
  guint                table_md5_ctx_size;
};

/* An xdelta consists of two pieces of information, the control and
 * data segments.  The control segment consists of instructions,
 * metadata, and redundency check information.  The data segment
 * consists of literal data not found in any of the sources.
 *
 * The library operates on two types of streams, random access and
 * non-seekable.  Briefly, you initialize a XdeltaGenerator with one
 * or more XdeltaSources.  These XdeltaSources contain a random access
 * stream.  If these XdeltaSources are themselves the data segment of
 * another delta, the control segment for that delta should be
 * supplied as well for the creation of normalized deltas.
 *
 * The generator is initialized with an input stream and an output
 * stream, these streams are not seekable.
 *
 * The creation of a XdeltaSource requires a complete pass through the
 * file.  This pass pre-computes an index which is used during delta
 * computation.  This index may be saved and restored to avoid
 * computing it multiple times.
 */

/* Create a new generator.  One of these must be used for all other
 * Xdelta transactions. */

#define xdp_generator_new(t,m) __xdp_generator_new (t,m,XDELTA_VERSION)

XdeltaGenerator* __xdp_generator_new      (const HandleFuncTable *table,
					   const XdeltaMD5Table  *md5,
					   const char            *version);

/* Clear any state assocated with the object. */
void             xdp_generator_reset      (XdeltaGenerator *gen);

/* Create a new source.  If non-null, DELTA_CONTROL_IN indicates
 * that SOURCE_IN is the data segment of another delta--the delta
 * produced will be normalized accordingly.  If non-null, INDEX_IN
 * indicates that the index was previously computed and may be read
 * from the stream.  If non-null, INDEX_OUT is a stream to which the
 * the computed index will be written.  INDEX_OUT is ignored when
 * INDEX_IN is non-null.  Returns the source on success, NULL on
 * failure. */

XdeltaSource*    xdp_source_new           (XdeltaGenerator *gen,
					   XdeltaControl   *delta_control_in,
					   XdeltaStream    *source_in,
					   XdeltaStream    *index_in,
					   XdeltaOutStream *index_out);

/* Simply index the source, do not save it in memory.  Returns true on
 * success, false on failure. */
gboolean         xdp_source_index         (XdeltaGenerator *gen,
					   XdeltaStream    *source_in,
					   XdeltaOutStream *index_out);

/* Add SRC to the generator.  The source will then be used for generating
 * deltas. */
void             xdp_source_add           (XdeltaGenerator *gen,
					   XdeltaSource    *src);

/* Actually generate a delta against the accumulated sources in GEN.
 * Returns the delta's controller or NULL on failure. */
XdeltaControl*   xdp_generate_delta       (XdeltaGenerator *gen,
					   XdeltaStream    *in,
					   gboolean         normalize,
					   XdeltaOutStream *control_out,
					   XdeltaOutStream *data_out);

/* Reads a control object from a stream. */
XdeltaControl*   xdp_control_read         (XdeltaGenerator *gen,
					   XdeltaStream    *cont_in);

/* Writes a control object to a stream. */
gboolean         xdp_control_write        (XdeltaGenerator *gen,
					   XdeltaControl   *cont,
					   XdeltaOutStream *cont_out);

/* Returns the number of files this delta depends on. */
gint             xdp_control_depends_on_count      (XdeltaControl   *cont);

/* Returns the MD5 checksum of the i-th file this delta depends on. */
const guint8*    xdp_control_depends_on            (XdeltaControl   *cont, guint i);

/* Returns the MD5 checksum of the i-th normalized file this delta depends on. */
const guint8*    xdp_control_depends_on_normalized (XdeltaControl   *cont, guint i);

/* Returns the length of file this delta constitutes. */
gint             xdp_control_length                (XdeltaControl   *cont);

/* Returns the MD5 checksum of file this delta constitutes. */
const guint8*    xdp_control_md5                   (XdeltaControl   *cont);

void             xdp_source_free          (XdeltaSource    *src);
void             xdp_generator_free       (XdeltaGenerator *gen);
void             xdp_control_free         (XdeltaControl   *con);

/* To apply these patches... */

/* This callback is called when the apply step requires a data source.
 *  USER_DATA is the argument provided to xdp_apply_delta.
 *  REQUIRED is true iff the range is for a normalize instruction
 *    or a regular instruction when a normalized instruction is not
 *    available (in other words, failure is guaranteed if this
 *    function fails while REQUIRED is true).
 *  OFFSET and LENGTH are the range to write to WRITE_TO using the
 *  function WRITE
 */

typedef gboolean (XdpSourceCallback) (XdeltaGenerator  *gen,
				      void             *user_data,
				      const guint8     *expected_md5,
				      gboolean          required,
				      guint             offset,
				      guint             length,
				      XdeltaOutStream  *write_to);

gboolean         xdp_apply_delta          (XdeltaGenerator   *gen,
					   XdeltaControl     *cont,
					   XdeltaStream      *data,
					   XdpSourceCallback *call,
					   void              *user_data,
					   XdeltaOutStream   *res);

/* For use in (especially recursive) callbacks, this function results
 * in the library calling CALL appropriately to write length LEN at
 * offset OFF from the file constructed by the delta CONT (without
 * actually constructing it).  WRITE and USER_DATA are passed to CALL.
 * Conceptually, this is just like xdp_apply_delta for an offset and
 * length.  Using this recursively in CALL allows application of
 * multiple deltas without constructing intermediate files. */
gboolean         xdp_copy_delta_region    (XdeltaGenerator   *gen,
					   XdeltaControl     *cont,
					   XdeltaStream      *data,
					   XdpSourceCallback *call,
					   guint              off,
					   guint              len,
					   void              *user_data,
					   XdeltaOutStream   *write);

/* Forward, backward annotation.  Can it be done well at this level? */

/* @@@ */

/* Rsync. */

XdeltaRsync*     xdp_rsync_index          (XdeltaGenerator   *gen,
					   XdeltaStream      *file,
					   guint              seg_len,
					   XdeltaStream      *cache_in,
					   XdeltaOutStream   *cache_out);

void             xdp_rsync_index_free     (XdeltaRsync       *rsync);

/* Returns an array of the needed chunks. */
GArray*          xdp_rsync_request        (XdeltaGenerator   *gen,
					   XdeltaStream      *file,
					   XdeltaRsync       *rsync);

gboolean         xdp_apply_rsync_reply    (XdeltaGenerator   *gen,
					   XdeltaRsync       *rsync,
					   XdeltaStream      *from,
					   XdeltaStream      *reply,
					   XdeltaStream      *out);

/* For error printing. */

const char* eventdelivery_stream_to_string  (XdeltaStream* x);

#endif
