//+ File:      Queue.cpp

#define STRICT
#include <windows.h>
#include <mq.h>
#pragma warning(disable: 4786)  // OK to ignore symbol > 255
#include <vector>
#include <string>
#include <algorithm>
using namespace std;
#include "Queue.h"

bool operator<(const MQPROPVARIANT& l, const MQPROPVARIANT& r)
{
    return false;
}

bool operator==(const MQPROPVARIANT& l, const MQPROPVARIANT& r)
{
    return false;
}

// Constants
#define MAX_FORMAT (MAX_PATH*2)

// Table of message and queue properties and their corresponding
// data types.  NOTE: The function that searches this table
// assumes it is sorted by ascending propID!  

struct DataTypeEntry
{
    MSGPROPID propID;
    VARTYPE   vt;
    MSGPROPID propLen;
};

bool operator<(const DataTypeEntry& l, const DataTypeEntry& r)
{
    return l.propID < r.propID;
}

static DataTypeEntry DataTypes[] = {
{PROPID_M_BODY,                 VT_UI1|VT_VECTOR, PROPID_M_BASE },
{PROPID_M_BODY_SIZE,            VT_UI4, PROPID_M_BASE },
{PROPID_M_LABEL,                VT_LPWSTR, PROPID_M_LABEL_LEN },
{PROPID_M_LABEL_LEN,            VT_UI4, PROPID_M_BASE }
};

static const size_t TypeSize = (sizeof DataTypes/sizeof *DataTypes);

static DataTypeEntry* FindDataType(MSGPROPID id)
{
    DataTypeEntry target;
    target.propID = id;
    DataTypeEntry* end = DataTypes + TypeSize;
    DataTypeEntry* p = lower_bound(DataTypes, end, target);
    return p == end ? 0 : p;
}

void InitializeVariant(MQPROPVARIANT& val, MSGPROPID propID)
{
    VARTYPE vt= VT_ILLEGAL;

    PropVariantInit(&val);
    DataTypeEntry* p = FindDataType(propID);
    if (p && p->propID == propID)
        vt = p->vt;
    val.vt = vt;
}

MSGPROPID GetLengthProperty(MSGPROPID propID)
{
    MSGPROPID lengthPropID = PROPID_M_BASE;

    DataTypeEntry* p = FindDataType(propID);
    if (p && p->propID == propID)
        lengthPropID = p->propLen;
    return lengthPropID;
}

namespace Util {
    HRESULT PropVariantClear(MQPROPVARIANT* pVnt)
    {
        switch (pVnt->vt)
        {
        case VT_LPWSTR:
            delete [] pVnt->pwszVal;
            break;
        case VT_CLSID:
            break;
            delete pVnt->puuid;
        case VT_UI1|VT_VECTOR:
            delete [] pVnt->caub.pElems;
            break;
        case VT_UI1:
        case VT_UI2:
        case VT_UI4:
            break;
        default:
            return S_OK;
        }
        ZeroMemory(pVnt, sizeof(MQPROPVARIANT));
        return S_OK;
    }

    HRESULT FreePropVariantArray(ULONG cVariant,
        MQPROPVARIANT* rgvar)
    {
        if (rgvar == NULL)
            return S_OK;

        while (cVariant-- > 0)
            PropVariantClear(rgvar++);
        return S_OK;
    }

}
// Construction/Destruction

CQueue::CQueue()
{
    ZeroMemory(&m_overlapped, sizeof m_overlapped);
    ZeroMemory(&m_msgProps, sizeof m_msgProps);
}

CQueue::~CQueue()
{
    try
    {
        Close();
    }
    catch(HRESULT )
    {
        // It's too late to do anything now
    }
}

void CQueue::CloseHandles()
{
    vector<QUEUEHANDLE>::iterator it;
    
    for (it = m_vecHQueue.begin(); it != m_vecHQueue.end(); ++it) {
        QUEUEHANDLE hQueue = *it;
        if (hQueue)
            MQCloseQueue(hQueue);
    }
    m_vecHQueue.clear();
}

void CQueue::Close()
{
    CloseHandles();
    Reset();
}

HRESULT CQueue::Open(LPCWSTR lpwcsPathName, DWORD dwAccess,
                     DWORD dwShareMode)
{
    LPWSTR lpwcsFormatName = 0;
    HRESULT hr = S_OK;
    try
    {
        Close();
        lpwcsFormatName = GetFmtName(lpwcsPathName);
    }
    catch (HRESULT hres)
    {
        delete [] lpwcsFormatName;
        return hres;
    }
    QUEUEHANDLE hQueue;
    hr = MQOpenQueue(lpwcsFormatName, dwAccess, dwShareMode,
        &hQueue);
    delete [] lpwcsFormatName;
    if (FAILED(hr)) return hr;
    m_vecHQueue.push_back(hQueue);
    return hr;
}

void CQueue::Reset()
{
    if (m_overlapped.hEvent) CloseHandle(m_overlapped.hEvent);
    if (m_msgPropIDs.size()) ResetProperties();
}

void CQueue::ResetProperties()
{
    m_msgPropIDs.clear();
    Util::FreePropVariantArray(m_msgVals.size(),
        m_msgVals.begin());
    m_msgVals.clear();
}

void CQueue::SetReadProperty(MSGPROPID propID, size_t cElems)
{
    MQPROPVARIANT val;
    InitializeVariant(val, propID);
    if (val.vt & VT_VECTOR) {
        val.caub.cElems = cElems;
        val.caub.pElems = new unsigned char[cElems];
    } else if ((val.vt & VT_TYPEMASK) == VT_LPWSTR) {
        MSGPROPID lengthProp = GetLengthProperty(propID);
        val.pwszVal = new WCHAR[cElems];
        MQPROPVARIANT vntLength;
        PropVariantInit(&vntLength);
        vntLength.vt = VT_UI4;
        vntLength.ulVal = cElems;
        m_msgPropIDs.push_back(lengthProp);
        m_msgVals.push_back(vntLength);
    }
    m_msgPropIDs.push_back(propID);
    m_msgVals.push_back(val);
}

MQPROPVARIANT* CQueue::GetReadProperty(MSGPROPID propID)
{
    HRESULT hr = 0x80040200;

    vector<MSGPROPID>::iterator it;

    // First, find the ID in the vector of IDs you are maintaining
    it = find(m_msgPropIDs.begin(), m_msgPropIDs.end(), propID);
    if (it == m_msgPropIDs.end())
        throw hr;

    // The variant is at the same index in the vector of values
    int index = it - m_msgPropIDs.begin();
    return &(m_msgVals.at(index));
}

void CQueue::SetWriteProperty(MSGPROPID propID, LPWSTR lpwString)
{
    // Make a copy of the string so the caller doesn't have to keep
    // his copy around 
    size_t alloc = lpwString ? wcslen(lpwString) + 1 : 1;
    LPWSTR copy = new WCHAR[alloc];
    if (lpwString)
        wcscpy(copy, lpwString);
    else
        *copy = 0;

    // Construct a variant and populate it
    MQPROPVARIANT vnt;
    PropVariantInit(&vnt);
    vnt.vt = VT_LPWSTR;
    vnt.pwszVal = copy;

    // Add another property to the vectors
    m_msgPropIDs.push_back(propID);
    m_msgVals.push_back(vnt);
}

void CQueue::SetWriteProperty(MSGPROPID propID, PBYTE blob,
                              size_t nBytes)
{
    // Do not make a copy of the blob because it could be very
    // large.  This is not consistent with the way strings are
    // handled, but it is done like this for efficiency's sake   

    // Construct a variant and populate it
    MQPROPVARIANT vnt;
    PropVariantInit(&vnt);
    vnt.vt = VT_UI1 | VT_VECTOR;
    vnt.caub.cElems = nBytes;
    vnt.caub.pElems = blob;

    // Add another property to the vectors
    m_msgPropIDs.push_back(propID);
    m_msgVals.push_back(vnt);
}

void CQueue::SetWriteProperty(MSGPROPID propID, const CLSID* pCLSID)
{
    // I don't know if it's better to copy the class ID or not. For
    // now, I'm going to copy it. 

    // Construct a variant and populate it
    MQPROPVARIANT vnt;
    PropVariantInit(&vnt);
    vnt.vt = VT_CLSID;
    vnt.puuid = new CLSID;
    memcpy(vnt.puuid, pCLSID, sizeof(CLSID));

    // Add another property to the vectors
    m_msgPropIDs.push_back(propID);
    m_msgVals.push_back(vnt);
}

void CQueue::ReceiveNonBlock()
{
    // Create an event kernel object that has default security
    // attributes (0) is manual reset (TRUE), is not initialized in
    // the signaled state (FALSE), and has no name (0). 

    m_overlapped.hEvent = CreateEvent(0, TRUE, FALSE, 0);

    // Set up the properties to receive
    m_msgProps.cProp = m_msgPropIDs.size();
    m_msgProps.aPropID = m_msgPropIDs.begin();
    m_msgProps.aPropVar = m_msgVals.begin();
    m_msgProps.aStatus = NULL;

    // Get the queue at the head of the vector for receiving
    QUEUEHANDLE hQueue = m_vecHQueue[0];

    // Call the Queue Manager
    HRESULT hr = MQReceiveMessage(
        hQueue,
        INFINITE,
        MQ_ACTION_RECEIVE,
        &m_msgProps,
        &m_overlapped,
        NULL,                           // no callback
        NULL,                           // no cursor
        NULL                            // not part of transaction
        );

    // Check what the Queue Manager had to say about this
    // request.  If there is nothing on the queue at the moment,
    // the return will be INFORMATION_ OPERATION_PENDING.  If
    // there is already something waiting in the queue, the
    // return will be MQ_OK.  Either of these results is
    // satisfactory, and in either case the event object will be
    // signalled, either now or when a message arrives.    

    if (hr != MQ_INFORMATION_OPERATION_PENDING && hr != MQ_OK)
        throw hr;
}

void CQueue::Send()
{
    // Set up the properties to send
    m_msgProps.cProp = m_msgPropIDs.size();
    m_msgProps.aPropID = m_msgPropIDs.begin();
    m_msgProps.aPropVar = m_msgVals.begin();
    m_msgProps.aStatus = NULL;

    vector<QUEUEHANDLE>::iterator it;
    for (it = m_vecHQueue.begin(); it != m_vecHQueue.end(); ++it) {
        QUEUEHANDLE hQueue = *it;
        HRESULT hr = MQSendMessage(hQueue, &m_msgProps, NULL);
        if (FAILED(hr))
            throw hr;
    }
}

LPWSTR CQueue::GetFmtName(LPCWSTR lpwcsPathName)
{
    LPWSTR lpwcsFormatName = new WCHAR[FMT_NAME_FIRST_TRY];
    DWORD dwCount = FMT_NAME_FIRST_TRY;
    LPDWORD lpdwCount = &dwCount;

    HRESULT hr = MQPathNameToFormatName(
        lpwcsPathName,
        lpwcsFormatName,
        lpdwCount );

    // If the buffer was too small, try again
    if (hr == MQ_ERROR_FORMATNAME_BUFFER_TOO_SMALL) {
        delete [] lpwcsFormatName;
        lpwcsFormatName = new WCHAR[dwCount];
        hr = MQPathNameToFormatName(
            lpwcsPathName,
            lpwcsFormatName,
            lpdwCount );
    }
    if (FAILED(hr)) {
        delete [] lpwcsFormatName;
        throw hr;
    }
    return lpwcsFormatName;
}

HRESULT CQueue::Create()
{
    HRESULT hr = S_OK;

    // Set up the properties
    m_msgProps.cProp = m_msgPropIDs.size();
    m_msgProps.aPropID = m_msgPropIDs.begin();
    m_msgProps.aPropVar = m_msgVals.begin();
    m_msgProps.aStatus = NULL;

    // Allocate some space for string buffer
    LPWSTR lpwcsFormatName = new WCHAR[FMT_NAME_FIRST_TRY];
    DWORD dwNumChars = FMT_NAME_FIRST_TRY;

    // Call the Queue Manager
    hr = MQCreateQueue(
        NULL,                        // IN: Default security
        (MQQUEUEPROPS*)&m_msgProps,  // IN/OUT: Queue properties
        lpwcsFormatName,             // OUT: Format Name
        &dwNumChars                  // IN/OUT: Size of format name
        );
    return hr;
}

HRESULT CQueue::Open(const CLSID * type)
{
    HRESULT hr = S_OK;

    MQPROPERTYRESTRICTION   propRes;
    MQRESTRICTION           Restriction;
    QUEUEPROPID             propID;
    MQCOLUMNSET             Column;
    MQPROPVARIANT           vntInstance;
    HANDLE                  hEnum;
    CLSID                   tempCLSID = *type;

    propRes.rel = PREQ;                // relation is EQUAL
    propRes.prop = PROPID_Q_TYPE;
    propRes.prval.vt = VT_CLSID;
    propRes.prval.puuid = &tempCLSID;
    Restriction.cRes = 1;
    Restriction.paPropRes = &propRes;
    propID = PROPID_Q_INSTANCE;
    Column.cCol = 1;
    Column.aCol = & propID;

    CloseHandles();
    hr = MQLocateBegin(NULL, &Restriction, &Column, NULL, &hEnum);
    if (FAILED(hr))
        return hr;

    DWORD cQueue = 1;
    PropVariantInit(&vntInstance);
    DWORD cch;
    WCHAR wcsFormat[MAX_FORMAT];
    QUEUEHANDLE hQueue;
    do {
        cQueue = 1;
        hr = MQLocateNext(hEnum, &cQueue, &vntInstance);
        if (FAILED(hr))
            return hr;
        if (cQueue > 0) {
            cch = MAX_FORMAT;
            hr = MQInstanceToFormatName(vntInstance.puuid,
                wcsFormat, &cch);
            if (FAILED(hr))
                return hr;
            hr = MQOpenQueue(
                wcsFormat,
                MQ_SEND_ACCESS,
                MQ_DENY_NONE,
                &hQueue
                );
            if (FAILED(hr))
                return hr;
            m_vecHQueue.push_back(hQueue);
            MQFreeMemory(vntInstance.puuid);
        }
    } while (cQueue > 0);
    hr = MQLocateEnd(hEnum);
    return hr;
}
