//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos;

import java.util.Vector;
import java.security.SecureRandom;

import com.dstc.security.kerberos.creds.Credential;
import com.dstc.security.kerberos.creds.CredentialInfo;
import com.dstc.security.kerberos.creds.KerberosTicket;
import com.dstc.security.kerberos.crypto.KerberosCipher;
import com.dstc.security.kerberos.crypto.KeyMaterial;
import com.dstc.security.kerberos.crypto.CryptoException;
import com.dstc.security.kerberos.v5.*;
import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.creds.Ticket;
import com.dstc.security.kerberos.v5.crypto.EncryptedData;

import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Explicit;
import com.dstc.security.asn1.Sequence;

public class KerberosCredential extends KerberosMessage
{
  private Vector creds = null;      // Vector of credentials in this message
  private Vector tickets = null;  
  private Vector credsInfo;
  private EncKrbCredPart encKrbCredPart = null; // plaintext of encrypted part
  private EncryptedData encData = null;
  private KerberosCipher cipher = null;
  private KrbCred krbCred = null;
  private int nonce;
  private String sAddr = null;
  private String rAddr = null;

  public KerberosCredential(KeyMaterial cred, SecureRandom rand) 
    throws CryptoException
  {
    this.cipher = new KerberosCipher(cred, rand, null);
    this.cipher.initEncrypt();
  }

  /**
   * @param creds   All the credentials to be forwarded
   */
  public void setCredentials(Vector creds) 
    throws Asn1Exception, BadVersionException
  {
    this.creds = creds;
    this.tickets = new Vector();
    for (int i = 0; i < creds.size(); i++)
      tickets.addElement(
        new Ticket(((Credential)creds.elementAt(i)).getTicket().getEncoded()));

    this.credsInfo = new Vector();
    for (int i = 0; i < creds.size(); i++)
     credsInfo.addElement(
      new KrbCredInfo(((Credential)
       creds.elementAt(i)).getCredentialInfo().getEncoded()));
  }

  /**
   * @param nonce   random nonce, should be generated by intended recipient,
   *                 a 0 means no nonce is used
   */
  public void setNonce(int nonce)
  {
    this.nonce = nonce;
  }

  public void setSenderAddress(String addr)
  {
    this.sAddr = addr;
  }

  public void setRecipientAddress(String addr)
  {
    this.rAddr = addr;
  }

  public void encrypt() throws CryptoException
  {
    encKrbCredPart = new EncKrbCredPart(credsInfo, nonce, sAddr, rAddr);
    encData = cipher.encrypt(encKrbCredPart.encode());

    this.krbCred = new KrbCred(tickets, encData);
  }

  /**
   * Constructs a KerberosCredential from a DER encoding
   *
   * @param encoded   The DER encoding
   */
  public KerberosCredential(byte[] encoded) 
    throws Asn1Exception, BadVersionException, BadMessageTypeException
  {
    this.krbCred = new KrbCred(encoded);
    this.tickets = krbCred.getTickets();
    this.encData = krbCred.getEncrypted();
  }

  public void initDecrypt(KeyMaterial cred) throws CryptoException
  {
    KerberosCipher cipher = new KerberosCipher(cred, null, null);
    cipher.initDecrypt();

    this.cipher = cipher;
  }

  public void decrypt() 
    throws Asn1Exception, CryptoException, KerberosException
  {
    if (this.cipher == null)
      throw new KerberosException("Cipher not yet set");

    // rebuild the Vector of Credential
    encKrbCredPart = new EncKrbCredPart(cipher.decrypt(encData));
    Vector credsInfo = encKrbCredPart.getCredsInfo();
    int size = 
      (credsInfo.size() <= tickets.size()) ? credsInfo.size() : tickets.size();
    creds = new Vector(size);
    for (int i = 0; i < size; i++)
      creds.addElement(new Credential(
        new CredentialInfo(((KrbCredInfo)credsInfo.elementAt(i)).encode()),
        new KerberosTicket(((Ticket)tickets.elementAt(i)).encode())));
  }

  /**
   * Returns the creds forwarded by this message
   */
  public Vector getCredentials() 
  {
    return creds;
  }

  /**
   * Returns the nonce associated with this message, a zero means no nonce
   */
  public int getNonce() 
  {
    return encKrbCredPart.getNonce();
  }

  /**
   * Returns the timestamp of this message, null if no timestamp was used
   */
  public KerberosTime getTimestamp() 
  {
    return encKrbCredPart.getTimestamp();
  }

  /**
   * Returns the microsecond portion of the timestamp
   */
  public int getUsec() 
  {
    return encKrbCredPart.getUsec();
  }

  /**
   * Returns the sender address if any
   */
  public String getSenderAddress() 
  {
    return encKrbCredPart.getSenderAddress().toString();
  }

  /**
   * Returns the receiver address if any
   */
  public String getReceiverAddress() 
  {
    return encKrbCredPart.getReceiverAddress().toString();
  }

  public byte[] getEncoded()
  {
    return krbCred.encode();
  }
}
