//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5;

import java.util.Vector;

import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptedData;

import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Explicit;

/**
 * <p>A class representing the Kerberos ASN.1 structure AP-REP.
 *
 * <pre>
 *   AP-REP ::=  [APPLICATION 15] SEQUENCE {
 *                  pvno[0]             INTEGER,
 *                  msg-type[1]         INTEGER,
 *                  enc-part[2]         EncryptedData
 *                }
 * </pre>
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class APRep extends Explicit
{
  public static final int TAG_KRB_AP_REP = 15;

  private static final int pvno = 5;
  private int msgType = TAG_KRB_AP_REP;
  private EncryptedData encPart = null;

  /**
   * Constructs an APRep from an EncryptedData
   */
  public APRep(EncryptedData encPart)
  {
    super(Asn1.TAG_CLASS_APPLICATION, TAG_KRB_AP_REP);

    Sequence seq = new Sequence();

    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(pvno));
    seq.addComponent(exp);

    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    exp.addComponent(new com.dstc.security.asn1.Integer(msgType));
    seq.addComponent(exp);

    this.encPart = encPart;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 2);
    exp.addComponent(encPart);
    seq.addComponent(exp);

    addComponent(seq);
  }

  /**
   * Constructs an APRep from a DER encoding
   */
  public APRep(byte[] encoded) 
    throws Asn1Exception, BadMessageTypeException, BadVersionException
  {
    doDecode(encoded);
    if (this.getTagNumber() != TAG_KRB_AP_REP)
      throw new BadMessageTypeException("Wrong tag for an APRep");

    Sequence seq = (Sequence)this.getComponent();
    Vector comps = seq.getComponents();

    int i = 0;

    Explicit exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 0)
      throw new Asn1Exception("Bad Tag Number");
    if (((com.dstc.security.asn1.Integer)exp.getComponent()).getInt() != pvno)
      throw new BadVersionException("Not version 5");

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 1)
      throw new Asn1Exception("Bad Tag number");
    if (((com.dstc.security.asn1.Integer)exp.getComponent()).getInt() 
         != msgType)
      throw new BadMessageTypeException("Wrong msgType for an APRep");

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 2)
      throw new Asn1Exception("Bad Tag number");
    this.encPart = new EncryptedData(exp.getComponent().encode());
  }

  /**
   * Returns the EncryptedData in this APRep
   */
  public EncryptedData getEncrypted()
  {
    return this.encPart;
  }
}
