//XXXNeeds Constructor for KDC
//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5;

import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptionKey;

import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Explicit;

/**
 * <p>A class representing the Kerberos ASN.1 structure EncKDCRepPart.
 *
 * <pre>
 *   EncKDCRepPart ::=   
 *      SEQUENCE {
 *               key[0]                       EncryptionKey,
 *               last-req[1]                  LastReq,
 *               nonce[2]                     INTEGER,
 *               key-expiration[3]            KerberosTime OPTIONAL,
 *               flags[4]                     TicketFlags,
 *               authtime[5]                  KerberosTime,
 *               starttime[6]                 KerberosTime OPTIONAL,
 *               endtime[7]                   KerberosTime,
 *               renew-till[8]                KerberosTime OPTIONAL,
 *               srealm[9]                    Realm,
 *               sname[10]                    PrincipalName,
 *               caddr[11]                    HostAddresses OPTIONAL
 *      }
 * </pre>
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class EncKDCRepPart extends Sequence
{
  private EncryptionKey key = null;
  private LastReq lastReq = null;
  private int nonce;
  private KerberosTime keyExp = null;
  private TicketFlags flags = null;
  private KerberosTime authTime = null;
  private KerberosTime startTime = null;
  private KerberosTime endTime = null;
  private KerberosTime renewTill = null;
  private Realm sRealm = null;
  private PrincipalName sName = null;
  private HostAddresses caddr = null;

  public EncKDCRepPart() {}

  /**
   * Constructs an EncKDCRepPart from a DER encoding
   */
  public EncKDCRepPart(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);
    int i=0;
    
    Explicit exp = (Explicit)components.elementAt(i++);
    key = new EncryptionKey(exp.getComponent().encode());

    exp = (Explicit)components.elementAt(i++);
    this.lastReq = new LastReq(exp.getComponent().encode());

    exp = (Explicit)components.elementAt(i++);
    this.nonce = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();

    exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 3)
    {
      this.keyExp = new KerberosTime(exp.getComponent().encode());
      i++;
    }
    
    exp = (Explicit)components.elementAt(i++);
    this.flags = new TicketFlags(exp.getComponent().encode());

    exp = (Explicit)components.elementAt(i++);
    this.authTime = new KerberosTime(exp.getComponent().encode());

    exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 6)
    {
      this.startTime = new KerberosTime(exp.getComponent().encode());
      i++;
    }
    
    exp = (Explicit)components.elementAt(i++);
    this.endTime = new KerberosTime(exp.getComponent().encode());

    exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 8)
    {
      this.renewTill = new KerberosTime(exp.getComponent().encode());
      i++;
    }
    
    exp = (Explicit)components.elementAt(i++);
    this.sRealm = new Realm(exp.getComponent().encode());

    exp = (Explicit)components.elementAt(i++);
    this.sName = new PrincipalName(exp.getComponent().encode());

    if (i < components.size()) {
      exp = (Explicit)components.elementAt(i++);
      caddr = new HostAddresses(exp.getComponent().encode());
    }
  }

  /**
   * Returns the nonce sent back by the KDC
   */
  public int getNonce() 
  {
    return nonce;
  }

  /**
   * Returns the last request information
   */
  public LastReq getLastReq() 
  {
    return lastReq;
  }

  /**
   * Returns the key expiration time
   */
  public KerberosTime getKeyExpiryTime() 
  {
    return keyExp;
  }

  /**
   * Returns the server name for this EncKDCRepPart
   */
  public PrincipalName getServerName() 
  {
    return this.sName;
  }

  /**
   * Returns the server realm for this EncKDCRepPart
   */
  public Realm getServerRealm() 
  {
    return this.sRealm;
  }

  /**
   * Returns the client addresses for this EncKDCRepPart
   */
  public HostAddresses getClientAddresses() 
  {
    return this.caddr;
  }

  /**
   * Returns the encryption key for this EncKDCRepPart
   */
  public EncryptionKey getEncryptionKey() 
  {
    return this.key;
  }

  /**
   * Returns the ticket flags for this EncKDCRepPart
   */
  public TicketFlags getTicketFlags() 
  {
    return this.flags;
  }

  /**
   * Returns the auth time for this EncKDCRepPart
   */
  public KerberosTime getAuthTime() 
  {
    return this.authTime;
  }

  /**
   * Returns the start time for this EncKDCRepPart
   */
  public KerberosTime getStartTime() 
  {
    return this.startTime;
  }

  /**
   * Returns the end time for this EncKDCRepPart
   */
  public KerberosTime getEndTime() 
  {
    return this.endTime;
  }

  /**
   * Returns the renew until time for this EncKDCRepPart
   */
  public KerberosTime getRenewTill() 
  {
    return this.renewTill;
  }
}
