//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5;

import java.util.Vector;
import java.util.Enumeration;
import java.util.Properties;

import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptionType;
import com.dstc.security.kerberos.v5.crypto.EncryptedData;

import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Explicit;

/**
 * <p>A class representing the Kerberos ASN.1 structure KDC-REQ.
 *
 * <pre>
 *    KDC-REQ-BODY ::=   
 *          SEQUENCE {
 *            kdc-options[0]       KDCOptions,
 *            cname[1]             PrincipalName OPTIONAL,
 *                                    -- Used only in AS-REQ
 *            realm[2]             Realm, -- Server's realm
 *                                        -- Also client's in AS-REQ
 *            sname[3]             PrincipalName OPTIONAL,
 *            from[4]              KerberosTime OPTIONAL,
 *            till[5]              KerberosTime,
 *            rtime[6]             KerberosTime OPTIONAL,
 *            nonce[7]             INTEGER,
 *            etype[8]             SEQUENCE OF INTEGER, -- EncryptionType,
 *                                        -- in preference order
 *            addresses[9]         HostAddresses OPTIONAL,
 *            enc-authorization-data[10]   EncryptedData OPTIONAL,
 *                         -- Encrypted AuthorizationData encoding
 *            additional-tickets[11]       SEQUENCE OF Ticket OPTIONAL
 *          }
 * </pre>
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class KDCReqBody extends Sequence 
{
  private KDCOptions kdcOptions = null;
  private PrincipalName cname = null;
  private Realm realm = null;
  private PrincipalName sname = null;
  private KerberosTime from = null;
  private KerberosTime till = null;
  private KerberosTime rtime = null;
  private int nonce;
  private EncryptionType encType = null;
  private HostAddresses addresses = null;
  private EncryptedData encAuthData = null;
  private TicketsSeq addTickets = null;

  /**
   * Constructs a KDCReqBody from client and server info
   */
  public KDCReqBody(KDCOptions kdcOptions, PrincipalName client, Realm realm, 
                    PrincipalName server, KerberosTime from, 
                    KerberosTime till, KerberosTime rtime, int nonce,
                    EncryptionType encType, HostAddresses addresses,
    EncryptedData encAuthData, TicketsSeq addTickets) 
  {
    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    this.kdcOptions = kdcOptions;
    exp.addComponent(kdcOptions);
    addComponent(exp);

    if (client != null) 
    {
      this.cname = client;
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
      exp.addComponent(client);
      addComponent(exp);
    }

    this.realm = realm;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 2);
    exp.addComponent(this.realm);
    addComponent(exp);

    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 3);
    exp.addComponent(server);
    addComponent(exp);

    if (from != null) 
    {
      this.from = from;
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 4);
      exp.addComponent(from);
      addComponent(exp);
    }

    this.till = till;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 5);
    exp.addComponent(till);
    addComponent(exp);

    // kdcOptions and config file should determine if rtime should be included
    if (rtime != null) 
    {
      this.rtime = rtime;
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 6);
      exp.addComponent(rtime);
      addComponent(exp);
    }

    this.nonce = nonce;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 7);
    exp.addComponent(new com.dstc.security.asn1.Integer(nonce));
    addComponent(exp);

    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 8);
    this.encType = encType;
    exp.addComponent(encType);
    addComponent(exp);

    if (addresses != null) 
    {
      this.addresses = addresses;
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 9);
      exp.addComponent(addresses);
      addComponent(exp);
    }

    if (encAuthData != null) 
    {
      this.encAuthData = encAuthData;
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 10);
      exp.addComponent(encAuthData);
      addComponent(exp);
    }

    if (addTickets != null) 
    {
      this.addTickets = addTickets;
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 11);
      exp.addComponent(addTickets);
      addComponent(exp);
    }
  }

  /**
   * Constructs a KDCReqBody from a DER encoding
   */
  public KDCReqBody(byte encoded[]) throws Asn1Exception, BadVersionException
  {
    doDecode(encoded);

    Enumeration comps = getComponents().elements();
   
    Explicit exp = (Explicit) comps.nextElement();
    if (exp.getTagNumber() != 0)
      throw new Asn1Exception("Bad tag number");
    kdcOptions = new KDCOptions(exp.getComponent().encode());

    exp = (Explicit) comps.nextElement();
    if (exp.getTagNumber() == 1) 
    {
      // there's a client name
      cname = new PrincipalName(exp.getComponent().encode());
      exp = (Explicit) comps.nextElement();
    }

    if (exp.getTagNumber() != 2)
      throw new Asn1Exception("Bad tag number");
    realm = new Realm(exp.getComponent().encode());

    exp = (Explicit) comps.nextElement();
    if (exp.getTagNumber() == 3) 
    {
      // there's a server name
      sname = new PrincipalName(exp.getComponent().encode());
      exp = (Explicit) comps.nextElement();
    }

    if (exp.getTagNumber() == 4) 
    {      
      // there's a from time
      from = new KerberosTime(exp.getComponent().encode());
      exp = (Explicit) comps.nextElement();
    }    

    if (exp.getTagNumber() == 5) 
    {      
      // there's a till time
      till = new KerberosTime(exp.getComponent().encode());
      exp = (Explicit) comps.nextElement();
    }

    if (exp.getTagNumber() == 6) 
    {      
      // there's a renewTill time
      rtime = new KerberosTime(exp.getComponent().encode());
      exp = (Explicit) comps.nextElement();
    }

    if (exp.getTagNumber() != 7)
      throw new Asn1Exception("Bad tag number");
    nonce = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();

    exp = (Explicit) comps.nextElement();
    if (exp.getTagNumber() != 8)
      throw new Asn1Exception("Bad tag number");
    encType = new EncryptionType(exp.getComponent().encode());

    if (!comps.hasMoreElements()) return;

    exp = (Explicit) comps.nextElement();
    if (exp.getTagNumber() == 9) 
    {
      // there's host addresses
      addresses = new HostAddresses(exp.getComponent().encode());
      if (comps.hasMoreElements())
    exp = (Explicit) comps.nextElement();
      else return;
    }

    if (exp.getTagNumber() == 10) 
    {    
      // there's encrypted auth data
      encAuthData = new EncryptedData(exp.getComponent().encode());
      if (comps.hasMoreElements()) 
exp = (Explicit) comps.nextElement();
      else return;
    }

    if (exp.getTagNumber() == 11)// there's additional tickets
      addTickets = new TicketsSeq(exp.getComponent().encode());
  }

  /**
   * Returns the KDCOptions of this request.
   */
  public KDCOptions getOptions() 
  {
    return kdcOptions;
  }

  /**
   * Returns the client name for this KDCReqBody.
   */
  public PrincipalName getClientName() 
  {
    return cname;
  }

  /**
   * Returns the realm of this request.
   */
  public Realm getRealm() 
  {
    return realm;
  }

  /**
   * Returns the server name for this KDCReqBody.
   */
  public PrincipalName getServerName() 
  {
    return sname;
  }

  /**
   * Returns the from time of this request.
   */
  public KerberosTime getFrom() 
  {
    return from;
  }

  /**
   * Returns the till time of this request.
   */
  public KerberosTime getTill() 
  {
    return till;
  }

  /**
   * Returns the renew till time of this request.
   */
  public KerberosTime getRenewTill() 
  {
    return rtime;
  }

  /**
   * Returns the nonce value of this request.
   */
  public int getNonce() 
  {
    return nonce;
  }

  /**
   * Returns the EncryptionType of this request.
   */
  public EncryptionType getEncryptionType() 
  {
    return encType;
  }

  /**
   * Returns the address embedded in this request, if any.
   */
  public HostAddresses getClientAddresses() 
  {
    return addresses;
  }

  /**
   * Returns the encypted AuthorizationData of this request.
   */
  public EncryptedData getEncAuthData() 
  {
    return encAuthData;
  }

  /**
   * Returns the additional tickets of this request.
   */
  public TicketsSeq getAdditionalTickets() 
  {
    return addTickets;
  }
}

