//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5.base;

import java.util.Vector;
import java.util.Iterator;

import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.GeneralString;
import com.dstc.security.asn1.Explicit;

/**
 * <p>A class representing the Kerberos ASN.1 structure PrincipalName.
 *
 * <pre>
 *   PrincipalName ::=  SEQUENCE {
 *                       name-type[0]     INTEGER,
 *                       name-string[1]   SEQUENCE OF GeneralString
 *                      }
 *
 * </pre>
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class PrincipalName extends Sequence
{
  public static final int NT_UNKNOWN = 0;
  public static final int NT_PRINCIPAL = 1;
  public static final int NT_SRV_INST = 2;
  public static final int NT_SRV_HST = 3;
  public static final int NT_SRV_XHST = 4;
  public static final int NT_UID = 5;

  private int nameType;
  private NameString ns;

  /**
   * Constructs a PrincipalName from a name type and the name components as
   * a Vector of String components
   */
  public PrincipalName(int type, Vector nameString)
  {
    this.nameType = type;
    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(type));
    addComponent(exp);

    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    this.ns = new NameString(nameString);
    exp.addComponent(ns);
    addComponent(exp);
  }

  /**
   * Constructs a PrincipalName from a DER encoding
   */
  public PrincipalName(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);

    Explicit exp = (Explicit)components.elementAt(0);
    if (exp.getTagNumber() != 0)
      throw new Asn1Exception("Bad Tag number");
    this.nameType 
      = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();

    exp = (Explicit)components.elementAt(1);
    if (exp.getTagNumber() != 1)
      throw new Asn1Exception("Bad Tag number");
    this.ns = new NameString(((Sequence)exp.getComponent()).encode());
  }

  /**
   * Returns the name type for this PrincipalName
   */
  public int getNameType()
  {
    return this.nameType;
  }

  /**
   * Returns the name components for this PrincipalName as a Vector of String
   */
  public Vector getNameString()
  {
    return this.ns.nameString;
  }

  public String toString()
  {
    Iterator it = this.ns.nameString.iterator();
    StringBuffer sb = new StringBuffer((String)it.next());
    while (it.hasNext())
    {
      sb.append("/").append(it.next());
    }
    return sb.toString();
  }

  /**
   * <p>Inner class representing an ASN.1 Sequence of GeneralString 
   */
  protected class NameString extends Sequence
  {
    private Vector nameString;

    /**
     * Constructs a NameString from its name components as a Vector of String
     */
    protected NameString(Vector nameString)
    {
      this.nameString = nameString;
      for (int i=0; i< nameString.size(); i++)
      {
        addComponent(new GeneralString((String)nameString.elementAt(i)));
      }
    }

    /**
     * Constructs a NameString from a DER encoding
     */
    protected NameString(byte encoded[]) throws Asn1Exception
    {
      this.nameString = new Vector();
      doDecode(encoded);
      for (int i=0; i<components.size(); i++)
      {
        nameString.addElement(
          ((GeneralString)components.elementAt(i)).getName());
      }
    }
  }
}
