//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5.creds;

import java.util.Vector;

import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptionKey;

import com.dstc.security.asn1.Explicit;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;

/**
 * <p>A class representing the Kerberos ASN.1 structure EncTicketPart.
 *
 * <pre>
 *  EncTicketPart ::= [APPLICATION 3] SEQUENCE {
 *                      flags[0]             TicketFlags,
 *                      key[1]               EncryptionKey,
 *                      crealm[2]            Realm,
 *                      cname[3]             PrincipalName,
 *                      transited[4]         TransitedEncoding,
 *                      authtime[5]          KerberosTime,
 *                      starttime[6]         KerberosTime OPTIONAL,
 *                      endtime[7]           KerberosTime,
 *                      renew-till[8]        KerberosTime OPTIONAL,
 *                      caddr[9]             HostAddresses OPTIONAL,
 *                      authorization-data[10]   AuthorizationData OPTIONAL
 *                    }
 * </pre>
 * @version 0.98, 98/12/25
 * @author Ming Yung
 */
public final class EncTicketPart extends Explicit
{
  public static final int TAG_KRB_ENC_TICKET = 3;

  private TicketFlags ticketFlags;
  private EncryptionKey key;
  private Realm crealm;
  private PrincipalName cName;
  //private TransitedEncoding transited;
  private KerberosTime authtime;
  private KerberosTime starttime;
  private KerberosTime endtime;
  private KerberosTime renewTill;
  private HostAddresses caddr;
  //private AuthorizationData authData;

  /**
   * Constructs an EncTicketPart from a DER encoding
   */
  public EncTicketPart(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);

    if (this.getTagNumber() != TAG_KRB_ENC_TICKET)
      throw new Asn1Exception("Bad Tag number");

    Sequence seq = (Sequence)this.getComponent();
    Vector comps = seq.getComponents();

    int i=0;

    Explicit exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 0)
      throw new Asn1Exception("Bad Tag Number");
    this.ticketFlags = new TicketFlags(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 1)
      throw new Asn1Exception("Bad Tag number");
    this.key = new EncryptionKey(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 2)
      throw new Asn1Exception("Bad Tag number");
    this.crealm = new Realm(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 3)
      throw new Asn1Exception("Bad Tag number");
    this.cName = new PrincipalName(exp.getComponent().encode());

    i++;

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 5)
      throw new Asn1Exception("Bad Tag number");
    this.authtime = new KerberosTime(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(i);
    if (exp.getTagNumber() == 6)
    {
      this.starttime = new KerberosTime(exp.getComponent().encode());
      i++;
    }

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 7)
      throw new Asn1Exception("Bad Tag number");
    this.endtime = new KerberosTime(exp.getComponent().encode());

    if (i == comps.size()) return;

    exp = (Explicit)comps.elementAt(i);
    if (exp.getTagNumber() == 8)
    {
      this.renewTill = new KerberosTime(exp.getComponent().encode());
      i++;
    }

    if (i == comps.size()) return;

    exp = (Explicit)comps.elementAt(i);
    if (exp.getTagNumber() == 9)
    {
      this.caddr = new HostAddresses(exp.getComponent().encode());
      i++;
    }
  } 
    
  /**
   * Returns the Realm for this EncTicketPart
   */
  public Realm getRealm()
  {
    return this.crealm;
  }

  /**
   * Returns the client principal name for this EncTicketPart
   */
  public PrincipalName getClientName()
  {
    return this.cName;
  }

  /**
   * Returns the EncryptionKey for this EncTicketPart
   */
  public EncryptionKey getKey()
  {
    return this.key;
  }
}
