//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5.crypto;

import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Explicit;
import com.dstc.security.asn1.OctetString;

/**
 * <p>Class representing the Kerberos ASN.1 structure EncryptedData.
 *
 * <pre>
 * EncryptedData ::=  SEQUENCE {
 *                      etype[0]    INTEGER,
 *                      kvno[1]     INTEGER OPTIONAL,
 *                      cipher[2]   OCTET STRING
 * }
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class EncryptedData extends Sequence
{
  private int eType;
  private int kvno;
  private byte[] cipherText;

  /**
   * Constructs an EncryptedData from the type, key version and ciphertext
   */
  public EncryptedData(int eType, int kvno, byte[] cipherText)
  {
    this.eType = eType;
    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(eType));
    addComponent(exp);

    this.kvno = kvno;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    exp.addComponent(new com.dstc.security.asn1.Integer(kvno));
    addComponent(exp);
      
    this.cipherText = cipherText;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 2);
    exp.addComponent(new OctetString(cipherText));
    addComponent(exp);
  }

  /**
   * Constructs an EncryptedData from the type and ciphertext
   */
  public EncryptedData(int eType, byte[] cipherText)
  {
    this.eType = eType;
    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(eType));
    addComponent(exp);

    this.cipherText = cipherText;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 2);
    exp.addComponent(new OctetString(cipherText));
    addComponent(exp);
  }

  /**
   * Constructs an EncryptedData from a  DER encoding
   */
  public EncryptedData(byte[] encoded) throws Asn1Exception
  {
    doDecode(encoded);
    int i = 0;

    Explicit exp = (Explicit)components.elementAt(i++);
    if (exp.getTagNumber() == 0)
      this.eType 
        = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();

    exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 1)
    {
      this.kvno = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();
      i++;
    }

    exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 2)
      this.cipherText = ((OctetString)exp.getComponent()).getBytes();
  }

  /**
   * Returns the encryption type
   */
  public int getEncryptionType()
  {
    return this.eType;
  }

  /**
   * Returns the ciphertext as a byte array
   */
  public byte[] getCipherText()
  {
    return (byte[])cipherText.clone();
  }
}
