//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.pki.pkcs10;

import java.security.Signature;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.security.spec.X509EncodedKeySpec;

import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.BitString;
import com.dstc.security.x509.AlgorithmId;
import com.dstc.security.provider.OID;

/**
 * <p> Implements the ASN.1 structure CertificationRequest.
 *
 * <pre>
 * CertificationRequest ::= SEQUENCE {
 *   certificationRequestInfo CertificationRequestInfo,
 *   signatureAlgorithm SignatureAlgorithmIdentifier,
 *   signature Signature }
 * 
 * SignatureAlgorithmIdentifier ::= AlgorithmIdentifier
 * 
 * Signature ::= BIT STRING
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class CertificationRequest extends Sequence
{
  private CertificationRequestInfo crInfo;
  private String sigAlgName;
  private byte[] signature;

  /**
   * Constructs a signed CertificateRequest from a CertificationRequestInfo,
   * the signature algorithm name and the Private Key
   */
  public CertificationRequest(CertificationRequestInfo info, 
                              String sigAlgName, PrivateKey priv)
  {
    this.crInfo = info;
    addComponent(info);

    this.sigAlgName = sigAlgName;
    addComponent(new AlgorithmId(sigAlgName));

    try
    {
      Signature sig =
        Signature.getInstance(OID.getAlgName(sigAlgName));

      sig.initSign(priv);
      sig.update(info.encode());

      this.signature = sig.sign();
      addComponent(new BitString(this.signature));
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }

  /**
   * Constructs a CertificationRequest from a DER encoding
   */
  public CertificationRequest(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);

    this.crInfo = 
      new CertificationRequestInfo(((Asn1)components.elementAt(0)).encode());

    AlgorithmId algoId = 
      new AlgorithmId(((Asn1)components.elementAt(1)).encode());
    this.sigAlgName = algoId.getOid();

    this.signature = ((BitString)components.elementAt(2)).getBytes();
  }

  /**
   * Returns the CertificationRequestInfo for this CertificationRequest
   */
  public CertificationRequestInfo getCertificationRequestInfo()
  {
    return this.crInfo;
  }

  /**
   * Returns the signature algorithm name for this CertificationRequest
   */
  public String getSigAlgName()
  {
    return OID.getAlgName(this.sigAlgName);
  }

  /**
   * Returns the signature for this CertificationRequest
   */
  public byte[] getSignature()
  {
    return this.signature;
  }

  /**
   * Verifies the signature for this CertificationRequest
   */
  public boolean verifySignature()
  {
    try
    {
      Signature sig =
        Signature.getInstance(getSigAlgName());

      KeyFactory keyFact
        = KeyFactory.getInstance(this.crInfo.getKeyAlgName());

      PublicKey pub = keyFact.generatePublic(new X509EncodedKeySpec(
          this.crInfo.getSubjectPublicKeyInfo().encode()));

      sig.initVerify(pub);
      sig.update(this.crInfo.encode());

      return sig.verify(this.signature);
    }
    catch (Exception e)
    {
      e.printStackTrace();
      return false;
    }
  }
}
