//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.pki.pkcs10;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import java.util.Vector;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Constructed;
import com.dstc.security.asn1.SetOf;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.x509.X500Name;
import com.dstc.security.x509.Attribute;
import com.dstc.security.x509.SubjectPublicKeyInfo;
import com.dstc.security.provider.OID;

/**
 * <p>Implements the ASN.1 structure CertificationRequestInfo.
 *
 * <pre>
 * CertificationRequestInfo ::= SEQUENCE {
 *   version Version,
 *   subject Name,
 *   subjectPublicKeyInfo SubjectPublicKeyInfo,
 *   attributes [0] IMPLICIT Attributes }
 * 
 * Version ::= INTEGER
 * 
 * Attributes ::= SET OF Attribute
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class CertificationRequestInfo extends Sequence
{
  private int version;
  private X500Name subject;
  private SubjectPublicKeyInfo spki;
  private Set atts;

  /**
   * Constructs a CertificationRequestInfo from the subject X500Name,
   * the SubjectPublicKeyInfo and a Set of Attributes
   */
  public CertificationRequestInfo(X500Name dn, SubjectPublicKeyInfo spki,
                                  Set attributes)
  {
    this.version = 0;
    addComponent(new com.dstc.security.asn1.Integer(0));

    this.subject = dn;
    addComponent(dn);

    this.spki = spki;
    addComponent(spki);

    this.atts = attributes;
    if (attributes != null)
    {
      SetOf setOf = new SetOf();
      Iterator it = attributes.iterator();
      while (it.hasNext())
      {
        setOf.addComponent((Attribute)it.next());
      }
      setOf.setTagClass(Asn1.TAG_CLASS_CONTEXT);
      setOf.setTagNumber(0);
      this.addComponent(setOf);
    }
  }

  /**
   * Constructs a CertificationRequestInfo from a DER encoding
   */
  public CertificationRequestInfo(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);
    int i=0;
  
    this.version = 
      ((com.dstc.security.asn1.Integer)components.elementAt(i++)).getInt();

    this.subject = new X500Name(((Asn1)components.elementAt(i++)).encode()); 

    this.spki = 
      new SubjectPublicKeyInfo(((Asn1)components.elementAt(i++)).encode()); 

    if (components.size() == i) return;

    this.atts = new HashSet();
    Vector comps = ((Constructed)components.elementAt(i)).getComponents();
    for (int j=0; j<comps.size(); j++)
    {
      this.atts.add(new Attribute(((Asn1)comps.elementAt(j)).encode()));
    }
  }

  /**
   * Returns the subject name for this CertificationRequestInfo
   */
  public X500Name getSubject()
  {
    return this.subject;
  }

  /**
   * Returns SubjectPublicKeyInfo for this CertificationRequestInfo
   */
  public SubjectPublicKeyInfo getSubjectPublicKeyInfo()
  {
    return this.spki;
  }

  /**
   * Returns the key algorithm name for this CertificationRequestInfo
   */
  public String getKeyAlgName()
  {
    return OID.getAlgName(getSubjectPublicKeyInfo().getAlgorithmId().getOid());
  }

  /**
   * Returns the Set of signed Attributes for this CertificationRequestInfo
   */
  public Set getSignedAttributes()
  {
    return this.atts;
  }
}
