//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.provider;

import java.util.Arrays;
import java.io.ByteArrayOutputStream;
import java.math.BigInteger;
import java.util.Random;
import java.security.SignatureSpi;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.security.MessageDigest;
import java.security.KeyFactory;
import java.security.InvalidParameterException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.SignatureException;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.X509EncodedKeySpec;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Integer;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.OctetString;
import com.dstc.security.x509.Attribute;
import com.dstc.security.cms.atts.*;
import java.security.*;
import java.security.spec.*;
import javax.crypto.*;
import javax.crypto.interfaces.*;
import javax.crypto.spec.*;

/**
 * <p>Implements the Diffie-Hellman (DH) signing algorithm for proof-of-possesion,
 * eg. in a PKCS#10 certification request
 *
 * @see java.security.Signature
 * @see com.dstc.security.provider.DSAParameters
 * @see com.dstc.security.provider.DSAPublicKey
 * @see com.dstc.security.provider.DSAPrivateKey
 *
 * @version 0.98, 99/02/09
 * @author Nishanth Chandran
 * @author Ming Yung
 */
public class DHEphemeralSignature extends DHSignature 
{
  protected void engineSetParameter(AlgorithmParameterSpec params)
    throws InvalidAlgorithmParameterException
  {
    //to do nothing
  }
 
  //Initializes for signing with a private key
  protected void engineInitSign(PrivateKey privateKey)
    throws InvalidKeyException
  {
    if(privateKey.getAlgorithm().equals("DH"))
    {
      this.CAPvt = (DHPrivateKey)privateKey;
    }
    else{
      throw new InvalidKeyException("Not a DH Private Key");
    }

    try
    {
      dhParamSpec = CAPvt.getParams();

      generateTempKeys();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }
  
  //Signs the message in the buffer and returns the signature
  protected byte[] engineSign() throws SignatureException
  {
    try 
    {
      text = bos.toByteArray();

      p = dhParamSpec.getP();
      g = dhParamSpec.getG();
      l = dhParamSpec.getL();
    
      //secret computation
      keyAgree = KeyAgreement.getInstance("DH","DSTC");
      keyAgree.init(CAPvt);
        
      keyAgree.doPhase(dhPubKey, true);
      byte[] signature = computeSign(); 
      DHSigEphemeral der = new DHSigEphemeral(p,g,l,dhPvtKey.getX(), signature);
      return der.encode();
    }
    catch(Exception e)
    {
      e.printStackTrace();
      throw new SignatureException(e.getMessage());
    }
  }

  //Initializes for verification with a public key
  protected void engineInitVerify(PublicKey publicKey)
    throws InvalidKeyException
  {
    if (publicKey.getAlgorithm().equals("DH"))
    {
      this.CAPub = (DHPublicKey)publicKey;
    }
    else
    {
      throw new InvalidKeyException("Not a DH Public Key");
    }
    try
    {
      bos = new ByteArrayOutputStream();    
    }
    catch(Exception e)
    {
      System.out.println(e.toString() + " in engineInitVerify");
    }
  }

  //Verifies Signature 
  protected boolean engineVerify(byte sigBytes[])
    throws SignatureException
  {
    try 
    {
      text = bos.toByteArray();

      DHSigEphemeral dhSign = new DHSigEphemeral(sigBytes);
  
      p = dhSign.domainParams.p;
      g = dhSign.domainParams.g;
      l = dhSign.domainParams.l;
        
      dhParamSpec = new DHParameterSpec(p,g,l);
    
      BigInteger x = dhSign.tempPrivateKey;
      dhPvtKey = new DHPrivateKey(x,p,g,l);
    
      keyAgree = KeyAgreement.getInstance("DH","DSTC");
      keyAgree.init(dhPvtKey);
      keyAgree.doPhase(CAPub,true);
    
      byte[] hashValue = computeSign(); 
 
      if (!Arrays.equals(dhSign.hashValue,hashValue))
        return false;
      else
        return true;
    }
    catch (Exception e)
    {
      e.printStackTrace();
      throw new SignatureException(e.getMessage());
    }
  }

  private void generateTempKeys()
  {
    try
    {
      bos = new ByteArrayOutputStream();   
      KeyPairGenerator keyGen = KeyPairGenerator.getInstance("DH","DSTC");
      keyGen.initialize(dhParamSpec);
    
      KeyPair tempPair = keyGen.generateKeyPair();
      dhPvtKey = (DHPrivateKey)tempPair.getPrivate();
      dhPubKey = (DHPublicKey)tempPair.getPublic();
    }
    catch (Exception e)
    {
      System.out.println(e.toString() + " in keyGeneration()");
    }
  }

  /**
   * <p> A class representing the ASN.1 structure DHSigEphemeral.
   */ 
  protected class DHSigEphemeral extends Sequence
  {
    private DomainParameters domainParams;
    private BigInteger tempPrivateKey;
    private byte[] hashValue;

    /**
     * Constructs a DHSigEphmeral with parameters p, g, l, the
     * temporary private key and the hash
     */
    protected DHSigEphemeral(BigInteger p, BigInteger g, int l,
                             BigInteger tempPriv, byte[] hash)
    {
      this.domainParams = new DomainParameters(p, g, l);
      addComponent(this.domainParams);
    
      this.tempPrivateKey = tempPriv;
      addComponent(new com.dstc.security.asn1.Integer(tempPriv));
  
      this.hashValue = hash;
      addComponent(new com.dstc.security.cms.atts.MessageDigest(hash));
    }

    /**
     * Constructs a DHSigEphemeral from a DER encoding
     */
    protected DHSigEphemeral(byte[] encoded) throws Asn1Exception
    {
      doDecode(encoded);
      int i=0;
  
      this.domainParams 
        = new DomainParameters(((Asn1)components.elementAt(i++)).encode());
  
      this.tempPrivateKey
        = ((com.dstc.security.asn1.Integer)components.elementAt(i++)).
            getBigInteger();
  
      Attribute att
        = new Attribute(((Asn1)components.elementAt(i++)).encode());
      this.hashValue = ((OctetString)att.getValues().elementAt(0)).getBytes();
    }

    /**
     * <p> Inner class representing ASN.1 structure DomainParameters.
     *
     */
    protected class DomainParameters extends Sequence
    {
      protected BigInteger p;
      protected BigInteger g;
      protected int l;
        
      /**
       * Constructs a DomainParameters from p, g, and q
       */
      protected DomainParameters(BigInteger p, BigInteger g, int l)
      {
        this.p = p;
        addComponent(new com.dstc.security.asn1.Integer(p));

        this.g = g;
        addComponent(new com.dstc.security.asn1.Integer(g));

        this.l = l;
        addComponent(new com.dstc.security.asn1.Integer(l));
      }
  
      /**
       * Constructs a DomainParameters from a DER encoding
       */
      protected DomainParameters(byte[] encoded) throws Asn1Exception
      {
        doDecode(encoded);
        int i=0;
    
        this.p = ((com.dstc.security.asn1.Integer)components.
                      elementAt(i++)).getBigInteger();
    
        this.g = ((com.dstc.security.asn1.Integer)components.
                elementAt(i++)).getBigInteger();

        this.l = ((com.dstc.security.asn1.Integer)components.
                elementAt(i++)).getInt();
      }
    }
  }
}
