//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.provider;

import java.math.BigInteger;
import java.security.Key;
import java.security.SecureRandom;
import java.security.InvalidKeyException;
import java.security.spec.AlgorithmParameterSpec;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.SecretKeySpec;
import javax.crypto.KeyAgreement;
import javax.crypto.KeyAgreementSpi;
import javax.crypto.ShortBufferException;

/**
 * <p>Implements the Diffie-Hellman Key Agreement algorithm.
 *
 * @see javax.crypto.KeyAgreement
 * @see com.dstc.security.provider.DHParameters
 * @see com.dstc.security.provider.DHPublicKey
 * @see com.dstc.security.provider.DHPrivateKey
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class DHKeyAgreement extends KeyAgreementSpi
{
  private SecureRandom random;
  private DHPrivateKey myPriv;
  private DHPublicKey yourPub;
  private BigInteger modulus;
  private BigInteger base;

  protected void engineInit(Key key, SecureRandom random) 
    throws InvalidKeyException
  {
    if (!(key instanceof DHPrivateKey))
      throw new InvalidKeyException("Not a DH Private Key");

    this.random = random;
    this.myPriv = (DHPrivateKey)key;
    this.modulus = myPriv.getParams().getP();
    this.base = myPriv.getParams().getG();
  }

  protected void engineInit(Key key, AlgorithmParameterSpec params, 
                            SecureRandom random) 
    throws InvalidKeyException, InvalidAlgorithmParameterException
  {
    if (!(key instanceof DHPrivateKey))
      throw new InvalidKeyException("Not a DH Private Key");

    this.random = random;
    this.myPriv = (DHPrivateKey)key;
    this.modulus = myPriv.getParams().getP();
    this.base = myPriv.getParams().getG();
  }

  protected Key engineDoPhase(Key key, boolean lastPhase) 
    throws InvalidKeyException, IllegalStateException
  {
    if (!(key instanceof DHPublicKey))
      throw new InvalidKeyException("Not a DH Public Key");

    if (lastPhase != true)
      throw new IllegalStateException("DH KeyAgreement has only two phases");

    this.yourPub = (DHPublicKey)key;

    if (!(yourPub.getParams().getG().equals(this.base) &&
        yourPub.getParams().getP().equals(this.modulus)))
      throw new InvalidKeyException("Public/Private Key parameters mismatch");

    return null;
  }

  protected byte[] engineGenerateSecret() throws IllegalStateException
  {
    return yourPub.getY().modPow(myPriv.getX(), this.modulus).toByteArray();
  }

  protected int engineGenerateSecret(byte[] sharedSecret, int offset) 
    throws IllegalStateException, ShortBufferException
  {
    byte val[] = yourPub.getY().modPow(myPriv.getX(), 
                                      this.modulus).toByteArray();
    if (sharedSecret.length - offset < val.length)
      throw new ShortBufferException("Buffer too short");

    System.arraycopy(val, 0, sharedSecret, offset, val.length);
    return val.length;
  }

  protected SecretKey engineGenerateSecret(String algorithm) 
    throws IllegalStateException, NoSuchAlgorithmException, 
           InvalidKeyException
  {
    byte val[] = yourPub.getY().modPow(myPriv.getX(), 
                                      this.modulus).toByteArray();
    try
    {
      SecretKeyFactory fact = SecretKeyFactory.getInstance(algorithm, "DSTC");
      SecretKeySpec spec = new SecretKeySpec(val, algorithm);

      //weird but since SecretKeySpec does not inherit from KeySpec ...
      return fact.translateKey(spec);
    }
    catch (NoSuchProviderException e)
    {
      throw new NoSuchAlgorithmException(algorithm + " Not Found");
    }
  }
}
