//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.provider;

import java.math.BigInteger;
import com.dstc.security.asn1.Integer;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;
import java.security.AlgorithmParametersSpi;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.DSAParameterSpec;
import java.security.spec.InvalidParameterSpecException;
import java.io.IOException;

/**
 * <p>Implements algorithm parameters for the DSA signature
 * algorithm following the DSS standard.
 *
 * @see java.security.AlgorithmParameters
 * @see com.dstc.security.provider.DSA
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class DSAParameters extends AlgorithmParametersSpi
{
  private DSAParameterSpec paramsSpec;
  private DSAParams params;

  protected void engineInit(AlgorithmParameterSpec paramSpec) 
    throws InvalidParameterSpecException
  {
    if (!(paramSpec instanceof DSAParameterSpec))
      throw new InvalidParameterSpecException("Not a DSA param spec");

    this.paramsSpec = (DSAParameterSpec)paramSpec;
    this.params = new DSAParams(paramsSpec.getP(),
                                paramsSpec.getQ(),
                                paramsSpec.getG());
  }

  protected void engineInit(byte[] params) throws IOException
  {
    try
    {
      this.params = new DSAParams(params);
      this.paramsSpec = new DSAParameterSpec(this.params.p,
                                             this.params.q,
                                             this.params.g);
    }
    catch (Exception e)
    {
      throw new IOException(e.toString());
    }
  }

  protected void engineInit(byte[] params, String format) throws IOException
  {
    engineInit(params);
  }

  protected AlgorithmParameterSpec engineGetParameterSpec(Class paramSpec) 
    throws InvalidParameterSpecException
  {
    if (!paramSpec.getName().equals("java.security.spec.DSAParameterSpec"))
      throw new InvalidParameterSpecException(paramSpec.getName() +
        " not supported");

    return this.paramsSpec;
  }

  protected byte[] engineGetEncoded() throws IOException
  {
    return params.encode();
  }

  protected byte[] engineGetEncoded(String format) throws IOException
  {
    return engineGetEncoded();
  }

  protected String engineToString()
  {
    return "DSA Parameters";
  }

  /**
   * <p>Inner class representing the ASN.1 structure DSAParameters.
   *
   * <p>DSAParameters ::= SEQUENCE {
   *       p INTEGER,
   *       q INTEGER,
   *       g INTEGER
   *    }
   */
  protected class DSAParams extends Sequence
  {
    private BigInteger g, p, q;
  
    /**
     * Constructs a DSAParams given parameters p, q, g
     */
    protected DSAParams (BigInteger p, BigInteger q, BigInteger g) 
    {
      this.p = p;
      this.q = q;
      this.g = g;
      this.addComponent(new com.dstc.security.asn1.Integer(p));
      this.addComponent(new com.dstc.security.asn1.Integer(q));
      this.addComponent(new com.dstc.security.asn1.Integer(g));
    }
  
    /**
     * Constructs a DSAParams from a DER encoding
     */
    protected DSAParams(byte encoded[]) throws Asn1Exception
    {
      doDecode(encoded);
      this.p = ((com.dstc.security.asn1.Integer)
                 components.elementAt(0)).getBigInteger();
      this.q = ((com.dstc.security.asn1.Integer)
                 components.elementAt(1)).getBigInteger();
      this.g = ((com.dstc.security.asn1.Integer)
                 components.elementAt(2)).getBigInteger();
    }
  }
}
