//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.provider;

import java.math.BigInteger;
import com.dstc.security.asn1.Integer;
import com.dstc.security.asn1.OctetString;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;
import java.security.AlgorithmParametersSpi;
import java.security.spec.AlgorithmParameterSpec;
import javax.crypto.spec.PBEParameterSpec;
import java.security.spec.InvalidParameterSpecException;
import java.io.IOException;

/**
 * <p>Implements java.security.AlgorithmParameters for the password-based
 * encryption algorithm in PKCS#5.
 *
 * @see java.security.AlgorithmParameters
 * @see com.dstc.security.provider.PBECipher
 * @see javax.crypto.Cipher
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class PBEParameters extends AlgorithmParametersSpi
{
  private PBEParameterSpec paramsSpec;
  private PBEParams params;

  protected void engineInit(AlgorithmParameterSpec paramSpec) 
    throws InvalidParameterSpecException
  {
    if (!(paramSpec instanceof PBEParameterSpec))
      throw new InvalidParameterSpecException("Not a PBE param spec");

    this.paramsSpec = (PBEParameterSpec)paramSpec;
    this.params = new PBEParams(paramsSpec.getSalt(),
                                paramsSpec.getIterationCount());
  }

  protected void engineInit(byte[] params) throws IOException
  {
    try
    {
      this.params = new PBEParams(params);
      this.paramsSpec = new PBEParameterSpec(this.params.salt, this.params.ic);
    }
    catch (Exception e)
    {
      throw new IOException(e.toString());
    }
  }

  protected void engineInit(byte[] params, String format) throws IOException
  {
    engineInit(params);
  }

  protected AlgorithmParameterSpec engineGetParameterSpec(Class paramSpec) 
    throws InvalidParameterSpecException
  {
    return this.paramsSpec;
  }

  protected byte[] engineGetEncoded() throws IOException
  {
    return params.encode();
  }

  protected byte[] engineGetEncoded(String format) throws IOException
  {
    return engineGetEncoded();
  }

  protected String engineToString()
  {
    return "PBE Parameters";
  }

  /**
   * <p>An Inner class representing the PKCS#5 ASN.1 structure PBEParameter.
   *
   *  PBEParameter ::= SEQUENCE {
   *    salt OCTET STRING SIZE(8),
   *    iterationCount INTEGER }
   */
  protected class PBEParams extends Sequence
  {
    private byte[] salt;
    private int ic; 

    /**
     * Constructs a PBEParams given salt and iteration count
     */
    protected PBEParams(byte[] salt, int ic) 
    {
      this.salt = salt;
      this.addComponent(new OctetString(salt));

      this.ic = ic;
      this.addComponent(new com.dstc.security.asn1.Integer(ic));
    }
  
    /** 
     * Constructs a PBEParams from a DER encoding
     */ 
    protected PBEParams(byte encoded[]) throws Asn1Exception
    {
      doDecode(encoded);
      this.salt = ((OctetString)components.elementAt(0)).getBytes();
      this.ic = ((com.dstc.security.asn1.Integer)
                 components.elementAt(1)).getBigInteger().intValue();
    }
  }
}
