//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.provider;

import java.math.BigInteger;
import com.dstc.security.asn1.Integer;
import com.dstc.security.asn1.OctetString;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;
import java.security.AlgorithmParametersSpi;
import java.security.spec.AlgorithmParameterSpec;
import javax.crypto.spec.RC2ParameterSpec;
import java.security.spec.InvalidParameterSpecException;
import java.io.IOException;

/**
 * <p>Implements java.security.AlgorithmParameters for the RC2-CBC
 * encryption algorithm
 *
 * @see java.security.AlgorithmParameters
 * @see com.dstc.security.provider.RC2Cipher
 * @see javax.crypto.Cipher
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class RC2Parameters extends AlgorithmParametersSpi
{
  private RC2Params params;
  private RC2ParameterSpec paramsSpec;

  protected void engineInit(AlgorithmParameterSpec paramSpec) 
    throws InvalidParameterSpecException
  {
    if (!(paramSpec instanceof RC2ParameterSpec))
      throw new InvalidParameterSpecException("Not an RC2 param spec");

    this.paramsSpec = (RC2ParameterSpec)paramSpec;
    this.params = new RC2Params(paramsSpec.getEffectiveKeyBits(),
                                paramsSpec.getIV());
  }

  protected void engineInit(byte[] encoded) throws IOException
  {
    try
    {
      this.params = new RC2Params(encoded);
      this.paramsSpec = new RC2ParameterSpec(this.params.eKB, this.params.iv);
    }
    catch (Exception e)
    {
      throw new IOException(e.toString());
    }
  }

  protected void engineInit(byte[] params, String format) throws IOException
  {
    engineInit(params);
  }

  protected AlgorithmParameterSpec engineGetParameterSpec(Class paramSpec) 
    throws InvalidParameterSpecException
  {
    if (!paramSpec.getName().equals("javax.crypto.spec.RC2ParameterSpec"))
      throw new InvalidParameterSpecException(paramSpec.getName()
        + " not supported");
    return this.paramsSpec;
  }

  protected byte[] engineGetEncoded() throws IOException
  {
    return params.encode();
  }

  protected byte[] engineGetEncoded(String format) throws IOException
  {
    return engineGetEncoded();
  }

  protected String engineToString()
  {
    return "RC2 Parameters";
  }

  /**
   * <p>An Inner class representing the ASN.1 structure RC2Parameter.
   * 
   * parameterVersions of 160, 120, 58 corresponds to
   * effective key bits of 40, 64 and 128 respectively
   */
  protected class RC2Params extends Sequence
  {
    private int eKB; 
    private int parameterVersion;
    private byte[] iv;

    /**
     * Constructs an RC2Params from the effective key bits and iv
     */
    protected RC2Params(int eKB, byte[] iv)
    {
      this.eKB = eKB;
      if (eKB == 40)
        this.parameterVersion = 160;
      else if (eKB == 64)
        this.parameterVersion = 120;
      else 
        this.parameterVersion = 58;
      addComponent(new com.dstc.security.asn1.Integer(parameterVersion));

      this.iv = iv;
      addComponent(new OctetString(iv));
    }

    /**
     * Constructs an RC2Params from a DER encoding
     */
    protected RC2Params(byte encoded[]) throws Asn1Exception
    {
      doDecode(encoded);

      this.parameterVersion = 
       ((com.dstc.security.asn1.Integer) components.elementAt(0)).getInt();
      if (parameterVersion == 160)
        this.eKB = 40;
      else if (parameterVersion == 120)
        this.eKB = 64;
      else 
        this.eKB = 128;

      this.iv = ((OctetString)components.elementAt(1)).getBytes();
    }
  }
}
