//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.ssl;

import java.util.Iterator;
import java.util.Vector;
import java.util.Calendar;
import java.io.IOException;

import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;

/**
 * A class for ClientHello Handshake messages sent by a client to request
 * a handshake be started
 *
 * @author Ming Yung
 */
final class ClientHello extends HandShake
{
  private byte[] version;
  private byte[] compression;
  private byte[] random;
  private byte[] sessionId;
  private Vector cipherSuites;

  /**
   * Constructs a ClientHello from an XDRInputStream with a 
   * callback to a controlling ServerHandShaker
   */
  protected ClientHello(HandShaker handShaker, byte[] data) 
    throws IOException
  {
    super(handShaker, data);

    if (Debug.debug >= Debug.DEBUG_MSG)
      Debug.debug("++ ClientHello");
  }

  /**
   * Constructs a ClientHello message with a callback to a ClientHandShaker
   *
   * <p>The body of this message contains a sessionID, a client random,
   *    and a list of ciphersuites, all obtained via a callback to 
   *    the ClientHandShaker
   */
  protected ClientHello(ClientHandShaker handShaker) 
    throws IOException
  {
    this.handShaker = handShaker;
    this.msgType = CLIENT_HELLO;
    
    this.version = handShaker.getProtocolVersion();
    this.compression = handShaker.getCompressionMethods();
    this.sessionId = handShaker.getSessionID();
    this.random = handShaker.getClientHelloRandom();
    this.cipherSuites = handShaker.getEnabledCipherSuites();

    if (this.cipherSuites == null)
      throw new SSLException("No enabled cipher suites to offer");

    if (Debug.debug >= Debug.DEBUG_MSG)
    {
      Debug.debug("\nOffering Cipher Suites");
      Iterator it = this.cipherSuites.iterator();
      while (it.hasNext())
      {
        Debug.debug(CipherSuites.getSuiteName((byte[])it.next()));
      }
      Debug.debug("");
    }

    this.bodyLength 
      = 2 + 32 + (1 + sessionId.length) + (2 + 2*cipherSuites.size()) 
        + (1 + compression.length);
  }

  /**
   * Reads the body of this ClientHello message from the XDRInputStream
   *
   * <p>The controlling ServerHandShaker upon receipt of this message
   *    sets its client random, sessionID, and ciphersuite
   */
  protected void readBody(XDRInputStream is) 
    throws IOException
  {
    version = is.readBytes(2);
    handShaker.setProtocolVersion(version);

    this.random = is.readBytes(32);
    ((ServerHandShaker)handShaker).setClientHelloRandom(this.random);

    this.sessionId = is.readVector(1);
    ((ServerHandShaker)handShaker).setSessionId(this.sessionId);
    
    int numSuites = is.readShort()/2;
    this.cipherSuites = new Vector();
    for (int i=0; i<numSuites; i++)
    {
      this.cipherSuites.addElement(is.readBytes(2));
    }
    ((ServerHandShaker)handShaker).setCipherSuite(this.cipherSuites);

    this.compression = is.readVector(1);
  }

  /**
   * Writes the body of this ClientHello message to the XDROutputStream
   */
  protected void writeBody(XDROutputStream os) throws IOException
  {
    os.write(version);
    os.write(random);
    os.writeVector(1, sessionId);

    os.writeShort(2*cipherSuites.size());
    for (int i=0; i<cipherSuites.size(); i++)
    {
      os.write((byte[])cipherSuites.elementAt(i));
    }

    os.writeVector(1, compression);
  }
}
