//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.ssl;

import java.io.IOException;

import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;

/**
 * A class for ServerHello messages sent by a server in response to the
 * ClientHello message initiating handshaking.
 *
 * @auithor Ming Yung
 */
final class ServerHello extends HandShake
{
  private byte[] version;
  private byte compression;
  private byte[] random;
  private byte[] sessionId;
  private byte[] cipherSuite;

  /**
   * Constructs a ServerHello message from an XDRInputStream with
   * a callback to the controlling ClientHandShaker
   */
  protected ServerHello(ClientHandShaker handShaker, byte[] data) 
    throws IOException, SSLHandshakeException
  {
    super(handShaker, data);

    if (Debug.debug >= Debug.DEBUG_MSG)
      Debug.debug("++ ServerHello");
  }

  /**
   * Constructs a ServerHello message with a callback to the controlling
   * ServerHandShaker
   *
   * <p>This message contains a server random, a session id, and
   *    a cipher suite, all obtained via the callback
   */
  protected ServerHello(ServerHandShaker handShaker) 
    throws IOException
  {
    this.handShaker = handShaker;
    this.msgType = SERVER_HELLO;

    this.version = handShaker.getProtocolVersion();
    this.compression = handShaker.getCompressionMethods()[0];
    this.random = handShaker.getServerHelloRandom();
    this.sessionId = handShaker.getSessionID();
    this.cipherSuite = handShaker.getPendingCipherSuite();

    this.bodyLength = 2 + 32 + (1 + sessionId.length) + 2 + 1;
  }

  /**
   * Reads the body of this ServerHello message from an XDRInputStream
   */ 
  protected void readBody(XDRInputStream is) 
    throws IOException, SSLHandshakeException
  {
    this.version = is.readBytes(2);
    this.handShaker.setProtocolVersion(version);

    this.random =  is.readBytes(32);
    this.handShaker.setServerHelloRandom(this.random);

    this.sessionId = is.readVector(1);
    this.handShaker.setSessionId(this.sessionId);
    this.cipherSuite = is.readBytes(2);

    if (Debug.debug >= Debug.DEBUG_MSG)
    {
      Debug.debug("\nChosen Cipher Suite");
      Debug.debug(CipherSuites.getSuiteName(this.cipherSuite));
      Debug.debug("");
    }

    this.compression = is.readByte();

    ((ClientHandShaker)handShaker).setPendingCipherSuite(this.cipherSuite);
  }

  /**
   * Writes the body of this ServerHello message to an XDROutputStream
   */
  protected void writeBody(XDROutputStream os) throws IOException
  {
    os.write(this.version);
    os.write(this.random);
    os.writeVector(1, this.sessionId);
    os.write(this.cipherSuite);
    os.writeByte(this.compression);
  }
}
