//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.ssl;

import java.io.IOException;
import java.io.ByteArrayOutputStream;

import java.security.interfaces.RSAPublicKey;

import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLException;

/**
 * An abstract class for ServerKeyExchange messages sent by a server to a
 * client during key exchange, when the server does not have an encryption
 * cert or when an export cipher suite is pending
 *
 * Subclassed by RSA and DH specific classes
 *
 * @author Ming Yung
 */
abstract class ServerKeyExchange extends HandShake
{
  protected byte[] signature;
  protected byte[] params;
  protected int keyLength;

  /**
   * Constructs a ServerKeyExchange message from an XDRInputStream with
   * a callback to the controlling ClientHandShaker
   */
  protected ServerKeyExchange(ClientHandShaker handShaker, byte[] data) 
    throws IOException
  {
    super(handShaker, data);
  }

  /**
   * Constructs a ServerKeyExchange message with a callback to the
   * controlling ServerHandShaker.
   *
   * <p>The body of the message contains the modulus and exponent
   *    of a temporary RSA key of key length 512, and a signature
   *    on them.
   */
  protected ServerKeyExchange(ServerHandShaker handShaker, int keyLength) 
    throws IOException
  {
    this.handShaker = handShaker;
    this.msgType = SERVER_KEY_EXCHANGE;

    this.keyLength = keyLength;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    XDROutputStream out = new XDROutputStream(baos);

    generateParamsAndSig(out);

    out.flush();
    params = baos.toByteArray();

    signature = handShaker.generateServerSignature(params);
    
    this.bodyLength 
      = 2 + params.length + signature.length;
  }

  /**
   * Reads the body (the modulus, exponent & server signature) of 
   * this ServerKeyExchange message from an XDRInputStream.
   *
   * <p>The server signature is verified.
   */
  protected void readBody(XDRInputStream is) 
    throws IOException
  {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    XDROutputStream out = new XDROutputStream(baos);

    verifyParamsSig(is, out);

    out.flush();
    params = baos.toByteArray();

    signature = is.readVector(2);
    ((ClientHandShaker)handShaker).verifyServerSignature(params, signature);
  }

  protected abstract void generateParamsAndSig(XDROutputStream os)
    throws IOException;

  protected abstract void verifyParamsSig(XDRInputStream is,
                                          XDROutputStream os)
    throws IOException;

  /**
   * Writes the body of this ServerKeyExchange message to an XDROutputStream
   */
  protected void writeBody(XDROutputStream os) throws IOException
  {
    os.write(params);
    os.writeVector(2, signature);
  }
}
