//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509;

import java.io.IOException;
import java.security.spec.AlgorithmParameterSpec;
import java.security.AlgorithmParameters;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Null;
import com.dstc.security.asn1.Oid;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.provider.OID;

/**
 * <p>A class representing the x.509 ASN.1 structure AlgorithmIdentifier.
 *
 * <pre>
 * AlgorithmIdentifier  ::=  SEQUENCE  {
 *       algorithm               OBJECT IDENTIFIER,
 *       parameters              ANY DEFINED BY algorithm OPTIONAL  }
 *                                  -- contains a value of the type
 *                                  -- registered for use with the
 *                                  -- algorithm object identifier value
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */  
public class AlgorithmId extends Sequence
{
  //Algorithm Object Identifier
  private String oid;

  //Parameters
  private AlgorithmParameters params;

  /** 
   * Constructs an AlgorithmID with null parameters from its oid 
   */
  public AlgorithmId(String oid) 
  {
    this.oid = oid;
    this.addComponent(new Oid(oid));

    this.params = null;
    this.addComponent(new Null());
  }

  /** 
   * Constructs an AlgorithmID with no parameters from its oid 
   */
  public AlgorithmId(String oid, boolean dummy) 
  {
    this.oid = oid;
    this.addComponent(new Oid(oid));
  }

  /** 
   * Constructs an AlgorithmID from its oid name and
   * supplied parameters
   */
  public AlgorithmId(String oid, AlgorithmParameters params) 
    throws Asn1Exception
  {
    this.oid = oid;
    this.addComponent(new Oid(oid));

    this.params = params;

    try
    {
      if (params == null) 
      {
        this.addComponent(new Null());
      } 
      else
      {
        this.addComponent(Asn1.decode(params.getEncoded()));
      }
    }
    catch (IOException e)
    {
      throw new Asn1Exception(e.toString());
    }
  }

  /**
   * Returns the string represntation of the Object Identifier
   */
  public String getOid() 
  {
    return oid;
  }

  /**
   * Returns the parameters associated with this AlgorithmId.
   */
  public AlgorithmParameters getParams() 
  {
    return params;
  }

  /**
   * Constructs an AlgorithmId from a DER encoding
   */
  public AlgorithmId(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);

    oid = ((Oid)components.elementAt(0)).getOid();

    //Any more stuff?
    if (1 >= components.size()) 
      return;

    // Try Null first
    try
    {
      Null nu = (Null)components.elementAt(1);
      params = null;
      return;
    }
    catch (ClassCastException e)
    {
      //Ignore
    }

    try
    {
      String algoName = OID.getAlgName(oid);
      params = AlgorithmParameters.getInstance(algoName);
      params.init(((Asn1)components.elementAt(1)).encode());
    }
    catch (Exception e)
    {
      System.out.println(e.toString());
      e.printStackTrace();
    }
  }
}
