//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509;

import java.util.Vector;
import com.dstc.security.asn1.*;

/**
 * <p>A class representing the X.500 ASN.1 structure Attribute.
 *
 * <pre>
 *   Attribute       ::=     SEQUENCE {
 *          type    AttributeType,
 *          values  SET OF AttributeValue
 *                  -- at least one value is required -- }
 *
 *   AttributeType           ::=   OBJECT IDENTIFIER
 *
 *   AttributeValue            ::=   CHOICE {
 *          DirectoryString,
 *          IA5String
 *          }
 *
 *   DirectoryString ::= CHOICE {
 *       teletexString           TeletexString (SIZE (1..MAX)),
 *       printableString         PrintableString (SIZE (1..MAX)),
 *       universalString         UniversalString (SIZE (1..MAX))
 *                                            }
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class Attribute extends Sequence
{
  //Object identifier as a string
  protected String type;

  //AttributeValues
  protected AttributeValues attributeValues;

  /**
   * Constructs an Attribute from a DER encoding
   */
  public Attribute(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);
 
    this.type = ((Oid)components.elementAt(0)).getOid();
 
    this.attributeValues 
      = new AttributeValues(((Asn1)components.elementAt(1)).encode());
  }

  /** 
   * Constructs an Attribute from type and Vector of Asn1 values
   */
  public Attribute(String type, Vector values)
  {
    this.type = type;
    addComponent(new Oid(type));

    this.attributeValues = new AttributeValues(values);
    addComponent(this.attributeValues);
  }

  /**
   * Constructs an Attribute from type and a Asn1
   */
  public Attribute(String type, Asn1 value)
  {
    Vector values = new Vector();
    values.addElement(value);

    this.type = type;
    addComponent(new Oid(type));

    this.attributeValues = new AttributeValues(values);
    addComponent(this.attributeValues);
  }

  /** 
   * Returns the AttributeType
   */
  public String getType()
  {
    return this.type;
  }

  /**
   * Returns the AttributeValues as a Vector of Asn1s
   */
  public Vector getValues()
  {
    return this.attributeValues.values;
  }

  /**
   * <p>Inner class representing the ASN.1 structure AttributeValues
   */
  protected class AttributeValues extends SetOf
  {
    //Vector of Asn1
    private Vector values;

    /**
     * Constructs an AttributeValues from a DER encoding
     */
    protected AttributeValues(byte[] encoded) throws Asn1Exception
    {
      doDecode(encoded);

      this.values = new Vector();
      for (int i=0; i<components.size(); i++)
      {
        this.values.addElement((Asn1)components.elementAt(i));
      }
    }

    /**
     * Constructs an AttributeValues from a Vector of Asn1s
     */
    protected AttributeValues(Vector values)
    {
      this.values = values;

      for (int i=0; i<values.size(); i++)
      {
        addComponent((Asn1)values.elementAt(i));
      }
    }
  }
}
