//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509;

import java.util.Set;
import java.util.HashSet;
import java.util.Date;
import java.util.Vector;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.spec.X509EncodedKeySpec;
import com.dstc.security.asn1.*;
import com.dstc.security.provider.OID;

/**
 * <p>Implements the X.509 ASN.1 structure TBSCertificate.
 *
 * <pre>
 *  <p>TBSCertificate  ::=  SEQUENCE  {
 *       version         [0]  EXPLICIT Version DEFAULT v1,
 *       serialNumber         CertificateSerialNumber,
 *       signature            AlgorithmIdentifier,
 *       issuer               Name,
 *       validity             Validity,
 *       subject              Name,
 *       subjectPublicKeyInfo SubjectPublicKeyInfo,
 *       issuerUniqueID  [1]  UniqueIdentifier OPTIONAL,
 *                            -- If present, version must be v2 or v3
 *       subjectUniqueID [2]  UniqueIdentifier OPTIONAL,
 *                            -- If present, version must be v2 or v3
 *       extensions      [3]  EXPLICIT Extensions OPTIONAL
 *                            -- If present, version must be v3
 *       }
 *
 *  <p>Version  ::=  INTEGER  {  v1(0), v2(1), v3(2)  }
 *
 *  <p>CertificateSerialNumber  ::=  INTEGER
 *
 *  <p>UniqueIdentifier  ::=  BIT STRING
 *
 *  <p>Extensions  ::=  SEQUENCE SIZE (1..MAX) OF Extension
 *
 *  <p>Extension  ::=  SEQUENCE  {
 *       extnID      OBJECT IDENTIFIER,
 *       critical    BOOLEAN DEFAULT FALSE,
 *       extnValue   OCTET STRING  }
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
final class TBSCertificate extends Sequence
{
  private int version = 0;
  private BigInteger serialNumber;
  private X500Name issuer;
  private X500Name subject;
  private Validity validity;
  private AlgorithmId algo;
  private SubjectPublicKeyInfo spki;
  private byte[] issuerUniqueId;
  private byte[] subjectUniqueId;
  private Extensions extns = null;
  
  /**
   * Constructs a TBSCertificate
   */
  TBSCertificate(int ver, BigInteger serialNumber, 
                 AlgorithmId sigAlg, X500Name issuer, 
                 Validity validity, X500Name subject, 
                 SubjectPublicKeyInfo spki, Extensions extns)
  {
    if (ver != 0)
    {
      this.version = ver;
      Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
      exp.addComponent(new com.dstc.security.asn1.Integer(this.version));
      addComponent(exp);
    }

    this.serialNumber = serialNumber;
    addComponent(new com.dstc.security.asn1.Integer(this.serialNumber));

    this.algo = sigAlg;
    addComponent(this.algo);

    this.issuer = issuer;
    addComponent(this.issuer);

    this.validity = validity;
    addComponent(this.validity);

    this.subject = subject;
    addComponent(this.subject);

    this.spki = spki;
    addComponent(this.spki);

    if (extns != null)
    {
      this.extns = extns;
      Explicit exp =  new Explicit(Asn1.TAG_CLASS_CONTEXT, 3);
      exp.addComponent(extns);
      addComponent(exp);
    }
  }

  /**
   * Constructs a TBSCertificate from a DER encoding
   */
  TBSCertificate(byte encoded[]) throws Asn1Exception
  {
    int i = 0;
    doDecode(encoded);
  
    //See if the optional field is there
    try
    {
      Explicit exp = (Explicit)components.elementAt(i);
      if (exp.getTagNumber() != 0)
        throw new Asn1Exception("Bad Tag Num");
      this.version = 
        ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();
      i++;
    }
    catch (ClassCastException e)
    {
    }
  
    this.serialNumber = 
      ((com.dstc.security.asn1.Integer)components.elementAt(i++))
         .getBigInteger();
  
    this.algo = new AlgorithmId(((Asn1)components.elementAt(i++)).encode()); 
    this.issuer = new X500Name(((Asn1)components.elementAt(i++)).encode()); 
  
    this.validity 
      = new Validity(((Asn1)components.elementAt(i++)).encode());
  
    this.subject = new X500Name(((Asn1)components.elementAt(i++)).encode()); 
  
    this.spki = 
      new SubjectPublicKeyInfo(((Asn1)components.elementAt(i++)).encode()); 

    // See if there is any more
    if (i == components.size()) return;

    Explicit exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 1)
    {
      this.issuerUniqueId = ((BitString)exp.getComponent()).getBytes();
      i++;
      if (i == components.size()) return;
    }

    exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 2)
    {
      this.subjectUniqueId = ((BitString)exp.getComponent()).getBytes();
      i++;
      if (i == components.size()) return;
    }

    exp = (Explicit)components.elementAt(i);
    if (exp.getTagNumber() == 3)
    {
      this.extns = new Extensions(exp.getComponent().encode());
    }

    //throw new Asn1Exception("Bad Tag Num");
  }

  /**
   * Returns the version for this TBSCertificate
   */
  int getVersion()
  {
    return this.version;
  }

  /**
   * Returns the serial number for this TBSCertificate
   */
  BigInteger getSerialNumber()
  {
    return this.serialNumber;
  }

  /**
   * Returns the Validity for this TBSCertificate
   */
  Validity getValidity()
  {
    return this.validity;
  }

  /**
   * Returns the SubjectPublicKeyInfo for this TBSCertificate
   */
  SubjectPublicKeyInfo getSubjectPubKeyInfo()
  {
    return this.spki;
  }

  /**
   * Returns the Subject for this TBSCertificate
   */
  X500Name getSubject()
  {
    return this.subject;
  }

  /**
   * Returns the Isuuer for this TBSCertificate
   */
  X500Name getIssuer()
  {
    return this.issuer;
  }

  /**
   * Returns the Extensions for this TBSCertificate
   */
  Extensions getExtensions()
  {
    return this.extns;
  }
}
